/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David BEILLOIN (Oct 2007)
**=======================================================================*/


#include "horizon.h"

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoVertexProperty.h>
#include <Inventor/nodes/SoScale.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoIndexedTriangleStripSet.h>

// TODO: using a float indexedTexture, should offer better flexibility 
// (colormap/range).
// set this macro to true to see per vertex material binding
// else we use a global color.
#define PER_VERTEX_COLOR 0

#define SWAP(type,a,b) { type tmp; tmp=a; a=b; b=tmp; }

SoNode *createHorizon(const SbString &fileName)
{
  fprintf(stdout,"Reading Horizon file %s\n",fileName.getString());

  FILE *fn = fopen(fileName.getString(),"r");
  if ( fn == NULL)
  {
    fprintf(stderr,"Unable to read file\n");
    return NULL;
  }

  int ret;
  int pinline, pcrossline;
  float px, py, time, velocity, depth, amplitude;
  char name[256];
  char cur_name[256]="\0";
  int cur_pinline=-1;

  SoScale *scale = new SoScale;
  scale->scaleFactor.setValue(1.0f,1.0f,9800.0f);

  SoShapeHints *shapehints=new SoShapeHints;
  shapehints->vertexOrdering.setValue(SoShapeHints::COUNTERCLOCKWISE);

  SoSeparator * sep = new SoSeparator;
  sep->addChild(scale);
  sep->addChild(shapehints);

  // read the first line that handle the column names :
  // inline	xline	X	Y	Horizon	Time	Velocity	Depth	Amplitude
  fscanf(fn,"%s %s %s %s %s %s %s %s %s\n",name,name,name,name,name,name,name,name,name);

  // read lines by lines
  SoVertexProperty *HZ_vp = NULL;
  SoIndexedTriangleStripSet *HZ_fs = NULL;

  int cIdx,ls0,ls1,le0,le1;
  int *crossTab1 = new int[1000];
  int *crossTab2 = new int[1000];

  int sizeTab1 = 0;

  int index = 0;
  bool finish=false;
  while ( !finish )
  {
    ret = fscanf(fn,"%d %d %f %f %s %f %f %f %f\n",
      &pinline, &pcrossline,
      &px, &py,
      name,
      &time, &velocity, &depth, &amplitude
      );
    finish = (ret == EOF);

    // create a previous face if any
    if ( !finish && pinline != cur_pinline )
    {
      cur_pinline = pinline;
      if ( HZ_vp!=NULL )
      {
        int idx1 = ls0;
        int idx2 = le0;
        while ( idx1< ls1 || idx2<le1 )
        {
          if ( crossTab2[idx1-ls0] >= crossTab1[idx2-le0] )
          {
            if (idx2<le1)
            {
              // create a triangle from right points
              HZ_fs->coordIndex.set1Value(cIdx++,idx1);
              HZ_fs->coordIndex.set1Value(cIdx++,idx2);
              HZ_fs->coordIndex.set1Value(cIdx++,idx2+1);
              HZ_fs->coordIndex.set1Value(cIdx++,SO_END_STRIP_INDEX);
              idx2++;
            }
            else if ( idx1<ls1 )
            {
              // create a triangle from left points
              HZ_fs->coordIndex.set1Value(cIdx++,idx1);
              HZ_fs->coordIndex.set1Value(cIdx++,idx2);
              HZ_fs->coordIndex.set1Value(cIdx++,idx1+1);
              HZ_fs->coordIndex.set1Value(cIdx++,SO_END_STRIP_INDEX);
              idx1++;
            }
          }
          else if ( crossTab2[idx1-ls0] <= crossTab1[idx2-le0])
          {
            if (idx1<ls1)
            {
              // create a triangle from right points
              HZ_fs->coordIndex.set1Value(cIdx++,idx2);
              HZ_fs->coordIndex.set1Value(cIdx++,idx1);
              HZ_fs->coordIndex.set1Value(cIdx++,idx1+1);
              HZ_fs->coordIndex.set1Value(cIdx++,SO_END_STRIP_INDEX);
              idx1++;
            }
            else if (idx2<le1)
            {
              // create a triangle from left points
              HZ_fs->coordIndex.set1Value(cIdx++,idx2);
              HZ_fs->coordIndex.set1Value(cIdx++,idx1);
              HZ_fs->coordIndex.set1Value(cIdx++,idx2+1);
              HZ_fs->coordIndex.set1Value(cIdx++,SO_END_STRIP_INDEX);
              idx2++;
            }
          }
        }

        ls0 = le0;
        ls1 = le1;
        le0 = index;

        sizeTab1 = 0;
        SWAP(int*,crossTab2,crossTab1);
      }
    }

    if ( strcmp(name,cur_name)!=0 || HZ_vp==NULL)
    {
      // finish previous Horizon if any
      if ( HZ_vp != NULL)
      {
#if !PER_VERTEX_COLOR
        SoBaseColor *color = new SoBaseColor;
        color->rgb.setValue(0.8f,0.8f,0.8f);
        sep->addChild(color);
#else
        // Now that we know amplitude min/max we can setup
        // material binding correctly

        for (int cc=0; cc<HZ_vp->orderedRGBA.getNum();cc++)
        {
          uint32_t color = HZ_vp->orderedRGBA[cc];
          float amplitude = *((float*)(&color));
          float fgray = 256.0f*(amplitude - min_amplitude)/(max_amplitude-min_amplitude);
          unsigned char gray;
          if ( fgray>255) gray=255;
          else if ( fgray<0) gray=0;
          else gray = (unsigned char)fgray;

          ((unsigned char *)&color)[0] = 255;  // Alpha
          ((unsigned char *)&color)[1] = gray; // blue
          ((unsigned char *)&color)[2] = gray; // green
          ((unsigned char *)&color)[3] = gray; // red
          HZ_vp->orderedRGBA.set1Value(cc,color);
        }
#endif
        sep->addChild(HZ_fs);
        //return sep;
      }

      if ( !finish )
      {
        // create a new vertex property
        HZ_vp = new SoVertexProperty;
#if PER_VERTEX_COLOR
        HZ_vp->materialBinding.setValue(SoVertexProperty::PER_VERTEX_INDEXED);
#endif
        // create a new faceset with this vp
        HZ_fs = new SoIndexedTriangleStripSet;
        HZ_fs->vertexProperty = HZ_vp;

        index =0;
        ls0=ls1=le0=le1=cIdx=0;
        strcpy(cur_name,name);
        fprintf(stdout," - loading horizon data %s\n",cur_name);
      }
    }
    if (!finish)
    {
      // set the vertex
      HZ_vp->vertex.set1Value(index,SbVec3f(px,py,time));
      crossTab1[sizeTab1++] = pcrossline;
      if (sizeTab1>1000)
        fprintf(stderr,"sizeTab1 is to low\n");

      // set the color binding through materialBinding property
#if PER_VERTEX_COLOR
      color = *((uint32_t*)(&amplitude));
      if ( amplitude < min_amplitude) min_amplitude = amplitude;
      if ( amplitude > max_amplitude) max_amplitude = amplitude;
      HZ_vp->orderedRGBA.set1Value(index,color);
#endif

      le1 = index;
      index++;
    }
  }

  delete [] crossTab1;
  delete [] crossTab2;

  fclose(fn);
  return sep;
}
