/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David BEILLOIN (Oct 2007)
**=======================================================================*/


#include "slice.h"

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoQuadMesh.h>
#include <Inventor/nodes/SoVertexShape.h>
#include <Inventor/nodes/SoVertexProperty.h>
#include <Inventor/nodes/SoIndexedTexture2.h>
#include <Inventor/fields/SoSFArray2D.h>
#include <Inventor/nodes/SoColorMap.h>
#include <Inventor/nodes/SoScale.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoPolygonOffset.h>
#include <Inventor/nodes/SoFaceSet.h>

#include <LDM/readers/SoVolumeReader.h>
#include <VolumeViz/readers/SoVRSegyFileReader.h>

///////////////////////////////////////////////////////////
// init static variables. 
///////////////////////////////////////////////////////////

SoScale *globalScale = NULL;
SoSwitch *globalRep = NULL;
SoSwitch *globalWiggleRep = NULL;


// local forward declaration
SoNode *densityRep(const SbVec2d *traceCoord, const float *traceData, const SbVec3i32 &inputDimension);
SoNode *wiggleRep(const SbVec2d *traceCoord, const float *traceData, const SbVec3i32 &inputDimension);
SoNode *wiggleVARep(const SbVec2d *traceCoord, const float *traceData, const SbVec3i32 &inputDimension);


SoNode *createSEGY2DSliceSceneGraph(const SbString &filename)
{
  fprintf(stdout,"Reading SEGY 2D file %s\n",filename.getString());

  ////////////////////////////////////////////
  // Read input slice data using SegY reader
  ////////////////////////////////////////////
  SbBox3f inputSize;
  SoVolumeData::DataType inputType;
  SbVec3i32 inputDimension;
  SbVec3i32 inputTileSize;
  SoVRSegyFileReader * inputReader;
  inputReader = new SoVRSegyFileReader;
  inputReader->ref();
  inputReader->setFilename(filename.getString());

  // Get data properties
  SoVRSegyFileReader::ReadError err = inputReader->getDataChar( inputSize, inputType, inputDimension );
  if ( err != SoVRSegyFileReader::RD_NO_ERROR )
  {
    SoError::post("unable to read input properties\n");
    exit(-1);
  }
  if (inputType!=SoDataSet::FLOAT)
  {
    SoError::post("Only FLOAT input is supported\n");
    exit(-1);
  }

  // Get trace coordinate
  SbVec2d *traceCoord = new SbVec2d [inputDimension[2]*inputDimension[1]];
  for (int j=0;j<inputDimension[2];j++)
    for (int i=0;i<inputDimension[1];i++)
      inputReader->getTraceCoord(i, j, traceCoord[j*(inputDimension[2]-1)+i]);

  // Get Trace signal
  float *traceData = new float [inputDimension[0]*inputDimension[1]];
  SbBox2i32 sliceBox = SbBox2i32(0,0,inputDimension[0]-1,inputDimension[1]-1);
  inputReader->getSubSlice(sliceBox, 0, traceData);

  // release the reader.
  inputReader->unref();

  // get density representation
  SoNode *sep_density   = densityRep(traceCoord,traceData,inputDimension);

  // get Wiggle representation
  SoNode *sep_wiggle    = wiggleRep(traceCoord,traceData,inputDimension);

  // get Wiggle Variable area representation
  SoNode *sep_va_wiggle = wiggleVARep(traceCoord,traceData,inputDimension);

  delete [] traceCoord;

  /////////////////////////////////////////////////////
  // main separator that handle each representation
  /////////////////////////////////////////////////////
  SoScale *scale = new SoScale;
  scale->scaleFactor.setValue(1.0f,1.0f,30.0f);

  SoShapeHints *shapehints=new SoShapeHints;
  shapehints->vertexOrdering.setValue(SoShapeHints::COUNTERCLOCKWISE);

  SoSwitch *sw1 = new SoSwitch;
  SoSwitch *sw2 = new SoSwitch;
  sw1->addChild(sep_density);
  sw1->addChild(sw2);
  sw2->addChild(sep_wiggle);
  sw2->addChild(sep_va_wiggle);

  sw1->whichChild.setValue(0);
  sw2->whichChild.setValue(0);

  SoSeparator *root = new SoSeparator;
  root->addChild(shapehints);
  root->addChild(scale);
  root->addChild(sw1);

  // create global parameters object
  if ( globalScale == NULL )
  {
    globalScale = new SoScale;
    globalScale->scaleFactor.setValue(1.0f,1.0f,z_scale);
    globalScale->ref();

    globalRep = new SoSwitch;
    globalRep->whichChild.setValue(0);
    globalRep->ref();

    globalWiggleRep = new SoSwitch;
    globalWiggleRep->whichChild.setValue(0);
    globalWiggleRep->ref();
  }

  // create some connection to sync parameters of all slices
  sw1->whichChild.connectFrom(&globalRep->whichChild);
  sw2->whichChild.connectFrom(&globalWiggleRep->whichChild);
  scale->scaleFactor.connectFrom(&globalScale->scaleFactor);

  return root;
}


/**
 * create 3D Slice with texture signal representation.
 */
SoNode *densityRep(const SbVec2d *traceCoord, const float *traceData, const SbVec3i32 &inputDimension)
{

  // create quadMesh
  SoQuadMesh * mesh = new SoQuadMesh;
  mesh->verticesPerColumn.setValue(inputDimension[1]);
  mesh->verticesPerRow.setValue(2);


  // create vertexProperty
  SoVertexProperty *vp = new SoVertexProperty;
  vp->vertex.setNum(inputDimension[1]*2);
  vp->texCoord.setNum(inputDimension[1]*2);

  SbVec3f *coord = vp->vertex.startEditing();
  SbVec2f *texcoord = vp->texCoord.startEditing();
  for (int j=0;j<inputDimension[2];j++)
    for (int i=0;i<inputDimension[1];i++)
    {
      // get first point trace coordinates
      SbVec2d P = traceCoord[j*(inputDimension[2]-1)+i];
      coord[i*2 + 0]= SbVec3f((float)P[0],(float)P[1], 0.0f);
      coord[i*2 + 1]= SbVec3f((float)P[0],(float)P[1], (float)inputDimension[0]);

      float tc_x = (float)(i)/(float)(inputDimension[1]);
      float epsilon_x = 1.0f / (float)(inputDimension[1]);
      float epsilon_y = 1.0f / (float)(inputDimension[0]);
      texcoord[i*2 + 0] = SbVec2f ( 0.0f + epsilon_y, tc_x + epsilon_x );
      texcoord[i*2 + 1] = SbVec2f ( 1.0f + epsilon_y, tc_x + epsilon_x );
    }
  vp->vertex.finishEditing();
  vp->texCoord.finishEditing();
  mesh->vertexProperty = vp;


  // create indexedTexture
  SoIndexedTexture2 *tex = new SoIndexedTexture2;
  tex->imageIndex.setValue(SbVec2s(inputDimension[0],inputDimension[1]),SoSFArray2D::FLOAT,traceData,SoSFArray2D::NO_COPY_AND_DELETE);
  tex->computeDataRange();

  // create colormap
  SoColorMap* colormap = new SoColorMap;
  colormap->predefinedColorMap = SoColorMap::BLUE_WHITE_RED;

  // used to avoid depth artifact
  SoPolygonOffset* polyOffset = new SoPolygonOffset;
  polyOffset->units.setValue(100.0f);

  SoSeparator *sep = new SoSeparator;
  sep->addChild(polyOffset);
  sep->addChild(colormap);
  sep->addChild(tex);
  sep->addChild(mesh);

  return sep;
}


/**
 * Ceate a simple wiggle representation of the given slice.
 */
SoNode *wiggleRep(const SbVec2d *traceCoord, const float *traceData, const SbVec3i32 &inputDimension)
{
  // create the lineset
  SoLineSet *lineset = new SoLineSet;
  lineset->numVertices.setNum(inputDimension[1]/reduce);
  int32_t *nv = lineset->numVertices.startEditing();

  // create the vertex property
  SoVertexProperty *vp_line = new SoVertexProperty;
  vp_line->vertex.setNum(inputDimension[1]*inputDimension[0]);
  SbVec3f *coord_line = vp_line->vertex.startEditing();

  for (int j=0;j<inputDimension[2];j++)
    for (int i=0;i<inputDimension[1]/reduce;i++)
    {
      // set the number of vertice for this polyline.
      nv[i]=inputDimension[0];

      // get trace coordinates and compute 3D direction of plane
      SbVec2d P, P1, P2;
      P  = traceCoord[j*(inputDimension[2]-1)+(i*reduce)];
      if (i>0)
        P1 = traceCoord[j*(inputDimension[2]-1)+((i-1)*reduce)];
      if (i<inputDimension[1]-2)
        P2 = traceCoord[j*(inputDimension[2]-1)+((i+1)*reduce)];

      SbVec2d dirN, dirP;
      dirN = P1 - P;
      dirP = P2 - P;
      if ( i<=0 )
        dirN = -dirP;
      if ( i>=inputDimension[1]-1 )
        dirP = -dirN;
      dirN.normalize();
      dirP.normalize();

      //fprintf (stderr,"coord of trace (%d,%d) is %.3f %.3f\n",i,j,P[0],P[1]);
      for (int z=0; z<inputDimension[0];z++)
      {
        SbVec2d tempP = P;
        float signal = traceData[i*inputDimension[0]+z];
        signal *= signal_scale;

        // move X, Y in the right Z direction depending on the plan
        // between 2 consecutive trace.
        if (signal>0)
          tempP += dirP * ( signal );
        else
          tempP -= dirN * ( signal );

        coord_line[i*inputDimension[0]+z]= SbVec3f((float)tempP[0],(float)tempP[1],(float)z);
      }
    }
  lineset->numVertices.finishEditing();
  vp_line->vertex.finishEditing();
  lineset->vertexProperty = vp_line;

  // create the color node
  SoBaseColor *color = new SoBaseColor;
  color->rgb.setValue(0.6f,0.6f,0.6f);

  SoSeparator *sep= new SoSeparator;
  sep->addChild(color);
  sep->addChild(lineset);

  return sep;
}

/**
 * Ceate a simple variable area wiggle representation of the given slice.
 */
SoNode *wiggleVARep(const SbVec2d *traceCoord, const float *traceData, const SbVec3i32 &inputDimension)
{

  ///////////////////////////////////////////////////
  // Wiggle Variable area : an SoFaceSet
  ///////////////////////////////////////////////////
  SoFaceSet* triset = new SoFaceSet;
  triset->numVertices.setNum(inputDimension[1]*inputDimension[0]);
  int32_t *nv_tri = triset->numVertices.startEditing();

  SoVertexProperty *vp_tri = new SoVertexProperty;
  vp_tri->vertex.setNum(inputDimension[1]*inputDimension[0]);
  SbVec3f *coord_tri = vp_tri->vertex.startEditing();
  int curVpIndex=0;
  int curTriIndex=0;

  for (int j=0;j<inputDimension[2];j++)
    for (int i=0;i<inputDimension[1]/reduce;i++)
    {
      // get trace coordinates and compute 3D direction of plane
      SbVec2d P, P1, P2;
      P  = traceCoord[j*(inputDimension[2]-1)+(i*reduce)];
      if (i>0)
        P1 = traceCoord[j*(inputDimension[2]-1)+((i-1)*reduce)];
      if (i<inputDimension[1]-2)
        P2 = traceCoord[j*(inputDimension[2]-1)+((i+1)*reduce)];

      SbVec2d dirN, dirP;
      dirN = P1 - P;
      dirP = P2 - P;
      if ( i<=0 )
        dirN = -dirP;
      if ( i>=inputDimension[1]-1 )
        dirP = -dirN;
      dirN.normalize();
      dirP.normalize();

      //fprintf (stderr,"coord of trace (%d,%d) is %.3f %.3f\n",i,j,P[0],P[1]);

      int curTriSize=0;
      SbVec3f tempPbefore, tempP;
      for (int z=0; z<inputDimension[0];z++)
      {
        // Compute 3D coordinate of the point
        // We try to move X, Y in the right direction depending on the plan
        // between 2 consecutive trace.
        // Z is kept with 1 step that will have to be scale by user
        // FIXME : a basic scale factor could be compute based on min/max signal
        // and max distance between trace
        SbVec2d tp = P;
        float signal = traceData[i*inputDimension[0]+z];
        signal *= signal_scale;
        if (signal>0)
          tp += dirP * signal;
        else
          tp -= dirN * signal;
        tempP = SbVec3f((float)tp[0],(float)tp[1],(float)z);


        // we are switching from positive to negative signal
        // check if we have to finish a previsouly started Faceset
        if (signal<0 && curTriSize>0)
        {
          // we are moving from positive to negative signal then start a end the current FaceSet at signal 0 position
          if (curTriSize>1)
          {
            // First point of the faceSet is the point between tempP and tempPbefore where signal is 0
            // do not forget to check if this point is the same than tempP.
            SbLine line0(tempP,tempPbefore);
            SbLine traceLine(SbVec3f((float)P[0],(float)P[1],0.0f),SbVec3f((float)P[0],(float)P[1],100.0f));
            SbVec3f Pinter1,Pinter2;
            line0.getClosestPoints(traceLine,Pinter1,Pinter2);
            if ( Pinter1 != tempP )
            {
              coord_tri[curVpIndex]=Pinter1;
              curVpIndex++;
              curTriSize++;
            }
          }

          if (curTriSize<3)
          {
            // no enough point to do a triangle
            // go back in the vertex list
            // and do not update the nbtriangle
            curVpIndex-=curTriSize;
          }
          else
          {
            // previous triangle is finished
            nv_tri[curTriIndex] = curTriSize;
            curTriIndex++;
          }
          curTriSize =0;
        }
        else if (signal>0)
        {
          // we are moving from negative to positive signal then start a new FaceSet at signal 0 position
          if (curTriSize==0)
          {
            // First point of the faceSet is the point between tempP and tempPbefore where signal is 0
            // do not forget to check if this point is the same than tempP.
            SbLine line0(tempP,tempPbefore);
            SbLine traceLine(SbVec3f((float)P[0],(float)P[1],0.0f),SbVec3f((float)P[0],(float)P[1],100.0f));
            SbVec3f Pinter1,Pinter2;
            line0.getClosestPoints(traceLine,Pinter1,Pinter2);
            if ( Pinter1 != tempP )
            {
              coord_tri[curVpIndex]=Pinter1;
              curVpIndex++;
              curTriSize++;
            }
          }

          coord_tri[curVpIndex] = tempP;
          curVpIndex++;
          curTriSize++;
        } /* if signal positif*/

        tempPbefore=tempP;
      } /* for data in 1 trace*/

      // manage the last triangle of the trace.
      if (curTriSize<3)
      {
        // no enough point to do a triangle
        // go back in the vertex list
        // and do not update the nbtriangle
        curVpIndex-=curTriSize;
      }
      else
      {
        // previous triangle is finished
        nv_tri[curTriIndex] = curTriSize;
        curTriIndex++;
      }
    }
  triset->numVertices.setNum(curTriIndex);
  vp_tri->vertex.setNum(curVpIndex);
  triset->vertexProperty = vp_tri;

  // create color
  SoBaseColor *color=new SoBaseColor;
  color->rgb.setValue(0.6f,0.6f,0.6f);

  SoSeparator *sep = new SoSeparator;
  sep->addChild(color);
  sep->addChild(triset);

  return sep;
}


