/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (Apr 2003)
**=======================================================================*/

/* Demonstrates converting a SEGY data file to an LDM format file for VolumeViz
 * and some of the features of the VolumeViz SEGY reader, including:
 *  - Query file header
 *  - Set byte positions for trace header fields (e.g. inline number)
 *  - Query trace header(s)
 *  - Query survey coordinates
 *
 * Parameters: 
 * - source volume filename
 * - tile dimension (power of 2). Default is 64x64x64.
 * - destination filename. Default is the source filename with extension .ldm.
 * - amount of main memory to do not exceed. Default is 128 Mb.
 * - option to indicate to output the xml header only.
 *
 * Execute this program without any parameters to show the full help text.
 *
 *============================================================================*/

//*

#define SEGY_XLINE_BYTE_POS  21 // 21
#define SEGY_INLINE_BYTE_POS  9 //  5
#define SEGY_NUM_TR_TO_READ 6
int SEGY_TraceToRead[SEGY_NUM_TR_TO_READ] = {1,2,3, 68,69, 135};
#define SEGY_SX_BYTE_FORMAT SoVRSegyTraceHeaderBytePosition::SEGY_IEEE_FLOAT_FORMAT
#define SEGY_SY_BYTE_FORMAT SoVRSegyTraceHeaderBytePosition::SEGY_IEEE_FLOAT_FORMAT
//*/

#include <LDM/converters/SoConverterParameters.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/converters/SoVolumeConverter.h>
#include <VolumeViz/readers/SoVRSegyFileReader.h>
#include <Inventor/helpers/SbFileHelper.h>

SbString SegyFileName = SbFileHelper::expandString("$OIVHOME/examples/data/VolumeViz/Waha8.sgy");

////////////////////////////////////////////////////////////////////////
class SegYConverter : public SoVolumeConverter
{
public:
  // Constructor
  SegYConverter():m_theSegYReader(NULL)
  {
  }
  
  // Destructor
  virtual ~SegYConverter()
  {
    if (m_theSegYReader)
    {
      m_theSegYReader->unref();
    };
  }

  // setup the specific reader to use
  void setReader(SoVRSegyFileReader* aSegYReader)
  {
    if (m_theSegYReader)
      m_theSegYReader->unref();

    m_theSegYReader = aSegYReader;

    if (m_theSegYReader)
      m_theSegYReader->ref();
  };

protected:
  // redefined from SoVolumeConverter
  virtual SoVolumeReader* getReader( const SbString& /*filename*/, const SbString& /*fileExt*/ )
  {
    return m_theSegYReader;
  };

  // Progress callback
  SoVolumeConverter::Abort
  progress( int /*numTilesGenerated*/, int /*numTilesToGenerate*/ )
  {
    return CVT_CONTINUE;
  };

private:
  // hold the reader to use for this converter
  SoVRSegyFileReader* m_theSegYReader;

};


////////////////////////////////////////////////////////////////////////
static bool cbabort = false;

SoVRSegyFileReader::SegyScanningResp
scanningCB( void* /*data*/, float percent, SoVRSegyFileReader* /*reader*/ )
{
  static int ipercentOld = -1;
  int ipercent = (int)(percent * 10);
  if (ipercent != ipercentOld) {
    printf( "scan: %5.1f%%\r", percent );
    ipercentOld = ipercent;
  }

  if (cbabort && percent > 10)
    return SoVRSegyFileReader::SCAN_ABORT;

  return SoVRSegyFileReader::SCAN_CONTINUE;
}


////////////////////////////////////////////////////////////////////////
int
main(int /*argc*/, char ** /*argv*/)
{
  // Initialize VolumeViz (core Open Inventor is automatically initialized)
  SoVolumeRendering::init();

  // SegY reader settings
  SoVRSegyFileReader *theSegYReader = new SoVRSegyFileReader();

  // Step 0 - Filename
  theSegYReader->setFilename( SegyFileName );
  theSegYReader->setScanningProgressCallback( scanningCB, NULL );

  // Step 1 - Textual header
# ifdef SEGY_TEXT_HEADER_ASCII
  theSegYReader->setSegyTextHeaderAscii( true );
# endif
  SbString textHeader = theSegYReader->getSegyTextHeader();
  { //output
    printf( "-- Step 1 -----------------------------------------------------------------------\n" );
    int l = textHeader.getLength();
    for (int i = 0; i < l; i+=80) {
      SbString tmp = textHeader.getSubString( i, i+80 );
      printf( "%s\n", tmp.getString() );
    }
  }

  // Step 2 - Binary header
  SoVRSegyFileHeader binHeader;
  SoVRSegyFileReader::SegyByteOrder byteOrder, actualByteOrder;
  actualByteOrder = theSegYReader->getSegyByteOrder();
  for( int step2 = 1; step2 <= 3; step2++) {
    switch (step2) {
      case 1: // default
        break;
      case 2:
        theSegYReader->setSegyByteOrder( SoVRSegyFileReader::SEGY_BIG_ENDIAN );
        break;
      case 3:
        theSegYReader->setSegyByteOrder( SoVRSegyFileReader::SEGY_LITTLE_ENDIAN );
        break;
    }
    theSegYReader->getSegyFileHeader( binHeader, true );
    byteOrder = theSegYReader->getSegyByteOrder();
    { // output
      const char* formatStr[10] = {
        "Fixed point (1 bytes)",
        "Floating point (4 bytes)",
        "Fixed point (4 bytes)",
        "Fixed point (2 bytes)",
        "Fixed point w/ gain code (4 bytes)",
        "Floating point IEEE (4 bytes)",
        "Signed integer (1 byte)",
        "Not currently use",
        "Two's complement integer (1 byte)",
      };
      printf( "-- Step 2.%d ---------------------------------------------------------------------\n", step2 );
      printf( "Byte order: %s\n", (byteOrder == SoVRSegyFileReader::SEGY_LITTLE_ENDIAN) ?
        "Little Endian (PC)" : "Big Endian (Workstation)" );
      printf( "Sample interval(us):    %d\n", binHeader.hdt );
      printf( "Num. samples per trace: %d\n", binHeader.hns );
      const char *fstr = "";
      if (binHeader.format >= 0 && binHeader.format <= 8)
        fstr = formatStr[binHeader.format];
      printf( "Data sample format:     %d = %s\n", binHeader.format, fstr );
    }
  }
  theSegYReader->setSegyByteOrder( actualByteOrder );

  // Step 3 - Trace header
  SoVRSegyTraceHeaderBytePosition trHdrBytePos = theSegYReader->getSegyTraceHeaderBytePosition();
  int numTraces = theSegYReader->getNumTraces();
  SoVRSegyTraceIdHeader traceHdr[SEGY_NUM_TR_TO_READ];
  trHdrBytePos.setBytePosition( SoVRSegyTraceHeaderBytePosition::SEGY_CROSSLINE, SEGY_XLINE_BYTE_POS );
  trHdrBytePos.setBytePosition( SoVRSegyTraceHeaderBytePosition::SEGY_INLINE, SEGY_INLINE_BYTE_POS );
# ifdef SEGY_SX_BYTE_POS
  trHdrBytePos.setBytePosition( SoVRSegyTraceHeaderBytePosition::SEGY_SX, SEGY_SX_BYTE_POS );
  trHdrBytePos.setBytePosition( SoVRSegyTraceHeaderBytePosition::SEGY_SY, SEGY_SY_BYTE_POS );
# endif
# ifdef SEGY_SX_BYTE_FORMAT
  trHdrBytePos.setByteFormat( SoVRSegyTraceHeaderBytePosition::SEGY_SX, SEGY_SX_BYTE_FORMAT );
  trHdrBytePos.setByteFormat( SoVRSegyTraceHeaderBytePosition::SEGY_SY, SEGY_SY_BYTE_FORMAT );
# endif
  theSegYReader->setSegyTraceHeaderBytePosition( trHdrBytePos );
  for (int t = 0; t < SEGY_NUM_TR_TO_READ; t++)
    theSegYReader->getSegyTraceHeader( SEGY_TraceToRead[t]-1, traceHdr[t] );
  { // output
    printf( "-- Step 3 -----------------------------------------------------------------------\n" );
    printf( "Num traces: %d\n", numTraces );
    for (int t = 0; t < SEGY_NUM_TR_TO_READ; t++) {
      printf( "Trace #%d\n", SEGY_TraceToRead[t] );
      printf( "Crossline number [%3d] %d\n",
        trHdrBytePos.getBytePosition( SoVRSegyTraceHeaderBytePosition::SEGY_CROSSLINE ),
        trHdrBytePos.getFieldValue( SoVRSegyTraceHeaderBytePosition::SEGY_CROSSLINE, traceHdr[t] ) );
      printf( "Inline number    [%3d] %d\n",
        trHdrBytePos.getBytePosition( SoVRSegyTraceHeaderBytePosition::SEGY_INLINE ),
        trHdrBytePos.getFieldValue( SoVRSegyTraceHeaderBytePosition::SEGY_INLINE, traceHdr[t] ) );
      printf( "Coord. scaling   [%3d] %d\n",
        trHdrBytePos.getBytePosition   ( SoVRSegyTraceHeaderBytePosition::SEGY_SCALCO ),
        trHdrBytePos.getFieldValue( SoVRSegyTraceHeaderBytePosition::SEGY_SCALCO, traceHdr[t] ) );
      printf( "Source coord. X  [%3d] %d\n",
        trHdrBytePos.getBytePosition( SoVRSegyTraceHeaderBytePosition::SEGY_SX ),
        trHdrBytePos.getFieldValue( SoVRSegyTraceHeaderBytePosition::SEGY_SX, traceHdr[t] ) );
      printf( "Source coord. Y  [%3d] %d\n",
        trHdrBytePos.getBytePosition( SoVRSegyTraceHeaderBytePosition::SEGY_SY ),
        trHdrBytePos.getFieldValue( SoVRSegyTraceHeaderBytePosition::SEGY_SY, traceHdr[t] ) );
      printf( "Delay rec. time  [%3d] %d\n",
        trHdrBytePos.getBytePosition   ( SoVRSegyTraceHeaderBytePosition::SEGY_DELRT ),
        trHdrBytePos.getFieldValue( SoVRSegyTraceHeaderBytePosition::SEGY_DELRT, traceHdr[t] ) );
    }
  }

  // Step 4 - Crossline,Inline,Time ranges
  SbBool segyRegular = theSegYReader->isSegyRegular();
  int crosslineFrom = 0, crosslineTo = 0, crosslineStep = 0;
  theSegYReader->getCrosslineRange( crosslineFrom, crosslineTo, crosslineStep );
  int inlineFrom = 0, inlineTo = 0, inlineStep = 0;
  theSegYReader->getInlineRange( inlineFrom, inlineTo, inlineStep );
  int zFrom = 0, zTo = 0, zStep = 0;
  theSegYReader->getZRange( zFrom, zTo, zStep );
  { // output
    printf( "-- Step 4 -----------------------------------------------------------------------\n" );
    printf( "SegY regular: %s\n", segyRegular ? "Yes" : "No" );
    printf( "Crossline from %d to %d [%d]\n", crosslineFrom, crosslineTo, crosslineStep );
    printf( "Inline    from %d to %d [%d]\n", inlineFrom, inlineTo, inlineStep );
    printf( "Z (ms)    from %d to %d [%d]\n", zFrom, zTo, zStep );
  }

  // Step 5 - Coordinates
  SbVec2d P1, P2, P3, P4;
  SbBool rightHanded = theSegYReader->getP1P2P3Coordinates( P1, P2, P3, P4 );
  theSegYReader->setDirectCoorSys( rightHanded );
  { // output
    printf( "-- Step 5 -----------------------------------------------------------------------\n" );
    printf( "P1 (%10.2f, %10.2f)\n", P1[0],P1[1] );
    printf( "Xline/Inline/Time axes %s handed.\n", rightHanded ? "right" : "left" );
    /*
    P4 -= P1;
    P3 -= P1;
    P2 -= P1;
    P1 -= P1;
    */
    printf( "P4 (%10.2f, %10.2f)   P3 (%10.2f, %10.2f)\n", P4[0],P4[1], P3[0],P3[1] );
    printf( "P1 (%10.2f, %10.2f)   P2 (%10.2f, %10.2f)\n", P1[0],P1[1], P2[0],P2[1] );
  }

  // Step 6 - Trace data
  void *traceData;
  int traceLength[SEGY_NUM_TR_TO_READ];
  for (int t = 0; t < SEGY_NUM_TR_TO_READ; t++)
    traceLength[t] = theSegYReader->getSegyTraceData( SEGY_TraceToRead[t]-1, traceData );
  { // output
    printf( "-- Step 6 -----------------------------------------------------------------------\n" );
    for (int t = 0; t < SEGY_NUM_TR_TO_READ; t++)
      printf( "Trace #%d, length=%d\n", SEGY_TraceToRead[t], traceLength[t] );
  }

  // Step 7 - Output data conversion
# ifdef SEGY_RANGE_MIN
    theSegYReader->setOutputDataType( true, SoVolumeData::UNSIGNED_SHORT );
    theSegYReader->setOutputDataRange( true, SEGY_RANGE_MIN, SEGY_RANGE_MAX );
# endif

  // data characteristic
  SbBox3f size;
  SoVolumeData::DataType type;
  SbVec3i32 dim;
  theSegYReader->getDataChar( size, type, dim );
  { // output
    printf( "---------------------------------------------------------------------------------\n" );
    printf( " size = %f %f %f - %f %f %f\n", size.getMin()[0], size.getMin()[1], size.getMin()[2], size.getMax()[0], size.getMax()[1], size.getMax()[2] );
    printf( " type = %d\n", type );
    printf( " dim  = %d %d %d\n", dim[0], dim[1], dim[2] );
    printf( "---------------------------------------------------------------------------------\n" );
  }

  // convert
  cbabort = false;
  int ret =0;
  {
    SegYConverter mySegYConverter;
    mySegYConverter.setReader( theSegYReader );

    // prepare parameters 
    SoConverterParameters mySegYConverterParameters;
    mySegYConverterParameters.setTileDim(128);
    mySegYConverterParameters.setInputFileName(SegyFileName);

    ret = mySegYConverter.convert( &mySegYConverterParameters );
  }

  SoVolumeRendering::finish();
  return ret;
}


