/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Jerome Hummel (Nov 2009)
**=======================================================================*/

#include "alphacurveeditor.h"

#include <QPainter>
#include <QMouseEvent>
#include <QVBoxLayout>
#include <QImage>
#include <QPixmap>

#define PENWIDTH 3 //Corresponds to one entry color value width in the colormap
#define IMAGEHEIGHT 100
#define WIDGETBORDER 10
#define KEYSIZE 7
#define CHECKERSIZE 10

AlphaCurveEditor::AlphaCurveEditor(SoTransferFunction* tf, QWidget* parent):
QWidget(parent),
mTransferFunction(tf),
mMousePressed(false),
mPickedKey(NULL),
mLeftKey(NULL),
mRightKey(NULL)
{
  mNumComponents = 0;
  if(mTransferFunction->colorMapType.getValue() == SoTransferFunction::RGBA)
    mNumComponents = 4;
  if(mTransferFunction->colorMapType.getValue() == SoTransferFunction::LUM_ALPHA)
    mNumComponents = 2;
  if(mTransferFunction->colorMapType.getValue() == SoTransferFunction::ALPHA)
    mNumComponents = 1;

  assert(mNumComponents == 4); //for now assume its RGBA

  mNumColors = mTransferFunction->actualColorMap.getNum() / mNumComponents;
  resize(mNumColors * PENWIDTH + 2* WIDGETBORDER, IMAGEHEIGHT + 2 * WIDGETBORDER);

  createColormapImage();
  computeAlphaKeys();
}

QSize AlphaCurveEditor::sizeHint ()
{
  return QSize(mNumColors * PENWIDTH + 2* WIDGETBORDER, IMAGEHEIGHT + 2 * WIDGETBORDER);
}


void AlphaCurveEditor::createColormapImage()
{
  mColorList.clear();
  int num = mTransferFunction->actualColorMap.getNum();
  //for now assume its RGBA
  const float* colors = mTransferFunction->actualColorMap.getValues(0);
  for(int i = 0; i < num; i+=4)
  {
    float r = colors[i]*255;
    float g = colors[i+1]*255;
    float b = colors[i+2]*255;
    float a = colors[i+3]*255;
    QColor color((int)r,(int)g,(int)b,(int)a);
    mColorList.append(color);
  }
  //Create checkerboard:
  QPixmap checker(CHECKERSIZE*2, CHECKERSIZE*2);
  QPainter p(&checker);
  QBrush brushw(Qt::white);
  p.setBrush(brushw);
  p.setPen(Qt::NoPen);
  p.drawRect(QRect(0,0, CHECKERSIZE*2, CHECKERSIZE*2));
  QBrush brushb(Qt::gray);
  p.setBrush(brushb);
  p.drawRect(QRect(0,0, CHECKERSIZE, CHECKERSIZE));
  p.drawRect(QRect(CHECKERSIZE, CHECKERSIZE, CHECKERSIZE, CHECKERSIZE));
  p.end();

  //Create colormap image:
  QPixmap pixmap(mNumColors * PENWIDTH, IMAGEHEIGHT);
  QPainter painter(&pixmap);
  QBrush brush;
  brush.setTexture(checker);
  brush.setStyle(Qt::TexturePattern);
  painter.setBrush(brush);
  painter.drawRect(0,0,mNumColors * PENWIDTH, IMAGEHEIGHT);

  painter.setBrush(Qt::NoBrush);

  QColor color;
  int x = 0;
  Q_FOREACH(color, mColorList)
  {
    QPen pen(color);
    pen.setWidth(PENWIDTH);
    painter.setPen(pen);
    painter.drawLine(x, 0, x, IMAGEHEIGHT);
    x += PENWIDTH;
  }
  painter.end();

  mColormapImage = pixmap.toImage();
}

void AlphaCurveEditor::computeAlphaKeys()
{
  mKeys.clear();
  int num = mTransferFunction->actualColorMap.getNum();
  //for now assume its RGBA
  const float* colors = mTransferFunction->actualColorMap.getValues(0);

  AlphaKey* key = new AlphaKey();
  key->mAlpha = colors[4];
  key->mX = 0; 
  key->mY = key->mAlpha * IMAGEHEIGHT; 
  mKeys.append(key);

  int keyindex = 0;
  int x = 0;

  float currentalpha;
  for(int i = 4; i < num; i+=4)
  {
    x += PENWIDTH;
    float a = colors[i+3];

    AlphaKey* newkey = new AlphaKey();
    newkey->mAlpha = a;
    newkey->mX = x; 
    newkey->mY = a * IMAGEHEIGHT; 

    //detect change of slope
    float a1 = key->mAlpha;
    int x1 = key->mX/PENWIDTH;
    float a2 = newkey->mAlpha;
    int x2 = newkey->mX/PENWIDTH; 

    float alpha = (a2 - a1)/(x2 - x1);

    if(keyindex == 0){//init
      currentalpha = (a2 - a1)/(x2 - x1);
      mKeys.append(key);
      keyindex++;
    }
    else
    {
      if(fabs(alpha - currentalpha) > 1e-4 )
      {
        mKeys.append(key);
        keyindex++;
        currentalpha = alpha;
      }
      else delete key;
    }
    key = newkey;

  }  
  //enter the last key before ending
  x += PENWIDTH;
  AlphaKey* lastkey = new AlphaKey();
  lastkey->mAlpha = key->mAlpha;
  lastkey->mX = x; 
  lastkey->mY = key->mAlpha * IMAGEHEIGHT; 
  mKeys.append(lastkey);
}

AlphaKey* AlphaCurveEditor::getPickedKey(int x, int y)
{
  x -= WIDGETBORDER;
  y -= WIDGETBORDER;
  AlphaKey* key;
  mLeftKey = NULL;
  mRightKey = NULL;
  int tolerance = KEYSIZE;
  for(int i = 0; i < mKeys.size(); i++)
  {
    key = mKeys[i];
    if(x >= key->mX - tolerance && x <= key->mX + tolerance && y >= IMAGEHEIGHT - key->mY - tolerance && y <= IMAGEHEIGHT - key->mY + tolerance){
      if( i > 0)
        mLeftKey = mKeys[i - 1];
      if( i + 1 < mKeys.size())
        mRightKey = mKeys[i + 1];
      return key;
    }
  }
  return NULL;
}

void AlphaCurveEditor::updatePickedKey(int x, int y)
{
  if(mPickedKey)
  {
    x -= WIDGETBORDER;
    y -= WIDGETBORDER;
    checkKeyPosition(x,y);
    repaint();

    //update alpha values
    updateAlphaValues();

    //update image
    createColormapImage();
  }

}

void AlphaCurveEditor::checkKeyPosition(int x, int y)
{
  if(mPickedKey)
  {
    if(mRightKey && mLeftKey)
    {
      if(x <= mLeftKey->mX + WIDGETBORDER)
        x = mLeftKey->mX;
      if(x >= mRightKey->mX + WIDGETBORDER)
        x = mRightKey->mX;
      mPickedKey->mX = x;
    }
    if(y <= 0)
      y = 0;
    if(y >= IMAGEHEIGHT)
      y = IMAGEHEIGHT;
    mPickedKey->mY = IMAGEHEIGHT - y;
    mPickedKey->mAlpha = (float)mPickedKey->mY / (float)IMAGEHEIGHT;
  }
}

void AlphaCurveEditor::updateAlphaValues()
{
  //Actually edit transfer function
  if(mPickedKey){
    //linear interpolation of alpha values
    float* colors = mTransferFunction->actualColorMap.startEditing();
    if(mLeftKey)
    {
      float a1 = mLeftKey->mAlpha;
      int x1 = mLeftKey->mX/PENWIDTH;
      float a2 = mPickedKey->mAlpha;
      int x2 = mPickedKey->mX/PENWIDTH;
      if (x2 != x1)
      {
        float factor = (a2 - a1)/(x2 - x1);
        for(int x = x1; x < x2; ++x)
        {
          float newalpha = factor * (x - x1) + a1;
          colors[x*4 + 3] = newalpha;
        }
      }
    }

    if(mRightKey)
    {
      float a1 = mPickedKey->mAlpha;
      int x1 = mPickedKey->mX/PENWIDTH;
      float a2 = mRightKey->mAlpha;
      int x2 = mRightKey->mX/PENWIDTH;
      if (x2 != x1)
      {
        float factor = (a2 - a1)/(x2 - x1);
        for(int x = x1; x < x2; ++x)
        {
          float newalpha = factor * (x - x1) + a1;
          colors[x*4 + 3] = newalpha;
        }
      }

    }

    mTransferFunction->actualColorMap.finishEditing();

  }
}

void AlphaCurveEditor::addKey(int x, int y)
{
  x -= WIDGETBORDER;
  y -= WIDGETBORDER;
  y = mColormapImage.height() - y;
  if(x < 0 || y < 0 || x > mColormapImage.width() || y > mColormapImage.height())
    return;
  else
  {

    //find insertion place:
    int index = 0;
    AlphaKey* key;
    Q_FOREACH(key, mKeys)
    {
      if(key->mX < x)
        index++;
    }
    //create new key 
    AlphaKey* newKey = new AlphaKey;
    newKey->mX = x;
    newKey->mY = y;
    newKey->mAlpha = y/mColormapImage.height();

    //check if it is not too close of another one
    bool closed = false;
    Q_FOREACH(key, mKeys)
    {
      if((fabs((double)(key->mX - x)) <= (KEYSIZE/2))  && (fabs((double)(key->mY - y)) <= (KEYSIZE/2)))
      {
        closed = true;
        break;
      }
    }
    if(!closed)
    {
      //insert it
      mKeys.insert(index , newKey);
    }
    repaint();

  }
}

void AlphaCurveEditor::removeKey(int x, int y)
{
  x -= WIDGETBORDER;
  y -= WIDGETBORDER;
  y = mColormapImage.height() - y;
  if(x < 0 || y < 0 || x > mColormapImage.width() || y > mColormapImage.height())
    return;
  else
  {

    //find insertion place:
    QVector<AlphaKey*>::iterator it;
    for(it = mKeys.begin(); it != mKeys.end(); ++it)
    {
      if((*it) == mPickedKey)
      {
        break;
      }

    }
    if(!mLeftKey || !mRightKey)
      return;
    else
    {
      //update picked alpha so that alpha values are modified accordingly
      float a1 = mLeftKey->mAlpha;
      int x1 = mLeftKey->mX/PENWIDTH;
      float a2 = mRightKey->mAlpha;
      int x2 = mRightKey->mX/PENWIDTH;
      if (x2 != x1)
      {
        float factor = (a2 - a1)/(x2 - x1);
        mPickedKey->mAlpha = factor * (mPickedKey->mX/PENWIDTH - x1) + a1;
      }
    }

    //delete key 
    mKeys.erase(it);
    repaint();

  }
}


void AlphaCurveEditor::paintEvent(QPaintEvent* /*event*/)
{
  QPainter painter(this);

  //draw colormap
  painter.drawImage(WIDGETBORDER, WIDGETBORDER, mColormapImage);

  //draw alpha keys
  QBrush brush(Qt::black);
  QBrush gbrush(Qt::green);
  QPen pen(Qt::black);
  pen.setWidth(2);
  painter.setPen(pen);
  painter.setBrush(brush);
  AlphaKey* key;
  QVector<QLine> lines;
  int i = 0;
  QPoint lastPoint;
  Q_FOREACH(key, mKeys)
  {
    i++;
    painter.drawRect(key->mX + WIDGETBORDER - (KEYSIZE/2), IMAGEHEIGHT + WIDGETBORDER - key->mY - (KEYSIZE/2), KEYSIZE, KEYSIZE);
    if(i == 2)
    {
      lines.append(QLine(lastPoint,QPoint(key->mX + WIDGETBORDER, IMAGEHEIGHT + WIDGETBORDER - key->mY)));
      i = 1;
    }
    lastPoint = QPoint(key->mX + WIDGETBORDER, IMAGEHEIGHT + WIDGETBORDER - key->mY);
  }
  painter.drawLines(lines);

  //draw picked key:
  if(mPickedKey){
    painter.setBrush(gbrush);
    int keysize = KEYSIZE + 4;
    painter.drawRect(mPickedKey->mX + WIDGETBORDER - (keysize/2), IMAGEHEIGHT + WIDGETBORDER - mPickedKey->mY - (keysize/2), keysize, keysize);
  }
  painter.end();
}

void AlphaCurveEditor::mouseDoubleClickEvent ( QMouseEvent * event )
{
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
  addKey(event->position().x(), event->position().y());
#else
  addKey(event->x(), event->y());
#endif
  QWidget::mouseDoubleClickEvent(event);
}

void AlphaCurveEditor::mouseMoveEvent ( QMouseEvent * event )
{
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
  updatePickedKey(event->position().x(), event->position().y());
#else
  updatePickedKey(event->x(), event->y());
#endif
  QWidget::mouseMoveEvent(event);
}

void AlphaCurveEditor::mousePressEvent ( QMouseEvent * event )
{
  //if(event->button() == Qt::LeftButton){
  mMousePressed = true;
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
  mPickedKey = getPickedKey(event->position().x(), event->position().y());
#else
  mPickedKey = getPickedKey(event->x(), event->y());
#endif
  //}
  repaint();
  QWidget::mousePressEvent(event);
}

void AlphaCurveEditor::mouseReleaseEvent ( QMouseEvent * event )
{
  if(event->button() == Qt::LeftButton)
  {
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
    updatePickedKey(event->position().x(), event->position().y());
#else
    updatePickedKey(event->x(), event->y());
#endif
    mMousePressed = false;
  }
  else if (event->button() == Qt::RightButton)
  {
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
    removeKey(event->position().x(), event->position().y());
#else
    removeKey(event->x(), event->y());
#endif
    //update alpha values
    updateAlphaValues();

    //update image
    createColormapImage();
  }
  mPickedKey = NULL;
  repaint();
  QWidget::mouseReleaseEvent(event);
}


