#include <QApplication>
#include <QBoxLayout>
#include <QCheckBox>
#include <QComboBox>
#include <QLabel>
#include <QMainWindow>
#include <QWidget>
#include <Inventor/ViewerComponents/Qt/QtHelper.h>
#include <Inventor/ViewerComponents/Qt/RenderAreaOrbiter.h>
#include <Inventor/ViewerComponents/nodes/SoViewingCube.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoFile.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoTranslation.h>
#include <VolumeViz/draggers/SoOrthoSliceDragger.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

// Forward declarations
SoVolumeData* createVolumeData( const int volDim );
SoSeparator* createBBox( const SbBox3f& bbox );

static const unsigned char ARBITRARY_UNDEFINED_VALUE = 0;

int
main( int argc, char** argv )
{
  QtHelper::addPlatformPluginsPath();

  QApplication app( argc, argv );
  SoVolumeRendering::init();

  ////////////////////////////////////////////////////////
  /// Scene Graph
  SoRef<SoSeparator> root = new SoSeparator;

  SoSeparator* volRendSep = new SoSeparator;
  root->addChild( volRendSep );

  const int volDim = 100;
  SoVolumeData* volData = createVolumeData(volDim);
  volRendSep->addChild( volData );
  volData->undefinedValue = ARBITRARY_UNDEFINED_VALUE;

  SoTransferFunction* transFunction = new SoTransferFunction;
  volRendSep->addChild( transFunction );
  transFunction->predefColorMap = SoTransferFunction::STANDARD;
  transFunction->actualColorMap.set1Value(3, 0.0f);

  SoOrthoSlice* orthoSlice = new SoOrthoSlice;
  volRendSep->addChild( orthoSlice );
  orthoSlice->sliceNumber = volDim * 3 / 4;
  orthoSlice->alphaUse = SoSlice::ALPHA_AS_IS;
  orthoSlice->clipping = TRUE;
  orthoSlice->enableBorder = TRUE;
  orthoSlice->borderWidth = 3.0f;

  SoOrthoSliceDragger* orthoSliceDragger = new SoOrthoSliceDragger;
  volRendSep->addChild( orthoSliceDragger );
  SoPath* path = new SoPath;
  path->append( orthoSlice );
  orthoSliceDragger->orthoSlicePath = path;

  SoVolumeRenderingQuality* volumeQuality = new SoVolumeRenderingQuality;
  volumeQuality->ambientOcclusion = TRUE;
  volRendSep->addChild( volumeQuality );

  SoVolumeRender* volumeRender = new SoVolumeRender;
  volumeRender->interpolation.connectFrom(&orthoSlice->interpolation);
  volRendSep->addChild( volumeRender );

  // Add Bbox
  root->addChild( createBBox( volData->extent.getValue() ) );

  ////////////////////////////////////////////////////////
  /// GUI
  QMainWindow* window = new QMainWindow;
  window->setWindowTitle( "UndefinedVoxel" );
  window->resize( 1280, 720 );
  window->show();

  QWidget* centralWidget = new QWidget( window );
  window->setCentralWidget( centralWidget );
  centralWidget->setLayout( new QVBoxLayout );
  centralWidget->layout()->setContentsMargins( 0, 0, 0, 0 );
  centralWidget->layout()->setSpacing( 0 );

  QWidget* topWidget = new QWidget( centralWidget );
  centralWidget->layout()->addWidget( topWidget );
  topWidget->setLayout( new QHBoxLayout );
  topWidget->layout()->setContentsMargins( 10, 10, 10, 10 );
  topWidget->setSizePolicy( QSizePolicy::Preferred, QSizePolicy::Maximum );


  qobject_cast<QBoxLayout*>( topWidget->layout() )->addSpacing( 10 );
  QCheckBox* checkbox = new QCheckBox( "Undefined value", centralWidget );
  checkbox->setChecked( true );
  topWidget->layout()->addWidget( checkbox );

  QLabel* interpLabel = new QLabel("Interpolation");
  interpLabel->setAlignment(Qt::AlignRight);
  topWidget->layout()->addWidget(interpLabel);

  QComboBox* interpBox = new QComboBox;
  interpBox->addItem("NEAREST");
  interpBox->addItem("LINEAR");
  interpBox->addItem("CUBIC");
  interpBox->setCurrentIndex(1);
  interpBox->setMaximumWidth(200);
  topWidget->layout()->addWidget(interpBox);

  ////////////////////////////////////////////////////////
  /// GUI Connection
  QObject::connect( checkbox, &QCheckBox::clicked, [volData]( bool checked ) {
    checked ? volData->undefinedValue = ARBITRARY_UNDEFINED_VALUE : volData->undefinedValue = std::numeric_limits<double>::quiet_NaN();
  } );

  QObject::connect(interpBox, static_cast<void (QComboBox::*)(int)>(&QComboBox::currentIndexChanged), [orthoSlice](int index) {
    orthoSlice->interpolation = (index > 1) ? SoVolumeShape::CUBIC : index;
  });

  ////////////////////////////////////////////////////////
  /// Viewer
  RenderAreaOrbiter* renderArea = new RenderAreaOrbiter( window );
  const float grayVal = 0.99f;
  renderArea->setClearColor( SbColorRGBA( grayVal, grayVal, grayVal, 1.0f ) );
  centralWidget->layout()->addWidget( renderArea->getContainerWidget() );
  renderArea->setSceneGraph( root.ptr() );
  SoViewingCube* vCube = renderArea->getSceneInteractor()->getViewingCube();
  vCube->opacityMin = 0.0f;
  vCube->opacityMax = 0.8f;
  SoFile* axis = new SoFile;
  axis->name = "$OIVHOME/examples/data/Inventor/ViewingCube/Compass/compass-rgb.iv";
  vCube->compass = axis;

  renderArea->viewAll( SbViewportRegion() );
  renderArea->setAntialiasingMode( SoSceneManager::AntialiasingMode::AUTO );
  renderArea->setAntialiasingQuality( 1 );


  ////////////////////////////////////////////////////////
  /// App Exec
  app.exec();


  ////////////////////////////////////////////////////////
  /// Exit
  root = nullptr;
  delete window;
  SoVolumeRendering::finish();
  return 0;
}

////////////////////////////////////////////////////////////////////////
// Helper for Bbox creation
SoSeparator*
createBBox( const SbBox3f& bbox )
{
  SoSeparator* bboxSep = new SoSeparator;

  SoPickStyle* pickStyle = new SoPickStyle;
  bboxSep->addChild( pickStyle );
  pickStyle->style = SoPickStyle::Style::UNPICKABLE;

  SoDrawStyle* drawStyle = new SoDrawStyle;
  bboxSep->addChild( drawStyle );
  drawStyle->style = SoDrawStyle::Style::LINES;
  drawStyle->lineWidth = 2.0f;

  SoBaseColor* bboxMat = new SoBaseColor;
  const float greyVal = 0.1f;
  bboxMat->rgb = SbVec3f( greyVal, greyVal, greyVal );
  bboxSep->addChild( bboxMat );

  SoLightModel* lightModel = new SoLightModel;
  bboxSep->addChild( lightModel );
  lightModel->model = SoLightModel::Model::BASE_COLOR;

  SoTranslation* translation = new SoTranslation;
  bboxSep->addChild( translation );
  translation->translation = bbox.getCenter();

  SoCube* cube = new SoCube;
  const SbVec3f bboxSize = bbox.getSize();
  cube->width = bboxSize[0];
  cube->height = bboxSize[1];
  cube->depth = bboxSize[2];
  bboxSep->addChild( cube );

  return bboxSep;
}

////////////////////////////////////////////////////////////////////////
// Fake volume
SoVolumeData *createVolumeData( const int volDim )
{
  unsigned char* data = new unsigned char[(size_t)volDim * volDim * volDim];
  const int center = volDim / 2;
  const int radius = volDim / 1.8;

  for ( int i = 0; i < volDim; ++i )
    for ( int j = 0; j < volDim; ++j )
      for ( int k = 0; k < volDim; ++k )
        if ( ( ( i - center ) * ( i - center ) + ( j - center ) * ( j - center ) + ( k - center ) * ( k - center ) ) < radius * radius )
          data[i * volDim * volDim + j * volDim + k] = ARBITRARY_UNDEFINED_VALUE;
        else
          data[i * volDim * volDim + j * volDim + k] = 150;

  SoVolumeData *volData = new SoVolumeData;
  volData->data.setValue( SbVec3i32( volDim, volDim, volDim ), SbDataType::UNSIGNED_BYTE, data, SoSFArray::NO_COPY_AND_DELETE );
  return volData;
}
