#include <Inventor/Qt/SbQtHelper.h>
#include <Inventor/SoDB.h>
#include <Inventor/ViewerComponents/Qt/RenderAreaOrbiter.h>
#include <Inventor/nodes/SoCamera.h>
#include <Inventor/nodes/SoComplexity.h>
#include <QApplication>
#include <QLabel>
#include <QMainWindow>
#include <QSplitter>
#include <QVBoxLayout>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include "RandomReader.h"
#include "SpinBoxSliderWidget.h"

/* This example illustrates the benefit of providing uniform tiles in a volume reader for each tile containing a constant value.
 * 2 render areas display an equivalent dataset having the same size, same extent and same tile size. The datasets have only tiles with a constant value which is random.
 * Thus, the rendering of such dataset is a mosaic of small cubes with a single color on each cube (a cube represents a tile).
 * Both render areas use the same volume reader class RandomReader. However the instance of RandomReader used by the left render area allocates only standard buffers (SoCPUBuffer) in the readTile method,
 * while the instance on the right render area allocates
 * only uniform buffers (SoCPUBufferUniform).
 *
 * The rendering are not equal on each render areas for 2 reasons:
 *  - the tile's value is defined randomly
 *  - the use of uniform buffer might lead to many more tiles loaded on the right renderarea.
 *
 * 2 sliders are provided in the user interface of this example to define the maximum amount of memory
 * allocated by the volume rendering. The first slider defines the maximum memory used to load tiles on the GPU.
 * The second slider defines the maximum amount of texture memory on the CPU used to render the dataset.
 *
 * The goal of this example is to illustrate the benefit of uniform buffer versus standard buffer when using the same
 * maximum amount of memory. If the maximum amount of memory is small, the right render area display many more tiles
 * which mimics an higher rendering quality. Moreover, if the maximum amount of the memory is quite large, both render area display
 * the same image but after a different duration. The right render area displays the highest resolution of the volume
 * much earlier than the left render area.
 */

int
main( int argc, char** argv )
{
  SbQtHelper::addPlatformPluginsPath();

  QApplication app( argc, argv );

  SoDB::init();
  SoVolumeRendering::init();
  RandomReader::initClass();

  // GUI
  QMainWindow* window = new QMainWindow;

  QWidget* centralWidget = new QWidget( window );
  QPalette pal = centralWidget->palette();
  pal.setColor( QPalette::Window, Qt::white );
  centralWidget->setPalette( pal );
  centralWidget->setAutoFillBackground( true );
  centralWidget->setLayout( new QVBoxLayout );
  centralWidget->layout()->setContentsMargins( 0, 0, 0, 0 );
  centralWidget->layout()->setSpacing( 0 );

  QWidget* topWidget = new QWidget( centralWidget );
  centralWidget->layout()->addWidget( topWidget );
  topWidget->setLayout( new QHBoxLayout );
  topWidget->layout()->setContentsMargins( 0, 0, 0, 0 );
  topWidget->setSizePolicy( QSizePolicy::Preferred, QSizePolicy::Maximum );


  SpinBoxSliderWidget* spinBoxGPUMem = new SpinBoxSliderWidget( topWidget );
  topWidget->layout()->addWidget( spinBoxGPUMem );
  spinBoxGPUMem->setRange( 1, 4096 );
  spinBoxGPUMem->setLabel( "Max GPU memory used (MB)" );

  SpinBoxSliderWidget* spinBoxCPUMem = new SpinBoxSliderWidget( topWidget );
  topWidget->layout()->addWidget( spinBoxCPUMem );
  spinBoxCPUMem->setRange( 1, 4096 );
  spinBoxCPUMem->setLabel( "Max CPU memory used (MB)" );

  qobject_cast<QBoxLayout*>( topWidget->layout() )->addStretch();

  QWidget* viewersWidget = new QWidget( centralWidget );
  viewersWidget->setLayout( new QHBoxLayout );
  centralWidget->layout()->addWidget( viewersWidget );

  RenderAreaOrbiter* renderAreaLeft = new RenderAreaOrbiter( window );
  RenderAreaOrbiter* renderAreaRight = new RenderAreaOrbiter( window );

  const float grayVal = 1.0f;
  renderAreaLeft->setClearColor( SbColorRGBA( grayVal, grayVal, grayVal, grayVal ) );
  renderAreaRight->setClearColor( SbColorRGBA( grayVal, grayVal, grayVal, grayVal ) );

  // Left side
  QWidget* leftWidget = new QWidget;
  {
    leftWidget->setLayout( new QVBoxLayout );
    leftWidget->layout()->setContentsMargins( 20, 0, 0, 20 );
    leftWidget->layout()->addWidget( renderAreaLeft->getContainerWidget() );
    QLabel* leftLabel = new QLabel( "Without uniform tiles", leftWidget );
    QFont font = leftLabel->font();
    font.setPointSize( 14 );
    font.setStyleStrategy( QFont::PreferAntialias );
    leftLabel->setFont( font );
    leftLabel->setSizePolicy( QSizePolicy::Maximum, QSizePolicy::Maximum );
    leftWidget->layout()->addWidget( leftLabel );
  }

  // Right Side
  QWidget* rightWidget = new QWidget;
  {
    rightWidget->setLayout( new QVBoxLayout );
    rightWidget->layout()->setContentsMargins( 20, 0, 0, 20 );
    rightWidget->layout()->addWidget( renderAreaRight->getContainerWidget() );
    QLabel* rightLabel = new QLabel( "With uniform tiles", rightWidget );
    QFont font = rightLabel->font();
    font.setPointSize( 14 );
    font.setStyleStrategy( QFont::PreferAntialias );
    rightLabel->setFont( font );
    rightLabel->setSizePolicy( QSizePolicy::Maximum, QSizePolicy::Maximum );
    rightWidget->layout()->addWidget( rightLabel );
  }

  viewersWidget->layout()->addWidget( leftWidget );
  viewersWidget->layout()->addWidget( rightWidget );
  window->setCentralWidget( centralWidget );


  // Scene graph
  SoRef<SoSeparator> sep1 = new SoSeparator;
  SoRef<SoSeparator> sep2 = new SoSeparator;
  SoVolumeData* volumeDataLeft;
  SoVolumeData* volumeDataLRight;

  int volumeEdge = 1024;
  SbVec3i32 volumeSize( volumeEdge, volumeEdge, volumeEdge );

  {
    SoInteractiveComplexity* interactiveComplexity = new SoInteractiveComplexity;
    interactiveComplexity->fieldSettings.set1Value( 0, "SoComplexity value 0.3 0.9" );
    sep1->addChild( interactiveComplexity );
    sep1->addChild( new SoComplexity() );

    volumeDataLeft = new SoVolumeData();
    RandomReader* reader = new RandomReader( false, volumeSize );
    volumeDataLeft->setReader( *reader, true );
    sep1->addChild( volumeDataLeft );

    // Transfer function in a file
    SoTransferFunction* transfertFunction = new SoTransferFunction;
    transfertFunction->predefColorMap = SoTransferFunction::STANDARD;
    sep1->addChild( transfertFunction );

    SoVolumeRenderingQuality* volRenderingQuality = new SoVolumeRenderingQuality;
    sep1->addChild( volRenderingQuality );

    SoVolumeRender* volRender = new SoVolumeRender;
    volRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
    sep1->addChild( volRender );
  }

  {
    SoInteractiveComplexity* interactiveComplexity = new SoInteractiveComplexity;
    interactiveComplexity->fieldSettings.set1Value( 0, "SoComplexity value 0.3 0.9" );
    sep2->addChild( interactiveComplexity );
    sep2->addChild( new SoComplexity() );

    volumeDataLRight = new SoVolumeData();
    RandomReader* reader = new RandomReader( true, volumeSize );
    volumeDataLRight->setReader( *reader, true );
    sep2->addChild( volumeDataLRight );


    // Transfer function in a file
    SoTransferFunction* transfertFunction = new SoTransferFunction;
    transfertFunction->predefColorMap = SoTransferFunction::STANDARD;
    sep2->addChild( transfertFunction );

    SoVolumeRenderingQuality* volRenderingQuality = new SoVolumeRenderingQuality;
    sep2->addChild( volRenderingQuality );

    SoVolumeRender* volRender = new SoVolumeRender;
    volRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
    sep2->addChild( volRender );
  }

  renderAreaLeft->setSceneGraph( sep1.ptr() );
  renderAreaRight->setSceneGraph( sep2.ptr() );


  // Viewer
  renderAreaLeft->getSceneInteractor()->enableViewingCube( false );
  renderAreaRight->getSceneInteractor()->enableViewingCube( false );

  window->setCentralWidget( centralWidget );
  window->resize( 1440, 1024 );

  // GUI Connections
  QObject::connect( spinBoxGPUMem, &SpinBoxSliderWidget::valueChanged, [volumeDataLeft, volumeDataLRight]( int value ) {
    volumeDataLeft->ldmResourceParameters.getValue()->maxTexMemory = volumeDataLRight->ldmResourceParameters.getValue()->maxTexMemory = value;
  } );

  // Connections
  QObject::connect( spinBoxCPUMem, &SpinBoxSliderWidget::valueChanged, [volumeDataLeft, volumeDataLRight]( int value ) {
    volumeDataLeft->ldmResourceParameters.getValue()->maxMainMemory = volumeDataLRight->ldmResourceParameters.getValue()->maxMainMemory = value;
  } );

  spinBoxGPUMem->setValue( 5 );
  spinBoxCPUMem->setValue( 10 );

  // Cam init
  renderAreaLeft->getSceneInteractor()->getCamera()->position = SbVec3f( 874.039f, 874.039f, 874.039f );
  renderAreaLeft->getSceneInteractor()->getCamera()->orientation = SbRotation( SbVec3f( -0.590284f, 0.769274f, 0.244504f ), 0.987861f );

  // Cam connection
  renderAreaRight->getSceneInteractor()->getCamera()->position.connectFrom( &renderAreaLeft->getSceneInteractor()->getCamera()->position );
  renderAreaRight->getSceneInteractor()->getCamera()->orientation.connectFrom( &renderAreaLeft->getSceneInteractor()->getCamera()->orientation );
  renderAreaLeft->getSceneInteractor()->getCamera()->position.connectFrom( &renderAreaRight->getSceneInteractor()->getCamera()->position );
  renderAreaLeft->getSceneInteractor()->getCamera()->orientation.connectFrom( &renderAreaRight->getSceneInteractor()->getCamera()->orientation );

  renderAreaLeft->viewAll( SbViewportRegion() );
  renderAreaRight->viewAll( SbViewportRegion() );

  window->show();

  app.exec();

  // We free the memory.
  sep1 = nullptr;
  sep2 = nullptr;
  delete window;
  RandomReader::exitClass();
  SoVolumeRendering::finish();
  SoDB::finish();
  return 0;
}
