/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Apr 2011)
**=======================================================================*/
/*=======================================================================
** Modified by : Aymeric Chataigner (Oct 2012)
**=======================================================================*/

#include "VVIZ-scenegraph.h"
#include "utils.h"

// Common OIV nodes used 
#include <Inventor/nodes/SoGroup.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoScale.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoComplexity.h>

#define USE_PROJECTION 0
#define USE_COMMON_SHADER_SWITCH 0

//
// Create the volumeData to be use for the generic scenegraph
//
#include <VolumeViz/nodes/SoVolumeData.h>
SoVolumeData* createData(const SbString& filename)
{
  SoVolumeData* volumeData = new SoVolumeData;
  volumeData->fileName = filename;

#if USE_PROJECTION
  // Force an extent value that has a maining for the Spherical projection
  volumeData->extent = SbBox3f(-80.f, -180.f, 900.f, 80.f, 180.f, 1000);
  volumeData->extent = SbBox3f(-180.f, -90.f, 900.f, 80.f, 60.f, 1000);
#endif

  //volumeData->ldmResourceParameters.getValue()->loadPolicy = SoLDMResourceParameters::ALWAYS;
  return volumeData;
}

//
// add Global properties to the scene graph
//
#include <Inventor/nodes/SoGradientBackground.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoProjection.h>
#include <Inventor/nodes/SoTransformProjection.h>
#include <Inventor/nodes/SoGeoProjection.h>

void addGlobalProperties(SoGroup* sep)
{
  // More esthetic gradient background
  SoGradientBackground* background = new SoGradientBackground;
  sep->addChild(background);

  // Interactive complexity node for better performance when interacting with the scene
  SoInteractiveComplexity* iComplexity = new SoInteractiveComplexity;
  iComplexity->fieldSettings.set1Value(0,"SoVolumeRender numSlices 500 2000 500" );
  iComplexity->fieldSettings.set1Value(1,"SoShadowGroup isActive FALSE 2" );
  iComplexity->fieldSettings.set1Value(2,"SoVolumeRenderingQuality gradientQuality LOW MEDIUM" );
  iComplexity->fieldSettings.set1Value(3,"SoVolumeRender projectedTileSubdivision 1 4 2" );
  //iComplexity->interactiveMode = SoInteractiveComplexity::FORCE_INTERACTION;
  sep->addChild(iComplexity);

  SoComplexity* cplx= new SoComplexity;
  sep->addChild(cplx);

  // add a geographic localization information
  // that might be used instead of the extent information
#if USE_PROJECTION
  SoGeoProjection* dataProjection = new SoGeoProjection;
  dataProjection->projection = SbProj4Projection::NUM_PROJ4_PROJECTIONS;
#else
  SoTransformProjection* dataProjection = new SoTransformProjection;
#endif
  // no projection by default
  dataProjection->isOn = FALSE;


  sep->addChild(dataProjection);

  // add a global scale to test with
  SoScale* globalScale = new SoScale;
  sep->addChild(globalScale);

  // add a global Material
  SoMaterial* material = new SoMaterial;
  //material->transparency=0.9f;
  sep->addChild(material);

  // add a fake switch for no manip
  SoSwitch* fakeSwitch = new SoSwitch;
  fakeSwitch->setName("NONE_swopt");
  sep->addChild(fakeSwitch);

}


#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <LDM/manips/SoROIManip.h>

void addDataSet(SoGroup* sep, const std::list<SoDataSet*> &datasetList)
{
  if ( datasetList.size() == 0)
    return;

  SoGroup* dataGroup = new SoGroup;
  dataGroup->setName("VolumeData");

  // add a geometric localization transformation
  // that might be used instead of the extent information
  SoTransform* dataPosition = new SoTransform;
  dataGroup->addChild(dataPosition);

  std::list<SoDataSet*>::const_iterator it = datasetList.begin();
  int curId = 0;
  while (it!=datasetList.end() )
  {
    SoDataSet* curDataSet = *it;
    if ( curDataSet->getDimension().getValue()[2] > 1 )
    {
      //curDataSet->dataSetId = curId;

      // add the dataSet
      dataGroup->addChild(curDataSet);

      // add a datarange
      SoDataRange* dataRange  = new SoDataRange;
      //dataRange->dataRangeId = curId;
      //dataRange->min = 10;
      //dataRange->max = 240;
      dataGroup->addChild(dataRange);

      // add a transfer function
      SoTransferFunction* colormap = new SoTransferFunction;
      colormap->transferFunctionId.connectFrom(&curDataSet->dataSetId);

      // First dataset use a specific colormap
      if ( it == datasetList.begin() )
      {
        //if ( !colormap->loadColormap("$OIVHOME/examples/data/VolumeViz/Amplitude_BlueWhiteRed_VolRend.am"))
        {
          colormap->predefColorMap = SoTransferFunction::INTENSITY;
          //colormap->minValue = 80;
          //colormap->maxValue = 200;
        }
      }
      else
      {
        colormap->predefColorMap = SoTransferFunction::TEMPERATURE;
        //colormap->minValue = 80;
        //colormap->maxValue = 200;
      }
      dataGroup->addChild(colormap);
    }
    ++curId;
    ++it;
  }

  // add a ROI manipulator under an optional switch
  SoSwitch* roiSwitch = new SoSwitch;
  roiSwitch->setName("ROIManip_swopt");
  dataGroup->addChild(roiSwitch);

  SoROIManip* roiManip = new SoROIManip;
  // take the first dataset as reference for dimension
  SbVec3i32 dimensions = datasetList.front()->getDimension();
  roiManip->box.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  roiManip->subVolume.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  roiManip->constrained = TRUE;
  roiSwitch->addChild(roiManip);

  sep->addChild(dataGroup);
}

#include <VolumeViz/nodes/SoVolumeClippingGroup.h>
#include <Inventor/nodes/SoTextureUnit.h>
#include <VolumeViz/nodes/SoUniformGridProjectionClipping.h>
#include <Inventor/nodes/SoClipPlane.h>
#include <Inventor/draggers/SoTransformBoxDragger.h>

void addVolumeClipping(SoGroup* sep, SoDataSet* dataset)
{
  SoSwitch* group = new SoSwitch;
  group->setName("VolumeClippingSwitch");

  // add a fake node to get no clipping at all
  {
    SoGroup* noclippingGroup = new SoGroup;
    noclippingGroup->setName("NoClipping");
    group->addChild(noclippingGroup);
  }

  // add an SoClipPlane
  {
    SoClipPlane* clippingPlane0 = new SoClipPlane;
    clippingPlane0->setName("ClippingPlane");
    group->addChild(clippingPlane0);
  }
  // add an SoVolumeClippingGroup
  {
    SoGroup* volGroup = new SoGroup;
    volGroup->setName("VolumeClippingGroup");

    // add manip option
    SoSwitch* manipSwitch = new SoSwitch;
    manipSwitch->setName("VolumeClippingGroup_swopt");
    volGroup->addChild(manipSwitch);

    SoTransformBoxDragger* manip=new SoTransformBoxDragger;
    manipSwitch->addChild(manip);

    SoVolumeClippingGroup* volClipping = new SoVolumeClippingGroup;
    {
      // use a non axis-aligned cube
      SoTransform* cubeTransform = new SoTransform;
      SbBox3f volumeBox = dataset->extent.getValue();
      cubeTransform->translation = volumeBox.getCenter();
      cubeTransform->scaleFactor = volumeBox.getSize()/4.0;
      cubeTransform->rotation = SbRotation(0.0f, 1.0f, 1.0f, 0.9f);

      // setup manip correctly
      manip->translation = volumeBox.getCenter();
      manip->scaleFactor = volumeBox.getSize()/4.0;
      manip->rotation = SbRotation(0.0f, 1.0f, 1.0f, 0.9f);
      cubeTransform->translation.connectFrom(&manip->translation);
      cubeTransform->scaleFactor.connectFrom(&manip->scaleFactor);
      cubeTransform->rotation.connectFrom(&manip->rotation);

      volClipping->addChild(cubeTransform);

      SoCube* cube = new SoCube;
      volClipping->addChild(cube);
    }
    volGroup->addChild(volClipping);
    group->addChild(volGroup);
  }

  // add an SoUniformGridProjectionClipping
  {
     SoGroup* gridGroup = new SoGroup;
     gridGroup->setName("GridProjectionClipping");

     SoTextureUnit* gridUnit = new SoTextureUnit;
     gridUnit->unit = 5;
     gridGroup->addChild(gridUnit);

    // based on a generic OIV shape horizon
    SoUniformGridProjectionClipping* gridProjectionClipping0 = new SoUniformGridProjectionClipping;
    gridProjectionClipping0->sceneGraph = generateSyntheticHorizon(dataset,0,1.0, SoUniformGridClipping::Y);
    // TODO : based on a generic VVIZ shape horizon 
    //gridProjectionClipping0->sceneGraph = generateSyntheticVolumeGeometry(dataset,0,1.0, SoUniformGridClipping::Y);
    // TODO : based on a generic VVIZ HeighField node 
    //gridProjectionClipping0->sceneGraph = generateSyntheticVolumeGeometry(dataset,0,1.0, SoUniformGridClipping::Y);
    gridGroup->addChild(gridProjectionClipping0);

    group->addChild(gridGroup);
  }

  // add a Fake group for ALL option
  {
    SoGroup* volRoot = new SoGroup;
    volRoot->setName("ALL");
    group->addChild(volRoot);
  }

  // no clipping by default
  //group->whichChild = 0;
  sep->addChild(group);
}

#include <VolumeViz/nodes/SoVolumeShader.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

void createShader(SoSwitch* shaderSwitch, const std::list<SoDataSet*> &datasetList)
{
  // add a fake node to get no shader and let VViz do its default job
  {
    SoGroup* noShaderGroup = new SoGroup;
    noShaderGroup->setName("NoShader");
    shaderSwitch->addChild(noShaderGroup);
  }

  // add a default SoVolumeShader
  {
    SoGroup* shaderGroup = new SoGroup;
    shaderGroup->setName("defaultVolumeShader");
    SoVolumeShader* defaultVolumeShader = new SoVolumeShader;
    shaderGroup->addChild(defaultVolumeShader);
    shaderSwitch->addChild(shaderGroup);
  }

  // add a default SoVolumeRenderingQuality
  {
    SoGroup* shaderGroup = new SoGroup;
    shaderGroup->setName("defaultVolumeRenderingQuality");
    SoVolumeRenderingQuality* volQuality = new SoVolumeRenderingQuality;

    shaderGroup->addChild(volQuality);
    shaderSwitch->addChild(shaderGroup);
  }

  // add a SoVolumeRenderingQuality with custom DATA_COMBINE_FUNCTION fragment
  {
    SoGroup* shaderGroup = new SoGroup;
    shaderGroup->setName("custom_DATA_COMBINE_FUNCTION");
    SoVolumeRenderingQuality* volQuality = new SoVolumeRenderingQuality;

    // Create our custom shader
    SoFragmentShader* fragmentShader = new SoFragmentShader;
    fragmentShader->sourceProgram.setValue("$OIVHOME/examples/source/VolumeViz/VVIZ-template-SG/DATA_COMBINE_frag.glsl");
    fragmentShader->addShaderParameter1i("data1",1);
    if ( datasetList.size()> 1 )
      fragmentShader->addShaderParameter1i("data2",2);
    else 
      fragmentShader->addShaderParameter1i("data2",1);
    
    // set it in the VolumeViz shader pipeline
    volQuality->shaderObject.set1Value(SoVolumeShader::DATA_COMBINE_FUNCTION, fragmentShader);

    shaderGroup->addChild(volQuality);
    shaderSwitch->addChild(shaderGroup);
  }

  // add a SoVolumeRenderingQuality with custom GET_DATA_FUNCTION fragment
  {
    SoGroup* shaderGroup = new SoGroup;
    shaderGroup->setName("custom_GET_DATA_FUNCTION");
    SoVolumeRenderingQuality* volQuality = new SoVolumeRenderingQuality;

    // Create our custom shader
    SoFragmentShader* fragmentShader = new SoFragmentShader;
    fragmentShader->sourceProgram.setValue("$OIVHOME/examples/source/VolumeViz/VVIZ-template-SG/GET_DATA_frag.glsl");

    volQuality->shaderObject.set1Value(SoVolumeShader::GET_DATA_FUNCTION, fragmentShader);

    shaderGroup->addChild(volQuality);
    shaderSwitch->addChild(shaderGroup);
  }

  // add a SoVolumeRenderingQuality with custom FRAGMENT_COMPUTE_COLOR fragment
  // which replicate exactly the behavior of default VolumeViz multiData shader.
  // but use a blending threshold parameter to compose each dataset color.
  // See #define macro in shader source to disable threshold and have exactly the same VViz behavior.
  {
    SoGroup* shaderGroup = new SoGroup;
    shaderGroup->setName("custom_FRAGMENT_COMPUTE_COLOR");
    SoVolumeRenderingQuality* volQuality = new SoVolumeRenderingQuality;

    // Create our custom shader
    SoFragmentShader* fragmentShader = new SoFragmentShader;
    fragmentShader->sourceProgram.setValue("$OIVHOME/examples/source/VolumeViz/VVIZ-template-SG/FRAGMENT_COMPUTE_COLOR_frag.glsl");
    fragmentShader->addShaderParameter1f("threshold",0.5f);
#if 1
    // we should not need this if VolumedataDrawstyle is managed for other object than VolumeRender
    fragmentShader->addShaderParameter1i("data1",1);
    if ( datasetList.size()> 1 )
      fragmentShader->addShaderParameter1i("data2",2);
    else 
      fragmentShader->addShaderParameter1i("data2",1);
#endif
    // set it in the VolumeViz shader pipeline
    volQuality->shaderObject.set1Value(SoVolumeShader::FRAGMENT_COMPUTE_COLOR, fragmentShader);

    shaderGroup->addChild(volQuality);
    shaderSwitch->addChild(shaderGroup);
  }

  // add a SoVolumeRenderingQuality with custom FRAGMENT_COMPUTE_COLOR fragment
  // which replicate exactly the behavior of default VolumeViz multiData shader.
  // but use a blending threshold parameter to compose each dataset color.
  {
    SoGroup* shaderGroup = new SoGroup;
    shaderGroup->setName("custom_FRAGMENT_COMPUTE_COLOR1");
    SoVolumeRenderingQuality* volQuality = new SoVolumeRenderingQuality;

    // Create our custom shader
    SoFragmentShader* fragmentShader = new SoFragmentShader;
    fragmentShader->sourceProgram.setValue("$OIVHOME/examples/source/VolumeViz/VVIZ-template-SG/FRAGMENT_COMPUTE_COLOR1_frag.glsl");
    fragmentShader->addShaderParameter1f("threshold",0.5f);
    fragmentShader->addShaderParameter1i("data1",1);
    if ( datasetList.size()> 1 )
      fragmentShader->addShaderParameter1i("data2",2);
    else 
      fragmentShader->addShaderParameter1i("data2",1);
    
    // set it in the VolumeViz shader pipeline
    volQuality->shaderObject.set1Value(SoVolumeShader::FRAGMENT_COMPUTE_COLOR, fragmentShader);

    shaderGroup->addChild(volQuality);
    shaderSwitch->addChild(shaderGroup);
  }
}

#if !USE_COMMON_SHADER_SWITCH

SoSwitch* g_shaderSwitch= NULL;
std::list<SoDataSet*> *g_dataSetList;

void createFakeShaderGroupForInterface(SoSwitch* shaderSwitch, const std::list<SoDataSet*> &datasetList)
{
  g_shaderSwitch = shaderSwitch;
  g_dataSetList = const_cast<std::list<SoDataSet*>*>(&datasetList);

  // add a fake node to get no shader and let VViz do its default job
  {
    SoGroup* noShaderGroup = new SoGroup;
    noShaderGroup->setName("NoShader");
    shaderSwitch->addChild(noShaderGroup);
  }

  // add a default SoVolumeShader
  {
    SoGroup* shaderGroup = new SoGroup;
    shaderGroup->setName("defaultVolumeShader");
    shaderSwitch->addChild(shaderGroup);
  }

  // add a default SoVolumeRenderingQuality
  {
    SoGroup* shaderGroup = new SoGroup;
    shaderGroup->setName("defaultVolumeRenderingQuality");
    shaderSwitch->addChild(shaderGroup);
  }

  // add a SoVolumeRenderingQuality with custom DATA_COMBINE_FUNCTION fragment
  {
    SoGroup* shaderGroup = new SoGroup;
    shaderGroup->setName("custom_DATA_COMBINE_FUNCTION");
    shaderSwitch->addChild(shaderGroup);
  }

  // add a SoVolumeRenderingQuality with custom GET_DATA_FUNCTION fragment
  {
    SoGroup* shaderGroup = new SoGroup;
    shaderGroup->setName("custom_GET_DATA_FUNCTION");
    shaderSwitch->addChild(shaderGroup);
  }

  // add a SoVolumeRenderingQuality with custom FRAGMENT_COMPUTE_COLOR fragment
  {
    SoGroup* shaderGroup = new SoGroup;
    shaderGroup->setName("custom_FRAGMENT_COMPUTE_COLOR");
    shaderSwitch->addChild(shaderGroup);
  }

  // add a SoVolumeRenderingQuality with custom FRAGMENT_COMPUTE_COLOR fragment
  {
    SoGroup* shaderGroup = new SoGroup;
    shaderGroup->setName("custom_FRAGMENT_COMPUTE_COLOR1");
    shaderSwitch->addChild(shaderGroup);
  }
}
#endif //!USE_COMMON_SHADER_SWITCH

void createLocalShaderSwitch(SoGroup* volRoot, const SbBool /*forVolumeOnly*/=FALSE)
{
#if !USE_COMMON_SHADER_SWITCH
  SoSwitch* shaderSwitch = new SoSwitch;
  createShader(shaderSwitch, *g_dataSetList);

  if ( g_shaderSwitch )
    shaderSwitch->whichChild.connectFrom(&g_shaderSwitch->whichChild);
  volRoot->addChild(shaderSwitch);
#endif //!USE_COMMON_SHADER_SWITCH
}

void addVolumeShader(SoGroup* group, const std::list<SoDataSet*> &datasetList)
{
  SoSwitch* shaderSwitch = new SoSwitch;
  shaderSwitch->setName("VolumeShaderSwitch");
  group->addChild(shaderSwitch);
#if USE_COMMON_SHADER_SWITCH
  createShader(shaderSwitch, datasetList);
#else
  createFakeShaderGroupForInterface(shaderSwitch, datasetList);
#endif
}

#include <VolumeViz/nodes/SoVolumeDataDrawStyle.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
void addVolumeRender(SoGroup* sep)
{
  SoSeparator* volRoot = new SoSeparator;
  volRoot->setName("VolumeRender");

  createLocalShaderSwitch(volRoot, TRUE);

  // choose the VolumeRender style to use
  SoVolumeDataDrawStyle* volStyle = new SoVolumeDataDrawStyle;
  volStyle->style = SoVolumeDataDrawStyle::VOLUME_RENDER;
  volRoot->addChild(volStyle);

  // add the volumeRender node 
  SoVolumeRender* volRender = new SoVolumeRender;
  volRender->numSlicesControl = SoVolumeRender::MANUAL;
  volRender->numSlices = 500;
  volRender->samplingAlignment = SoVolumeRender::VIEW_ALIGNED;
  volRender->subdivideTile = TRUE;

  volRoot->addChild(volRender);

  sep->addChild(volRoot);
}

void addVolumeGeometry( VolumeGeometryType vgType, SoGroup* sep, SoDataSet* volumeData )
{
  SoSeparator* volRoot = new SoSeparator;
  SbName rootName( "NO_NAME" );

  switch( vgType )
  {
  case INDEXED_FACE_SET:
    rootName = "VolumeIndexedFaceSet";
    break;

  case BUFFERED_QUAD_SET:
    rootName = "VolumeBufferedShape_QUADS";
    break;

  case BUFFERED_LINE_SET:
    rootName = "VolumeBufferedShape_LINES";
    break;

  case BUFFERED_POINT_SET:
    rootName = "VolumeBufferedShape_POINTS";
    break;

  default:
    SoDebugError::post( "addVolumeGeometry", "vgType argument is NOT correct." );
    return;
  }

  volRoot->setName( rootName );

  createLocalShaderSwitch(volRoot);

  // add a synthetic volume geometry node 
  volRoot->addChild( generateSyntheticVolumeGeometry( vgType, volumeData, 0.0f, 0.5f, SoUniformGridClipping::Z ) );

  sep->addChild(volRoot);
}

void addVolumeGeometry_Skin( VolumeGeometryType vgType, SoGroup* sep, SoDataSet* volumeData )
{
  SoSeparator* volRoot = new SoSeparator;
  SbName rootName( "NO_NAME" );

  switch( vgType )
  {
  case INDEXED_FACE_SET:
    rootName = "Skin_VolumeIndexedFaceSet";
    break;

  case BUFFERED_QUAD_SET:
    rootName = "Skin_VolumeBufferedShape_QUADS";
    break;

  case BUFFERED_LINE_SET:
    rootName = "Skin_VolumeBufferedShape_LINES";
    break;

  case BUFFERED_POINT_SET:
    rootName = "Skin_VolumeBufferedShape_POINTS";
    break;

  default:
    SoDebugError::post( "addVolumeGeometry_Skin", "vgType argument is NOT correct." );
    return;
  }

  volRoot->setName( rootName );
  
  createLocalShaderSwitch(volRoot);

  // add a synthetic volume geometry node 
  volRoot->addChild( generateSyntheticVolumeGeometry_Skin( vgType, volumeData ) );

  sep->addChild(volRoot);
}

#include <VolumeViz/nodes/SoOrthoSlice.h>
void addVolumeGeometry_OrthoSlice( VolumeGeometryType vgType, SoGroup* sep, SoDataSet* volumeData, SoOrthoSlice::Axis axis )
{
  SoSeparator* volRoot = new SoSeparator;
  SbName rootName( "NO_NAME" );

  switch( vgType )
  {
  case INDEXED_FACE_SET:
    rootName = "OrthoSlice_VolumeIndexedFaceSet";
    break;

  case BUFFERED_QUAD_SET:
    rootName = "OrthoSlice_VolumeBufferedShape_QUADS";
    break;

  case BUFFERED_LINE_SET:
    rootName = "OrthoSlice_VolumeBufferedShape_LINES";
    break;

  case BUFFERED_POINT_SET:
    rootName = "OrthoSlice_VolumeBufferedShape_POINTS";
    break;

  default:
    SoDebugError::post( "addVolumeGeometry_OrthoSlice", "vgType argument is NOT correct." );
    return;
  }

  volRoot->setName( rootName );
  
  createLocalShaderSwitch(volRoot);

  // add a synthetic volume geometry node 
  volRoot->addChild( generateSyntheticVolumeGeometry_OrthoSlice( vgType, volumeData, axis ) );

  sep->addChild(volRoot);
}

// 2D modes
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoFenceSlice.h>
#include <VolumeViz/nodes/SoVolumeSkin.h>
#include <VolumeViz/nodes/SoObliqueSlice.h>
void addVolumeRendering(SoGroup* sep,SoDataSet* dataset)
{
  SoSwitch* group = new SoSwitch;
  group->whichChild = 0;
  group->setName("VolumeRenderingSwitch");

#if 1
  // 3D Volume Render
  addVolumeRender(group);
#endif

#if 1
  // 3D Volume Geometries representing an horizon
  addVolumeGeometry( INDEXED_FACE_SET, group, dataset );
  addVolumeGeometry( BUFFERED_QUAD_SET, group, dataset );
  addVolumeGeometry( BUFFERED_LINE_SET, group, dataset );
  addVolumeGeometry( BUFFERED_POINT_SET, group, dataset );

  // 3D Volume Geometries representing a skin
  addVolumeGeometry_Skin( INDEXED_FACE_SET, group, dataset );
  addVolumeGeometry_Skin( BUFFERED_QUAD_SET, group, dataset );
  addVolumeGeometry_Skin( BUFFERED_LINE_SET, group, dataset );
  addVolumeGeometry_Skin( BUFFERED_POINT_SET, group, dataset );

  // 3D Volume Geometries representing a slice
  addVolumeGeometry_OrthoSlice( INDEXED_FACE_SET, group, dataset, SoOrthoSlice::Y );
  addVolumeGeometry_OrthoSlice( BUFFERED_QUAD_SET, group, dataset, SoOrthoSlice::Y );
  addVolumeGeometry_OrthoSlice( BUFFERED_LINE_SET, group, dataset, SoOrthoSlice::Y );
  addVolumeGeometry_OrthoSlice( BUFFERED_POINT_SET, group, dataset, SoOrthoSlice::Y );
#endif

#if 1
  // 2D VolumeSkin
  {
    SoSeparator* volRoot = new SoSeparator;
    volRoot->setName("VolumeSkin");

    createLocalShaderSwitch(volRoot);

    SoTransform* skinTransform = new SoTransform;
    volRoot->addChild(skinTransform);

    SoVolumeSkin* volSkin = new SoVolumeSkin;
    volSkin->interpolation=SoVolumeSkin::MULTISAMPLE_12;
    volSkin->alphaUse=SoVolumeSkin::ALPHA_BINARY;
    volSkin->enableBumpMapping=TRUE;
    volSkin->faceMode=SoVolumeSkin::FRONT_AND_BACK;
    volRoot->addChild(volSkin);

    group->addChild(volRoot);
  }
#endif
#if 1
  // 2D Orthoslice
  {
    SoSeparator* volRoot = new SoSeparator;
    volRoot->setName("OrthoSlice");
    SbVec3i32 volDim = dataset->getDimension();

    createLocalShaderSwitch(volRoot);

    // add an orthoslice centered for each axis
    for (int i=0;i<3;++i)
    {
      SoOrthoSlice* orthoSlice = new SoOrthoSlice;
      orthoSlice->interpolation=SoOrthoSlice::MULTISAMPLE_12;
      orthoSlice->axis=SoOrthoSlice::Axis(i);
      orthoSlice->sliceNumber=volDim[i]/2;
      orthoSlice->enableBumpMapping=TRUE;
      volRoot->addChild(orthoSlice);
    }
    group->addChild(volRoot);
  }
#endif
#if 1
  // 2D ObliqueSlice
  {
    SoSeparator* volRoot = new SoSeparator;
    volRoot->setName("ObliqueSlice");

    createLocalShaderSwitch(volRoot);

    SbVec3f _min,_max;
    dataset->extent.getValue().getBounds(_min,_max);

    // add a diagonal obliqueSlice centered for 3 diagonals
    for (int i=0;i<3;++i)
    {
      SoObliqueSlice* obliqueSlice = new SoObliqueSlice;
      //obliqueSlice->interpolation=SoObliqueSlice::MULTISAMPLE_12;
      SbVec3f lastPoint = _min;
      lastPoint[i] = _max[i];
      obliqueSlice->plane = SbPlane(_min,_max,lastPoint);
      obliqueSlice->enableBumpMapping=TRUE;
      volRoot->addChild(obliqueSlice);
    }
    group->addChild(volRoot);
  }
#endif
#if 1
  // 2D FenceSlice
  {
    SoSeparator* volRoot = new SoSeparator;
    volRoot->setName("FenceSlice");

    createLocalShaderSwitch(volRoot);

    SbVec3f _min,_max;
    dataset->extent.getValue().getBounds(_min,_max);

    // add a FenceSlice centered for each axis
    for (int i=0;i<3;++i)
    {
      // TODO: position correctly the fence slice
      SoFenceSlice* fenceSlice = new SoFenceSlice;
      fenceSlice->axis = SoFenceSlice::Axis(i);
      fenceSlice->interpolation=SoFenceSlice::MULTISAMPLE_12;
      fenceSlice->enableBumpMapping=TRUE;

      // for if axis is i then coord are in i+1 i+2 space
      const int numPoints=6;
      SbVec2f startPoint = SbVec2f(_min[(i+1)%3],_min[(i+2)%3]);
      SbVec2f endPoint   = SbVec2f(_max[(i+1)%3],_max[(i+2)%3]);
      SbVec2f delta2D    = (endPoint - startPoint)/(float)numPoints;
      SbVec2f newPoint   = startPoint;
      for (int pt=0; pt<=numPoints; ++pt)
      {
        SbVec2f noise(0,0);
        // keep extrmety on corner of dataset
        if ( pt!=0 && pt!=numPoints)
          noise[0] = pt*delta2D[0]*0.2f*(( (pt%2)==0)?(1):(-1));
        fenceSlice->points.set1Value(pt, newPoint + noise);
        newPoint +=delta2D;
      }


      volRoot->addChild(fenceSlice);
    }
    group->addChild(volRoot);
  }
#endif

  // add a Fake group for ALL option
  {
    SoGroup* volRoot = new SoGroup;
    volRoot->setName("ALL");
    group->addChild(volRoot);
  }

  sep->addChild(group);
}

#include <VolumeViz/nodes/SoHeightFieldRender.h>
void addHorizonRendering(SoGroup* sep)
{
  SoSwitch* group = new SoSwitch;
  group->whichChild = -1;
  group->setName("HorizonRenderingSwitch");

  SoSeparator* volRoot = new SoSeparator;
  volRoot->setName("HeightFieldRender");

  createLocalShaderSwitch(volRoot, FALSE);

  SoHeightFieldRender* hf = new SoHeightFieldRender;
  volRoot->addChild(hf);
  group->addChild(volRoot);

  sep->addChild(group);
}

// HeightField mode
#include <VolumeViz/nodes/SoHeightFieldProperty.h>
#include <VolumeViz/nodes/SoHeightFieldGeometry.h>
void addHorizonDataSet(SoGroup* sep,const std::list<SoDataSet*> &datasetList)
{
  if ( datasetList.size() == 0)
    return;

  SoGroup* dataGroup = new SoGroup;
  dataGroup->setName("VolumeData");

  // add a geometric localization transformation
  // that might be used instead of the extent information
  SoTransform* dataPosition = new SoTransform;
  dataGroup->addChild(dataPosition);

  int curId = 0;
#if 1
  std::list<SoDataSet*>::const_iterator it = datasetList.begin();
  while ( it!=datasetList.end() )
  {
    SoDataSet* dataSet = (*it);
    const SbString& fileName = dataSet->fileName.getValue();
    ++it;
    if ( dataSet->getDimension().getValue()[2] <= 1 )
    {
#else
  {
    {
      dataPosition->scaleFactor = SbVec3f(2.f, 2.f, 0.0001f);
      dataPosition->translation = SbVec3f(-1.f, -1.f, -2.f);
      const SbString& fileName = "H:/_LDM_DATA/CGG/h3_highres_reg_bin.iv.ldm";
#endif
      // add the SoHeightFieldGeometry
      SoHeightFieldGeometry* hg = new SoHeightFieldGeometry();
      hg->fileName = fileName;
      hg->dataSetId = ++curId;
      dataGroup->addChild(hg);

      // add a datarange
      SoDataRange* dataRange  = new SoDataRange;
      dataGroup->addChild(dataRange);

      // add a transfer function
      SoTransferFunction* colormap = new SoTransferFunction;
      colormap->predefColorMap = SoTransferFunction::STANDARD;
      dataGroup->addChild(colormap);
    }
    ++curId;
  }

  // add a ROI manipulator under an optional switch
  SoSwitch* roiSwitch = new SoSwitch;
  roiSwitch->setName("ROIManip_swopt_hf");
  dataGroup->addChild(roiSwitch);

  SoROIManip* roiManip = new SoROIManip;
  // take the first dataset as reference for dimension
  SbVec3i32 dimensions = datasetList.front()->getDimension();
  roiManip->box.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  roiManip->subVolume.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  roiManip->constrained = TRUE;
  roiSwitch->addChild(roiManip);

  sep->addChild(dataGroup);

  //TODO: generate SoHeighField and corresponding scene graph
  //generateSyntheticHorizonViz(dataSet);
}

#include <LDM/nodes/SoMultiDataSeparator.h>

SoSeparator* buildGenericSceneGraph(const std::list<SoDataSet*> &datasetList)
{
  // This is the root of all our scene graph
  SoSeparator* root = new SoSeparator;
  root->setName("RootSceneGraph");
  root->ref();

  // add Global properties to the scene graph
  addGlobalProperties(root);

  // add bbox wireframe
  root->addChild(makeVolBBox(datasetList.front()->extent.getValue()));

  // Setup the VolumeViz part of the scene graph
  SoMultiDataSeparator* dataSeparator0 = new SoMultiDataSeparator;
  dataSeparator0->setName("VolumeRootSceneGraph");
  root->addChild(dataSeparator0);
  {
    // add dataSet to the scene graph with its base properties
    addDataSet(dataSeparator0,datasetList);

    // add clipping nodes
    addVolumeClipping(dataSeparator0,datasetList.front());

    // add Volume Shaders nodes
    addVolumeShader(dataSeparator0, datasetList);

    // add volume rendering
    addVolumeRendering(dataSeparator0,datasetList.front());
  }

  // Setup the HorizonViz part of the scene graph
  SoMultiDataSeparator* dataSeparator1 = new SoMultiDataSeparator;
  dataSeparator1->setName("HorizonRootSceneGraph");
  root->addChild(dataSeparator1);
  {
    // add datasets
    addHorizonDataSet(dataSeparator1, datasetList);
    
    SoNode* refNode = NULL;

    // add ref to same clipping switch than for 3D Volume datasets
    refNode = searchName<SoSwitch>(root, "VolumeClippingSwitch");
    if ( refNode )
      dataSeparator1->addChild(refNode);

    // add ref to same shaders switch than for 3D Volume datasets
    refNode = searchName<SoSwitch>(root, "VolumeShaderSwitch");
    if ( refNode )
      dataSeparator1->addChild(refNode);

    // add horizon rendering
    addHorizonRendering(dataSeparator1);
  }

  root->unrefNoDelete();
  return root;
}


