/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Apr 2011)
**=======================================================================*/

#include "VVIZ-scenegraph.h"
#include "VVIZ-interface.h"
#include "utils.h"

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/Xt/SoXtRenderArea.h>
#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/sensors/SoOneShotSensor.h>
#include <Inventor/sensors/SoTimerSensor.h>
#include <Inventor/nodes/SoTransform.h>

#include <DialogViz/SoDialogVizAll.h>
#include <DialogViz/dialog/SoTopLevelDialog.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>


/** default dataset used as input if none given on command line */
#define DEFAULT_DATASET "$OIVHOME/examples/data/VolumeViz/colt-float.ldm"
//#define DEFAULT_DATASET "$OIVHOME/examples/data/VolumeViz/3DHEAD.ldm"

/** default path to the source of the demo to find interface file */
#define DEMOROOT "$OIVHOME/examples/source/VolumeViz/VVIZ-template-SG/"

/**
 * Comment/Uncomment this line to test VViz multi-data feature if only 1 dataset 
 * is given on the command line the same dataset will be used to do coblending.
 */
//#define ENABLE_MULTIDATA

/**
 * Change this macro to the number of additional view to create in order to test multi-viewer
 */
#define ENABLE_MULTIVIEWER 0


void continuousRedrawCB(void* userData, SoSensor*sensor)
{
  SoNode* sg = (SoNode*)userData;
  if ( sg ) sg->touch();
  if (sensor) sensor->schedule();
}

void volgeomAnimCB(void* userData)
{
  SoTransform* trans = (SoTransform*)userData;
  if ( trans )
  {
    SbVec3f axis;
    float radian;
    trans->rotation.getValue().getValue(axis,radian);
    radian+=0.1f;
    trans->rotation.setValue(axis,fmod(radian,float(2.0*M_PI)));
  }
}

int
main(int argc, char **argv)
{
  Widget mainWindow = SoXt::init(argv[0]);
  SoDialogViz::init();
  SoVolumeRendering::init();

  //SoLDMGlobalResourceParameters::setIgnoreFullyTransparentTiles(TRUE);

  std::list<SoDataSet*> dataList;

  // check if we have dataset filenames given on the command line
  if ( argc > 1 )
  {
    for (int i=1; i<argc;++i)
    {
      SoVolumeData* volumeData = createData(argv[i]);
      volumeData->dataSetId = i;
      dataList.push_back(volumeData);
    }
  }

  // if no dataset on command line then use default file
  if ( dataList.size() == 0 )
  {
    SoVolumeData* volumeData=createData(DEFAULT_DATASET);
    dataList.push_back(volumeData);
  }

#ifdef ENABLE_MULTIDATA
  // If only one dataset is given then setup the same dataset to
  // test multiData mode (in particular with custom shader)
  if ( dataList.size() == 1 )
  {
    SoVolumeData* volumeData=new SoVolumeData;
    volumeData->dataSetId = 2;
    volumeData->fileName.setValue(dataList.front()->fileName.getValue());
    dataList.push_back(volumeData);
  }
#endif

  /** Global root scene graph loaded from VVIZ-scenegraph.iv */
  SoSeparator *g_rootSceneGraph = buildGenericSceneGraph(dataList);
  g_rootSceneGraph->ref();

  // Interface setup
  SbString InterfaceFile = SbString(DEMOROOT) + "VVIZ-interface.iv";
  SoTopLevelDialog *myTopLevelDialog;
  Widget parent = buildInterface(mainWindow, InterfaceFile.toLatin1(), "Viewer", &myTopLevelDialog,g_rootSceneGraph);

  // Viewer setup
  SoXtExaminerViewer* mainViewer = new SoXtExaminerViewer(parent);
  mainViewer->setTitle("Volume Template Test Scene graph");
  mainViewer->setSceneGraph(g_rootSceneGraph);
  mainViewer->setBackgroundColor(SbColor(0.f, 0.f, 0.f));
  mainViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
  mainViewer->setFloatingColorBuffer(TRUE, SoXtRenderArea::FLOAT_32_COLOR_BUFFER);
  mainViewer->show();
  mainViewer->viewAll();

  // Create additional viewer sharing the same scene graph
  std::list<SoXtExaminerViewer*> viewList;
  for (int curView=0; curView<ENABLE_MULTIVIEWER;++curView)
  {
    SoXtExaminerViewer* view = new SoXtExaminerViewer(mainWindow, "", FALSE);

    std::stringstream title;
    title << "Additionnal view " << curView;
    view->setTitle( title.str().c_str() );
    view->setSceneGraph(mainViewer->getSceneGraph());
    view->setBackgroundColor(mainViewer->getBackgroundColor());
    view->setTransparencyType(mainViewer->getTransparencyType());

    SbBool isFloatingBuffer;
    SoXtGLWidget::FloatColorBufferSize floatBufferSize;
    view->getFloatingColorBuffer(isFloatingBuffer,floatBufferSize);
    view->setFloatingColorBuffer(isFloatingBuffer, floatBufferSize);

    view->show();
    view->viewAll();
    viewList.push_back(view);
  }

  // Main Inventor event loop
  SoXt::show(mainWindow);

  SoXt::mainLoop();

  // release global object
  g_rootSceneGraph->unref();


  std::list<SoXtExaminerViewer*>::const_iterator viewListIterator;
  for (viewListIterator=viewList.begin();viewListIterator!=viewList.end();++viewListIterator)
    delete *viewListIterator;
  delete mainViewer;

  // release modules
  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();

  return 0;
}


