/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Apr 2011)
**=======================================================================*/
/*=======================================================================
** Modified by : Aymeric Chataigner (Oct 2012)
**=======================================================================*/

#include "utils.h"

#include <Inventor/actions/SoSearchAction.h>

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoQuadMesh.h>
#include <Inventor/nodes/SoIndexedLineSet.h>

#include <Inventor/draggers/SoTransformBoxDragger.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoUniformGridClipping.h>

/** Resolution of the geometry that will be generated */
#define SYNTHETIC_SIZE 100


///////////////////////////////////////////////////////////////////////
// Create a wireframe box showing bounding box of a SoVolumeData node
///////////////////////////////////////////////////////////////////////
SoSeparator *makeVolBBox( const SbBox3f &volSize )
{
  // The box will be easier to see without lighting and with wide lines
  SoLightModel *pLModel = new SoLightModel;
  pLModel->model = SoLightModel::BASE_COLOR;

  SoDrawStyle *pStyle = new SoDrawStyle;
  pStyle->lineWidth = 1;

  // The box should be unpickable so manip can be used inside it
  SoPickStyle *pPickable = new SoPickStyle;
  pPickable->style = SoPickStyle::UNPICKABLE;

  // Create a cube with the geometric size of the volume
  float xmin, xmax, ymin, ymax, zmin, zmax;
  volSize.getBounds( xmin,ymin, zmin, xmax, ymax, zmax );
  SoVertexProperty *pProp = new SoVertexProperty;
  pProp->vertex.set1Value( 0, SbVec3f(xmin,ymin,zmin) );
  pProp->vertex.set1Value( 1, SbVec3f(xmax,ymin,zmin) );
  pProp->vertex.set1Value( 2, SbVec3f(xmax,ymax,zmin) );
  pProp->vertex.set1Value( 3, SbVec3f(xmin,ymax,zmin) );
  pProp->vertex.set1Value( 4, SbVec3f(xmin,ymin,zmax) );
  pProp->vertex.set1Value( 5, SbVec3f(xmax,ymin,zmax) );
  pProp->vertex.set1Value( 6, SbVec3f(xmax,ymax,zmax) );
  pProp->vertex.set1Value( 7, SbVec3f(xmin,ymax,zmax) );
  pProp->orderedRGBA.set1Value( 0, 0xFF0000FF );

  // Draw it with a line set
  int coordIndices[] = {0,1,2,3,0,-1,4,5,6,7,4,-1,
                        0,4,-1, 1,5,-1, 2,6,-1, 3,7};
  int numCoordIndices = sizeof(coordIndices)/sizeof(int);
  SoIndexedLineSet *pLines = new SoIndexedLineSet;
  pLines->vertexProperty = pProp;
  pLines->coordIndex.setValues( 0, numCoordIndices, coordIndices );

  // Assemble scene graph
  SoSeparator *pBoxSep = new SoSeparator;
  pBoxSep->setName("VolumeBBox");
  pBoxSep->addChild( pLModel );
  pBoxSep->addChild( pPickable );
  pBoxSep->addChild( pStyle );
  pBoxSep->addChild( pLines );
  return pBoxSep;
}

/**
 * Generate a synthetic horizon based on the extent of a volumData 
 */
SoNode* generateSyntheticHorizon(SoDataSet* volumeData, const float zcenter, const float zscale, const SoUniformGridClipping::Axis axis)
{
  SbBox3f volumeBox = volumeData->extent.getValue();
  SoSeparator* shapeSep = new SoSeparator;

  // By default elevation is generated on Y axis
  // then change the index indirection to generate on correct axis
  SbVec3i32 axisOrder;
  switch (axis)
  {
  case SoUniformGridClipping::X : axisOrder = SbVec3i32(2,0,1); break;
  case SoUniformGridClipping::Y : axisOrder = SbVec3i32(0,1,2); break;
  case SoUniformGridClipping::Z : axisOrder = SbVec3i32(1,2,0); break;
  }

  // We generate a synthetic shape in range [-1,-1,-1] [1,1,1] range
  SoTransform* transform = new SoTransform;
  transform->translation = volumeBox.getCenter();
  transform->scaleFactor = volumeBox.getSize()/2.0;

  shapeSep->addChild(transform);

  // setup two side lighting
  SoShapeHints *shapeHints = new SoShapeHints;
  shapeHints->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE;
  shapeSep->addChild(shapeHints);

  SoQuadMesh* IShape = new SoQuadMesh;
  SoVertexProperty* vp = new SoVertexProperty;
  IShape->vertexProperty.setValue(vp);

  // Define the resolution of the HeightMap
  float sizeX=SYNTHETIC_SIZE;
  float sizeY=SYNTHETIC_SIZE;

  // Define the range of the synthetic HeightMap
  SbVec3f vmin(-1,-1,-0.2f);
  SbVec3f vmax( 1, 1, 0.2f);

  SbVec3f vrange(vmax-vmin);
  float deltaX =vrange[0]/sizeX;
  float deltaY =vrange[1]/sizeY;

  // setup vertices
  vp->vertex.setNum(int(sizeX*sizeY));
  SbVec3f* vertices = vp->vertex.startEditing();
  for (int vy=0;vy<sizeY;++vy)
  {
    for (int vx=0;vx<sizeX;++vx)
    {
      float elevation = sin( 3*((float)vx/float(sizeX) + 4*(float)(vy)/float(sizeY) ) );
      elevation/= (2.0f/vrange[2]);
      (*vertices)[axisOrder[0]] = (float)vmin[0]+vx*deltaX ;
      (*vertices)[axisOrder[1]] = (float)elevation*zscale + zcenter ;
      (*vertices)[axisOrder[2]] = (float)vmin[1]+vy*deltaY ;
      ++vertices;
    }
  }
  vp->vertex.finishEditing();

  // setup indices
  IShape->verticesPerColumn = int(sizeX);
  IShape->verticesPerRow = int(sizeY);

  shapeSep->addChild(IShape);

  return shapeSep;
}

/**
* Generate a synthetic VolumeGeometry representing a sine shape based on the VolumeData extent.
*/
SoNode* generateSyntheticVolumeGeometry( VolumeGeometryType type, SoDataSet* volumeData, const float zcenter, const float zscale, SoUniformGridClipping::Axis axis )
{
  SbBox3f volumeBox = volumeData->extent.getValue();
  SoSeparator* shapeSep = new SoSeparator;

  // By default elevation is generated on Y axis
  // then change the index indirection to generate on correct axis
  SbVec3i32 axisOrder;
  switch (axis)
  {
  case SoUniformGridClipping::X : axisOrder = SbVec3i32(2,0,1); break;
  case SoUniformGridClipping::Y : axisOrder = SbVec3i32(0,1,2); break;
  case SoUniformGridClipping::Z : axisOrder = SbVec3i32(1,2,0); break;
  }

  // add manip option
  SoSwitch* manipSwitch = new SoSwitch;


  SoTransformBoxDragger* manip=new SoTransformBoxDragger;
  manipSwitch->addChild(manip);

  // We generate a synthetic shape in range [-1,-1,-1] [1,1,1] range
  SoTransform* transform = new SoTransform;
  transform->setName("VolumeGeometry_trans");
  transform->translation = volumeBox.getCenter();
  transform->scaleFactor = volumeBox.getSize()/2.0;
  transform->rotation = SbRotation(0.0f,1.0f,0.0f,2.4f);

  manip->translation = transform->translation.getValue();
  manip->scaleFactor = transform->scaleFactor.getValue();
  manip->rotation    = transform->rotation.getValue();

  transform->translation.connectFrom(&manip->translation);
  transform->scaleFactor.connectFrom(&manip->scaleFactor);
  transform->rotation.connectFrom(&manip->rotation);

  //Set the names of the manip switch and the transform node
  SbName manipSwitchName( "NO_NAME" );
  SbName transformName( "NO_NAME" );

  switch( type )
  {
  case INDEXED_FACE_SET:
    manipSwitchName = "VolumeIndexedFaceSet_swopt";
    transformName = "VolumeIndexedFaceSet_trans";
    break;

  case BUFFERED_QUAD_SET:
    manipSwitchName = "VolumeBufferedShape_QUADS_swopt";
    transformName = "VolumeBufferedShape_QUADS_trans";
    break;

  case BUFFERED_LINE_SET:
    manipSwitchName = "VolumeBufferedShape_LINES_swopt";
    transformName = "VolumeBufferedShape_LINES_trans";
    break;

  case BUFFERED_POINT_SET:
    manipSwitchName = "VolumeBufferedShape_POINTS_swopt";
    transformName = "VolumeBufferedShape_POINTS_trans";
    break;

  default:
    SoDebugError::post( "generateSyntheticVolumeGeometry", "type parameter is NOT correct" );
    return NULL;
  }

  manipSwitch->setName( manipSwitchName );
  transform->setName( transformName );

  //Add the manip switch and the transform node to the main separator
  shapeSep->addChild(manipSwitch);
  shapeSep->addChild(transform);


  // setup two side lighting
  SoShapeHints *shapeHints = new SoShapeHints;
  shapeHints->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE;
  shapeSep->addChild(shapeHints);

  // Define the resolution of the HeightMap
  const SbVec2f size( SYNTHETIC_SIZE, SYNTHETIC_SIZE );
  const float sizeX = size[0];
  const float sizeY = size[1];

  // Define the range of the synthetic HeightMap
  const SbVec3f vmin( -1, -1, -0.2f );
  const SbVec3f vmax( 1, 1, 0.2f );

  const SbVec3f vrange( vmax - vmin );
  const float deltaX = vrange[0] / sizeX;
  const float deltaY = vrange[1] / sizeY;

  //Set the number of vertices
 const unsigned int numVertices = (unsigned int)(sizeX * sizeY);
  
  // Instance the vertex buffer
  std::vector< SbVec3f >* vertexBuffer = new std::vector< SbVec3f >( numVertices, SbVec3f( 0.f, 0.f, 0.f ) );
  
  //Set the vertex buffer
  SbVec3f* vertices = &( ( *vertexBuffer )[0] );

  //Fill the vertex buffer
  for( int vy = 0; vy < sizeY; ++vy )
  {
    for( int vx = 0; vx < sizeX; ++vx )
    {
      float elevation = sin( 3 * ( (float) vx / float( sizeX ) + 4 * ( float ) vy / float( sizeY ) ) );
      elevation /= ( 2.0f / vrange[2] );
      ( *vertices )[ axisOrder[0] ] = ( float ) vmin[0] + vx * deltaX ;
      ( *vertices )[ axisOrder[1] ] = ( float ) elevation * zscale + zcenter ;
      ( *vertices )[ axisOrder[2] ] = ( float ) vmin[1] + vy * deltaY ;
      ++vertices;
    }
  }

  //Set a const vertex buffer pointer
  const float* vertexBufferPtr = ( const float* ) &( ( *vertexBuffer )[0] );

  // Create the right volume geometry node
  SoNode* vg = createVolumeGeometryInstance( type, vertexBufferPtr, size );
  
  // Check if the volume geometry node was correctly got.
  if( vg == NULL )
  {
    SoDebugError::post( "generateSyntheticVolumeGeometry",
      "vg == NULL" );
    return NULL;
  }

  //Add the created node to the separator
  shapeSep->addChild( ( SoNode* ) vg );

  return shapeSep;
}

/**
 * Generate a synthetic VolumeGeometry representing an horizon (skin) based on the extent of a VolumeData.
 */
SoNode* generateSyntheticVolumeGeometry_Skin( VolumeGeometryType type,  SoDataSet* volumeData )
{
  SbBox3f volumeBox = volumeData->extent.getValue();
  SoSeparator* shapeSep = new SoSeparator;

  // add manip option
  SoSwitch* manipSwitch = new SoSwitch;

  SoTransformBoxDragger* manip=new SoTransformBoxDragger;
  manipSwitch->addChild(manip);

  // We generate a synthetic shape in range [-1,-1,-1] [1,1,1] range
  SoTransform* transform = new SoTransform;

  manip->translation = transform->translation.getValue();
  manip->scaleFactor = transform->scaleFactor.getValue();
  manip->rotation    = transform->rotation.getValue();

  transform->translation.connectFrom(&manip->translation);
  transform->scaleFactor.connectFrom(&manip->scaleFactor);
  transform->rotation.connectFrom(&manip->rotation);

  //Set the names of the manip switch and the transform node
  SbName manipSwitchName( "NO_NAME" );
  SbName transformName( "NO_NAME" );

  switch( type )
  {
  case INDEXED_FACE_SET:
    manipSwitchName = "Skin_VolumeIndexedFaceSet_swopt";
    transformName = "Skin_VolumeIndexedFaceSet_trans";
    break;

  case BUFFERED_QUAD_SET:
    manipSwitchName = "Skin_VolumeBufferedShape_QUADS_swopt";
    transformName = "Skin_VolumeBufferedShape_QUADS_trans";
    break;

  case BUFFERED_LINE_SET:
    manipSwitchName = "Skin_VolumeBufferedShape_LINES_swopt";
    transformName = "Skin_VolumeBufferedShape_LINES_trans";
    break;

  case BUFFERED_POINT_SET:
    manipSwitchName = "Skin_VolumeBufferedShape_POINTS_swopt";
    transformName = "Skin_VolumeBufferedShape_POINTS_trans";
    break;

  default:
    SoDebugError::post( "generateSyntheticVolumeGeometry_Skin", "type parameter is NOT correct" );
    return NULL;
  }

  manipSwitch->setName( manipSwitchName );
  transform->setName( transformName );

  //Add the manip switch and the transform node to the main separator
  shapeSep->addChild(manipSwitch);
  shapeSep->addChild(transform);

  // setup two side lighting
  SoShapeHints *shapeHints = new SoShapeHints;
  shapeHints->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE;
  shapeSep->addChild(shapeHints);

  // Define the resolution of the HeightMap
  // Define the resolution of the HeightMap
  const SbVec3f size( SYNTHETIC_SIZE, SYNTHETIC_SIZE, SYNTHETIC_SIZE );
  const float sizeX = size[0];
  const float sizeY = size[1];
  const float sizeZ = size[2];

  // Define the range of the synthetic HeightMap
  const SbVec3f vmin = volumeBox.getMin();
  const SbVec3f vmax = volumeBox.getMax();
  const SbVec3f vrange(vmax-vmin);
  
  const float deltaX = vrange[0] / ( sizeX - 1 );
  const float deltaY = vrange[1] / ( sizeY - 1 );
  const float deltaZ = vrange[2] / ( sizeZ - 1 );

  //Skin bbox => 6 faces
  const unsigned int numFaces = 6;
  const unsigned int numVerticesPerFace = (unsigned int)( sizeX * sizeY );
  const unsigned int numVertices = numVerticesPerFace * numFaces;
  
  //Set the face offsets
  int faceOffset[6];
  for( int face = 0; face < 6; ++face )
    faceOffset[face] = face * numVerticesPerFace;

  // Instance the vertex buffer
  std::vector< SbVec3f >* vertexBuffer = new std::vector< SbVec3f >( numVertices, SbVec3f( 0.f, 0.f, 0.f ) );

  //Set the vertex buffer
  SbVec3f* vertices = &( ( *vertexBuffer )[0] );

  //Fill the vertex buffer
  for( int vy = 0; vy < sizeY; ++vy )
  {
    for( int vx = 0; vx < sizeX; ++vx )
    {
      int vertOffsetInFace = ( vx + vy * int( sizeY ) ) * 3;

      // face 0
      ( *vertices )[faceOffset[0] * 3 + vertOffsetInFace + 0] = ( float ) vmin[0] + vx * deltaX ;
      ( *vertices )[faceOffset[0] * 3 + vertOffsetInFace + 1] = ( float ) vmin[1] + vy * deltaY ;
      ( *vertices )[faceOffset[0] * 3 + vertOffsetInFace + 2] = ( float ) vmin[2];

      // face 1
      ( *vertices )[faceOffset[1] * 3 + vertOffsetInFace + 0] = ( float ) vmin[0] + vx * deltaX ;
      ( *vertices )[faceOffset[1] * 3 + vertOffsetInFace + 1] = ( float ) vmin[1] + vy * deltaY ;
      ( *vertices )[faceOffset[1] * 3 + vertOffsetInFace + 2] = ( float ) vmax[2];

      // face 2
      ( *vertices )[faceOffset[2] * 3 + vertOffsetInFace + 0] = ( float ) vmin[0];
      ( *vertices )[faceOffset[2] * 3 + vertOffsetInFace + 1] = ( float ) vmin[1] + vx * deltaY ;
      ( *vertices )[faceOffset[2] * 3 + vertOffsetInFace + 2] = ( float ) vmin[2] + vy * deltaZ ;

      // face 3
      ( *vertices )[faceOffset[3] * 3 + vertOffsetInFace + 0] = ( float ) vmax[0];
      ( *vertices )[faceOffset[3] * 3 + vertOffsetInFace + 1] = ( float ) vmin[1] + vx * deltaY ;
      ( *vertices )[faceOffset[3] * 3 + vertOffsetInFace + 2] = ( float ) vmin[2] + vy * deltaZ ;

      // face 4
      ( *vertices )[faceOffset[4] * 3 + vertOffsetInFace + 0] = ( float ) vmin[0] + vx * deltaX ;
      ( *vertices )[faceOffset[4] * 3 + vertOffsetInFace + 1] = ( float ) vmin[1];
      ( *vertices )[faceOffset[4] * 3 + vertOffsetInFace + 2] = ( float ) vmin[2] + vy * deltaZ ;

      // face 5
      ( *vertices )[faceOffset[5] * 3 + vertOffsetInFace + 0] = ( float ) vmin[0] + vx * deltaX ;
      ( *vertices )[faceOffset[5] * 3 + vertOffsetInFace + 1] = ( float ) vmax[1];
      ( *vertices )[faceOffset[5] * 3 + vertOffsetInFace + 2] = ( float ) vmin[2] + vy * deltaZ ;
    }
  }

  //Set a const vertex buffer pointer
  const float* vertexBufferPtr = ( const float* ) &( ( *vertexBuffer )[0] );

  // Create the right volume geometry node
  SoNode* vg = createVolumeGeometry_SkinInstance( type, vertexBufferPtr, size );

  // Check if the volume geometry node was correctly got.
  if( vg == NULL )
  {
    SoDebugError::post( "generateSyntheticVolumeGeometry_Skin",
      "vg == NULL" );
    return NULL;
  }

  shapeSep->addChild( ( SoNode* ) vg );

  return shapeSep;
}

/**
 * Generate a synthetic VolumeGeometry representing an OrthoSlice based on the extent of a volumData
 */
SoNode* generateSyntheticVolumeGeometry_OrthoSlice( VolumeGeometryType type, SoDataSet* volumeData, SoOrthoSlice::Axis axis ) 
{
  SbBox3f volumeBox = volumeData->extent.getValue();
  SoSeparator* shapeSep = new SoSeparator;

  // By default elevation is generated on Y axis
  // then change the index indirection to generate on correct axis
  SbVec3i32 axisOrder;
  switch (axis)
  {
  case SoOrthoSlice::X : axisOrder = SbVec3i32(2,0,1); break;
  case SoOrthoSlice::Y : axisOrder = SbVec3i32(0,1,2); break;
  case SoOrthoSlice::Z : axisOrder = SbVec3i32(1,2,0); break;
  }

  // add manip option
  SoSwitch* manipSwitch = new SoSwitch;

  SoTransformBoxDragger* manip=new SoTransformBoxDragger;
  manipSwitch->addChild(manip);

  // We generate a synthetic shape in range [-1,-1,-1] [1,1,1] range
  SoTransform* transform = new SoTransform;
  transform->setName("VolumeGeometry_OrthoSlice_trans");

  //transform->translation = volumeBox.getCenter();
  //transform->scaleFactor = volumeBox.getSize()/2.0;
  //transform->rotation = SbRotation(0.0f,1.0f,0.0f,2.4f);

  // to test boundary
  //transform->translation = volumeBox.getMin();
  //transform->scaleFactor = volumeBox.getSize();

  manip->translation = transform->translation.getValue();
  manip->scaleFactor = transform->scaleFactor.getValue();
  manip->rotation    = transform->rotation.getValue();

  transform->translation.connectFrom(&manip->translation);
  transform->scaleFactor.connectFrom(&manip->scaleFactor);
  transform->rotation.connectFrom(&manip->rotation);

    //Set the names of the manip switch and the transform node
  SbName manipSwitchName( "NO_NAME" );
  SbName transformName( "NO_NAME" );

  switch( type )
  {
  case INDEXED_FACE_SET:
    manipSwitchName = "OrthoSlice_VolumeIndexedFaceSet_swopt";
    transformName = "OrthoSlice_VolumeIndexedFaceSet_trans";
    break;

  case BUFFERED_QUAD_SET:
    manipSwitchName = "OrthoSlice_VolumeBufferedShape_QUADS_swopt";
    transformName = "OrthoSlice_VolumeBufferedShape_QUADS_trans";
    break;

  case BUFFERED_LINE_SET:
    manipSwitchName = "OrthoSlice_VolumeBufferedShape_LINES_swopt";
    transformName = "OrthoSlice_VolumeBufferedShape_LINES_trans";
    break;

  case BUFFERED_POINT_SET:
    manipSwitchName = "OrthoSlice_VolumeBufferedShape_POINTS_swopt";
    transformName = "OrthoSlice_VolumeBufferedShape_POINTS_trans";
    break;

  default:
    SoDebugError::post( "generateSyntheticVolumeGeometry_OrthoSlice", "type parameter is NOT correct" );
    return NULL;
  }

  manipSwitch->setName( manipSwitchName );
  transform->setName( transformName );

  //Add the manip switch and the transform node to the main separator
  shapeSep->addChild(manipSwitch);
  shapeSep->addChild(transform);

  // setup two side lighting
  SoShapeHints *shapeHints = new SoShapeHints;
  shapeHints->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE;
  shapeSep->addChild(shapeHints);

  // Define the resolution of the HeightMap
  const SbVec2f size( SYNTHETIC_SIZE, SYNTHETIC_SIZE );
  const float sizeX = size[0];
  const float sizeY = size[1];

  // Define the range of the synthetic HeightMap
  SbVec3f vmin = volumeBox.getMin();
  SbVec3f vmax = volumeBox.getMax();

  SbVec3f vrange(vmax-vmin);
  float deltaX = vrange[ axisOrder[0] ] / ( sizeX - 1 );
  float deltaY = vrange[ axisOrder[1] ] / ( sizeY - 1 );

  //Set the number of vertices
  const unsigned int numVertices = (unsigned int)(sizeX * sizeY);

  // Instance the vertex buffer
  std::vector< SbVec3f >* vertexBuffer = new std::vector< SbVec3f >( numVertices, SbVec3f( 0.f, 0.f, 0.f ) );
  
  //Set the vertex buffer pointer
  SbVec3f* vertices = &( ( *vertexBuffer )[0] );

  //Fill the vertex buffer
  for( int vy = 0; vy < sizeY; ++vy )
  {
    for (int vx = 0; vx < sizeX; ++vx )
    {
      ( *vertices )[ axisOrder[0] ] = ( float ) vmin[ axisOrder[0] ] + vx * deltaX;
      ( *vertices )[ axisOrder[1] ] = ( float ) vmin[ axisOrder[1] ] + vy * deltaY;
      ( *vertices )[ axisOrder[2] ] = ( float ) ( vmin[ axisOrder[2] ] + vmax[ axisOrder[2] ] ) / 2.0f;
      ++vertices;
    }
  }

  //Set a const vertex buffer pointer
  const float* vertexBufferPtr = ( const float* ) &( ( *vertexBuffer )[0] );

  // Create the right volume geometry node
  SoNode* vg = createVolumeGeometry_OrthoSliceInstance( type, vertexBufferPtr, size );

  // Check if the volume geometry node was correctly got.
  if( vg == NULL )
  {
    SoDebugError::post( "generateSyntheticVolumeGeometry_OrthoSlice",
      "vg == NULL" );
    return NULL;
  }

  shapeSep->addChild( ( SoNode* ) vg );

  return shapeSep;
}

/**
* Create a VolumeGeometry instance set with vertexBufferPtr. 
*/
SoNode* createVolumeGeometryInstance( 
                                      const VolumeGeometryType vgType, 
                                      const float* vertexBufferPtr,
                                      const SbVec2f& size
                                    )
{
  //The createVolumeGeometryInstance create a volume geometry instance in normal rendering mode
  const RenderingMode rMode = NORMAL;
  SoNode*volumeGeometry = NULL;
  const SbVec3f threeDimensionSize( size[0], size[1], 0.f );

  switch( vgType )
  {
  case BUFFERED_QUAD_SET:
   volumeGeometry = ( SoNode* ) createVolumeBufferedQuadSet( rMode, vertexBufferPtr, threeDimensionSize );
    break;
  case BUFFERED_LINE_SET:
   volumeGeometry = ( SoNode* ) createVolumeBufferedLineSet( rMode, vertexBufferPtr, threeDimensionSize );
    break;
  case BUFFERED_POINT_SET:
   volumeGeometry = ( SoNode* ) createVolumeBufferedPointSet( rMode, vertexBufferPtr, threeDimensionSize );
    break;
  default:
   volumeGeometry = ( SoNode* ) createVolumeIndexedFaceSet( rMode, vertexBufferPtr, threeDimensionSize );
    break;
  }

  return volumeGeometry;
}


/**
 * Create a VolumeGeometry instance set with vertexBufferPtr for the Skin Rendering mode. 
*/
SoNode* createVolumeGeometry_SkinInstance( 
                                      const VolumeGeometryType vgType, 
                                      const float* vertexBufferPtr,
                                      const SbVec3f& size
                                    )
{
  //The createVolumeGeometryInstance create a volume geometry instance in skin rendering mode
  const RenderingMode rMode = SKIN;
  SoNode*volumeGeometry = NULL;

  switch( vgType )
  {
  case BUFFERED_QUAD_SET:
   volumeGeometry = ( SoNode* ) createVolumeBufferedQuadSet( rMode, vertexBufferPtr, size );
    break;
  case BUFFERED_LINE_SET:
   volumeGeometry = ( SoNode* ) createVolumeBufferedLineSet( rMode, vertexBufferPtr, size );
    break;
  case BUFFERED_POINT_SET:
   volumeGeometry = ( SoNode* ) createVolumeBufferedPointSet( rMode, vertexBufferPtr, size );
    break;
  default:
   volumeGeometry = ( SoNode* ) createVolumeIndexedFaceSet( rMode, vertexBufferPtr, size );
    break;
  }

  return volumeGeometry;
}


/**
 * Create a VolumeGeometry instance set with vertexBufferPtr for the OrthoSlice Rendering mode. 
*/
SoNode* createVolumeGeometry_OrthoSliceInstance( 
                                      const VolumeGeometryType vgType, 
                                      const float* vertexBufferPtr,
                                      const SbVec2f& size
                                    )
{
  //The createVolumeGeometryInstance create a volume geometry instance in orthoslice rendering mode
  const RenderingMode rMode = ORTHOSLICE;
  SoNode*volumeGeometry = NULL;
  const SbVec3f threeDimensionSize( size[0], size[1], 0.f );

  switch( vgType )
  {
  case BUFFERED_QUAD_SET:
   volumeGeometry = ( SoNode* ) createVolumeBufferedQuadSet( rMode, vertexBufferPtr, threeDimensionSize );
    break;
  case BUFFERED_LINE_SET:
   volumeGeometry = ( SoNode* ) createVolumeBufferedLineSet( rMode, vertexBufferPtr, threeDimensionSize );
    break;
  case BUFFERED_POINT_SET:
   volumeGeometry = ( SoNode* ) createVolumeBufferedPointSet( rMode, vertexBufferPtr, threeDimensionSize );
    break;
  default:
   volumeGeometry = ( SoNode* ) createVolumeIndexedFaceSet( rMode, vertexBufferPtr, threeDimensionSize );
    break;
  }

  return volumeGeometry;
}


/**
* Provides the VolumeIndexedFaceSet shape set with the buffer pointed by vertexBufferPtr
* for the rmode Rendering Mode.
*/
SoVolumeIndexedFaceSet* createVolumeIndexedFaceSet(
                                                  const RenderingMode rMode,
                                                  const float* vertexBufferPtr,
                                                  const SbVec3f& size
                                                  )
{
  SoVolumeIndexedFaceSet* IShape = new SoVolumeIndexedFaceSet;

  //Set its name for debug
  IShape->setName( "VolumeIndexedFaceSet" );

  //Set the sizes
  const float sizeX = size[0];
  const float sizeY = size[1];

  //Set the number of vertices
  int numVertices = -1;
  int numIndices = -1;

  //For the skin mode
  int faceOffset[6] = { -1, -1, -1, -1, -1, -1 };

  switch( rMode )
  {
  case NORMAL:
  case ORTHOSLICE:
    numVertices = int( sizeX * sizeY );
    numIndices = int( ( sizeX - 1 ) * ( sizeY - 1 ) * 5 );
    break;

  case SKIN:
    numVertices = int( sizeX * sizeY ) * 6;
    numIndices = int( ( sizeX - 1 ) * ( sizeY - 1 ) * 5 ) * 6;
    for( int face = 0; face < 6; ++face )
      faceOffset[face] = face * int( sizeX * sizeY );
    break;

  default:
    SoDebugError::post( "utils.cxx : createVolumeIndexedFaceSet", "Bad Rendering type" );
    return NULL;
  }

  //Set the vertex buffer pointer
  IShape->vertexProperty = new SoVertexProperty;
  SoVertexProperty* vp = ( SoVertexProperty* ) IShape->vertexProperty.getValue();
  vp->vertex.setValuesPointer( numVertices, vertexBufferPtr );

  //Set the number of indices in the index buffer
  IShape->coordIndex.setNum( numIndices );

  //Set the index buffer
  if( rMode == NORMAL || rMode == ORTHOSLICE )
  {
    int32_t* indices = IShape->coordIndex.startEditing();
    for( int vy = 0; vy < sizeY - 1; ++vy )
    {
      for( int vx=0; vx < sizeX - 1 ; ++vx )
      {
        indices[0] = int32_t( ( vy + 0 ) * sizeY + ( vx + 0 ) );
        indices[1] = int32_t( ( vy + 0 ) * sizeY + ( vx + 1 ) );
        indices[2] = int32_t( ( vy + 1 ) * sizeY + ( vx + 1 ) );
        indices[3] = int32_t( ( vy + 1 ) * sizeY + ( vx + 0 ) );
        indices[4] = -1; // endup the quad
        indices += 5;
      }
    }
    IShape->coordIndex.finishEditing();
  }

  if( rMode == SKIN )
  {
    int32_t* indices = IShape->coordIndex.startEditing();
    for( int face = 0; face < 6; ++face )
    {
      for( int vy = 0; vy < sizeY - 1; ++vy )
      {
        for( int vx = 0; vx < sizeX - 1; ++vx )
        {
          indices[0] = int32_t( faceOffset[face] + ( vy + 0 ) * sizeY + ( vx + 0 ) );
          indices[1] = int32_t( faceOffset[face] + ( vy + 0 ) * sizeY + ( vx + 1 ) );
          indices[2] = int32_t( faceOffset[face] + ( vy + 1 ) * sizeY + ( vx + 1 ) );
          indices[3] = int32_t( faceOffset[face] + ( vy + 1 ) * sizeY + ( vx + 0 ) );
          indices[4] = -1; // endup the quad
          indices += 5;
        }
      }
    }
    IShape->coordIndex.finishEditing();
  }

  return IShape;
}

/**
* Provides the VolumeBufferedShape shape in QUADS mode set with the buffer pointed by vertexBufferPtr
* for the rmode Rendering Mode.
*/
SoVolumeBufferedShape* createVolumeBufferedQuadSet(
                                      const RenderingMode rMode,
                                      const float* vertexBufferPtr,
                                      const SbVec3f& size
                                  )
{
  SoVolumeBufferedShape* BShape = new SoVolumeBufferedShape;

  //Set the Shape type
  BShape->shapeType = SoBufferedShape::QUADS; //Works with the indices
  const unsigned int numVerticesPerPrimitive = 4; //QUADS => 4 vertices per primitive

  //Set the sizes
  const float sizeX = size[0];
  const float sizeY = size[1];

  //Set the sizes in bytes
  const int vertexSize = 3 * sizeof( float ); //3 float components per points
  const int indexSize = sizeof( unsigned int );

  int numVertices = -1;
  int numIndices = -1;
  
  //For the skin mode
  const int numFaces = 6;
  int faceOffset[6] = { -1, -1, -1, -1, -1, -1 };

  switch( rMode )
  {
  case NORMAL:
    //Set its name to simplify debug
    BShape->setName( "VolumeBufferedQuadSet" );
    numVertices = int( sizeX * sizeY );
    numIndices = int( ( sizeX - 1 ) * ( sizeY - 1 ) * numVerticesPerPrimitive );
    break;

  case ORTHOSLICE:
    BShape->setName( "VolumeBufferedQuadSet_OrthoSlice" );
    numVertices = int( sizeX * sizeY );
    numIndices = int( ( sizeX - 1 ) * ( sizeY - 1 ) * numVerticesPerPrimitive );
    break;

  case SKIN:
    BShape->setName( "VolumeBufferedQuadSet_Skin" );
    numVertices = int( sizeX * sizeY ) * numFaces;
    numIndices = int( ( sizeX - 1 ) * ( sizeY - 1 ) * numVerticesPerPrimitive ) * numFaces;
    for( int face = 0; face < numFaces; ++face )
      faceOffset[face] = face * int( sizeX * sizeY );
    break;

  default:
    SoDebugError::post( "utils.cxx : createVolumeBufferedQuadSet", "Bad Rendering type" );
    return NULL;
  }

  //Set the vertex, index and color buffer sizes.
  const unsigned int vertexBufferSize = numVertices * vertexSize;
  const unsigned int indexBufferSize = numIndices * indexSize;

  //Set the number of vertices = number of indices to have a correct display
  BShape->numVertices.set1Value( 0, numIndices );

  //SET THE VERTICES
  //Use a CPUBuffer for the vertices
  SoCpuBufferObject* cpuVertexBuffer = new SoCpuBufferObject;

  //Set its size
  cpuVertexBuffer->setSize( vertexBufferSize );

  //Make it point to the correct address
  cpuVertexBuffer->setBuffer( ( void* ) vertexBufferPtr, vertexBufferSize );

  //The vertices of the buffered shape node point on the cpu buffer
  BShape->vertexBuffer = cpuVertexBuffer;

  //SET THE INDICES
  //Use a CPUBuffer for the indices
  SoCpuBufferObject* cpuIndexBuffer = new SoCpuBufferObject;

  //Set its size
  cpuIndexBuffer->setSize( indexBufferSize );

  //Fill it 
  unsigned int* indices = ( unsigned int* ) cpuIndexBuffer->map( SoBufferObject::SET );

  if( rMode == NORMAL || rMode == ORTHOSLICE )
  {
    for( int vy = 0; vy < sizeY - 1; ++vy )
    {
      for( int vx = 0; vx < sizeX - 1; ++vx )
      {
        indices[0] = (unsigned int)( ( vy + 0 ) * sizeX + ( vx + 0 ) );
        indices[1] = (unsigned int)( ( vy + 0 ) * sizeX + ( vx + 1 ) );
        indices[2] = (unsigned int)( ( vy + 1 ) * sizeX + ( vx + 1 ) );
        indices[3] = (unsigned int)( ( vy + 1 ) * sizeX + ( vx + 0 ) );
        indices += 4;
      }
    }
  }
  if( rMode == SKIN )
  {
    for( int face = 0; face < 6; ++face )
    {
      for( int vy = 0; vy < sizeY - 1; ++vy )
      {
        for( int vx = 0; vx < sizeX - 1; ++vx )
        {
          indices[0] = (unsigned int)( faceOffset[face] + ( vy + 0 ) * sizeX + ( vx + 0 ) );
          indices[1] = (unsigned int)( faceOffset[face] + ( vy + 0 ) * sizeX + ( vx + 1 ) );
          indices[2] = (unsigned int)( faceOffset[face] + ( vy + 1 ) * sizeX + ( vx + 1 ) );
          indices[3] = (unsigned int)( faceOffset[face] + ( vy + 1 ) * sizeX + ( vx + 0 ) );
          indices += 4;
        }
      }
    }
  }
  
  cpuIndexBuffer->unmap();

  //The indices of the buffered shape node point on the cpu index buffer
  BShape->indexBuffer = cpuIndexBuffer;

  //Set the type of the indices
  BShape->indexType = SbDataType::UNSIGNED_INT32;

  return BShape;
}

/**
* Provides the VolumeBufferedShape shape in LINES mode set with the buffer pointed by vertexBufferPtr
* for the rmode Rendering Mode.
*/
SoVolumeBufferedShape* createVolumeBufferedLineSet(
                                      const RenderingMode rMode,
                                      const float* vertexBufferPtr,
                                      const SbVec3f& size
                                  )
{
  SoVolumeBufferedShape* BShape = new SoVolumeBufferedShape;


  //Set the Shape type
  BShape->shapeType = SoBufferedShape::LINES; //Works with the indices

  //Set the sizes
  const float sizeX = size[0];
  const float sizeY = size[1];

  //Set the sizes in bytes
  const int vertexSize = 3 * sizeof( float ); //3 float components per points

  int numVertices = -1;
  
  //For the skin mode
  const int numFaces = 6;

  switch( rMode )
  {
  case NORMAL:
    //Set its name to simplify debug
    BShape->setName( "VolumeBufferedLineSet" );
    numVertices = int( sizeX * sizeY );
    break;

  case ORTHOSLICE:
    BShape->setName( "VolumeBufferedLineSet_OrthoSlice" );
    numVertices = int( sizeX * sizeY );
    break;

  case SKIN:
    BShape->setName( "VolumeBufferedLineSet_Skin" );
    numVertices = int( sizeX * sizeY ) * numFaces;
    break;

  default:
    SoDebugError::post( "utils.cxx : createVolumeBufferedQuadSet", "Bad Rendering type" );
    return NULL;
  }

  //Set the vertex buffer size
  const unsigned int vertexBufferSize = numVertices * vertexSize;

  //Set the number of vertices = number of indices to have a correct display
  BShape->numVertices.set1Value( 0, numVertices );

  //SET THE VERTICES
  //Use a CPUBuffer for the vertices
  SoCpuBufferObject* cpuVertexBuffer = new SoCpuBufferObject;

  //Set its size
  cpuVertexBuffer->setSize( vertexBufferSize );

  //Make it point to the correct address
  cpuVertexBuffer->setBuffer( ( void* ) vertexBufferPtr, vertexBufferSize );

  //The vertices of the buffered shape node point on the cpu buffer
  BShape->vertexBuffer = cpuVertexBuffer;
  
  return BShape;
}


/**
* Provides the VolumeBufferedShape shape in POINTS mode set with the buffer pointed by vertexBufferPtr
* for the rmode Rendering Mode.
*/
SoVolumeBufferedShape* createVolumeBufferedPointSet(
                                      const RenderingMode rMode,
                                      const float* vertexBufferPtr,
                                      const SbVec3f& size
                                  )
{
  SoVolumeBufferedShape* BShape = new SoVolumeBufferedShape;

  //Set the Shape type
  BShape->shapeType = SoBufferedShape::POINTS; //Works with the indices

  //Set the sizes
  const float sizeX = size[0];
  const float sizeY = size[1];

  //Set the sizes in bytes
  const int vertexSize = 3 * sizeof( float ); //3 float components per points

  int numVertices = -1;
    
  //For the skin mode
  const int numFaces = 6;

  switch( rMode )
  {
  case NORMAL:
    //Set its name to simplify debug
    BShape->setName( "VolumeBufferedPointSet" );
    numVertices = int( sizeX * sizeY );
    break;

  case ORTHOSLICE:
    BShape->setName( "VolumeBufferedPointSet_OrthoSlice" );
    numVertices = int( sizeX * sizeY );
    break;

  case SKIN:
    BShape->setName( "VolumeBufferedPointSet_Skin" );
    numVertices = int( sizeX * sizeY ) * numFaces;
    break;

  default:
    SoDebugError::post( "utils.cxx : createVolumeBufferedQuadSet", "Bad Rendering type" );
    return NULL;
  }

  //Set the vertex, index and color buffer sizes.
  const unsigned int vertexBufferSize = numVertices * vertexSize;

  //Set the number of vertices = number of indices to have a correct display
  BShape->numVertices.set1Value( 0, numVertices );

  //SET THE VERTICES
  //Use a CPUBuffer for the vertices
  SoCpuBufferObject* cpuVertexBuffer = new SoCpuBufferObject;

  //Set its size
  cpuVertexBuffer->setSize( vertexBufferSize );

  //Make it point to the correct address
  cpuVertexBuffer->setBuffer( ( void* ) vertexBufferPtr, vertexBufferSize );

  //The vertices of the buffered shape node point on the cpu buffer
  BShape->vertexBuffer = cpuVertexBuffer;

  return BShape;
}

