/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Apr 2011)
**=======================================================================*/
/*=======================================================================
** Modified by : Aymeric Chataigner (Oct 2012)
**=======================================================================*/

#include <VolumeViz/nodes/SoUniformGridClipping.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <Inventor/actions/SoSearchAction.h>
#include <VolumeViz/nodes/SoVolumeIndexedFaceSet.h>
#include <VolumeViz/nodes/SoVolumeBufferedShape.h>

/*to select the right type of volume geometry*/
enum VolumeGeometryType
{
  INDEXED_FACE_SET,
  BUFFERED_QUAD_SET,
  BUFFERED_TRIANGLE_SET,
  BUFFERED_LINE_SET,
  BUFFERED_POINT_SET
};

/*to select the right type of rendering mode*/
enum RenderingMode
{
  NORMAL,
  SKIN,
  ORTHOSLICE
};

class SoNode;
class SoDataSet;
class SoSeparator;


/** helper to search a node by type */
template<typename T> T* searchNode(SoNode* scene);

/** helper to search a node by name */
template<typename T> T* searchName(SoNode* scene, SbName name);


/** create a wireframe representation of a SoVolumeData node */
SoSeparator *makeVolBBox( const SbBox3f &volSize );

/**
 * Generate a synthetic horizon based on the extent of a volumData 
 *
 * @param volumeData
 * @param zcenter
 * @param zscale
 * @param axis
 *
 * @return a shape representing the synthetic horizon
 *
 */
SoNode* generateSyntheticHorizon(
                                 SoDataSet* volumeData,
                                 const float zcenter, 
                                 const float zscale, 
                                 const SoUniformGridClipping::Axis axis
                                 );

/**
 * Generate a synthetic VolumeGeometry representing a sine shape based on the VolumeData extent.
 *
 * @param type
 * @param volumeData
 * @param zcenter
 * @param zscale
 * @param axis
 *
 * @return a shape representing the synthetic sine shape
 *
 */
SoNode* generateSyntheticVolumeGeometry(
                                        VolumeGeometryType type,
                                        SoDataSet* volumeData,
                                        const float zcenter,
                                        const float zscale,
                                        const SoUniformGridClipping::Axis axis
                                        );

/**
 * Generate a synthetic VolumeGeometry representing a sine shape based on the VolumeData extent.
 *
 * @param type
 * @param volumeData
 *
 * @return a shape representing the synthetic skin
 *
 */
SoNode* generateSyntheticVolumeGeometry_Skin(
                                              VolumeGeometryType type,
                                              SoDataSet* volumeData
                                            );


/**
 * Generate a synthetic VolumeGeometry representing an horizon (skin) based on the extent of a VolumeData.
 *
 * @param type
 * @param volumeData
 * @param axis
 *
 * @return a shape representing the synthetic skin
 *
 */
SoNode* generateSyntheticVolumeGeometry_OrthoSlice(
                                                    VolumeGeometryType type,
                                                    SoDataSet* volumeData, 
                                                    SoOrthoSlice::Axis axis
                                                   );

/**
 * Create a VolumeGeometry instance set with vertexBufferPtr. 
 *
 * @param vgType
 * @param vertexBufferPtr
 * @param size
 *
 * @return a shape representing a shape with the vertices pointed by vertexBufferPtr.
 *
 */
SoNode* createVolumeGeometryInstance(
                                      const VolumeGeometryType vgType,
                                      const float* vertexBufferPtr,
                                      const SbVec2f& size
                                    );  

/**
 * Create a VolumeGeometry instance set with vertexBufferPtr for the Skin Rendering mode. 
 *
 * @param vgType
 * @param vertexBufferPtr
 * @param size
 *
 * @return a shape representing a shape with the vertices pointed by vertexBufferPtr
 * for the skin rendering mode.
 */
SoNode* createVolumeGeometry_SkinInstance(
                                          const VolumeGeometryType vgType,
                                          const float* vertexBufferPtr,
                                          const SbVec3f& size
                                          );

/**
 * Create a VolumeGeometry instance set with vertexBufferPtr for the OrthoSlice Rendering mode. 
 *
 * @param vgType
 * @param vertexBufferPtr
 * @param size
 *
 * @return a shape representing a shape with the vertices pointed by vertexBufferPtr
 * for the OrthoSlice rendering mode.
 */
SoNode* createVolumeGeometry_OrthoSliceInstance(
                                      const VolumeGeometryType vgType,
                                      const float* vertexBufferPtr,
                                      const SbVec2f& size
                                    );


/**
 * Provides the VolumeIndexedFaceSet shape set with the buffer pointed by vertexBufferPtr
 * for the rmode Rendering Mode.
 *
 * @param rMode
 * @param vertexBufferPtr
 * @param size
 *
 * @return a pointer on the created VolumeIndexedFaceSet shape.
 *
 */
SoVolumeIndexedFaceSet* createVolumeIndexedFaceSet(
                                      const RenderingMode rMode,
                                      const float* vertexBufferPtr,
                                      const SbVec3f& size
                                      );

/**
 * Provides the VolumeBufferedShape shape in QUADS mode set with the buffer pointed by vertexBufferPtr
 * for the rmode Rendering Mode.
 *
 * @param rMode
 * @param vertexBufferPtr
 * @param size
 *
 * @return a pointer on the created VolumeBufferedShape in QUADS mode.
 *
 */
SoVolumeBufferedShape* createVolumeBufferedQuadSet(
                                      const RenderingMode rMode,
                                      const float* vertexBufferPtr,
                                      const SbVec3f& size
                                  );

/**
 * Provides the VolumeBufferedShape shape in LINES mode set with the buffer pointed by vertexBufferPtr
 * for the rmode Rendering Mode.
 *
 * @param rMode
 * @param vertexBufferPtr
 * @param size
 *
 * @return a pointer on the created VolumeBufferedShape in LINES mode.
 *
 */
SoVolumeBufferedShape* createVolumeBufferedLineSet(
                                      const RenderingMode rMode,
                                      const float* vertexBufferPtr,
                                      const SbVec3f& size
                                  );


/**
 * Provides the VolumeBufferedShape shape in POINTS mode set with the buffer pointed by vertexBufferPtr
 * for the rmode Rendering Mode.
 *
 * @param rMode
 * @param vertexBufferPtr
 * @param size
 *
 * @return a pointer on the created VolumeBufferedShape in POINTS mode.
 *
 */
SoVolumeBufferedShape* createVolumeBufferedPointSet(
                                      const RenderingMode rMode,
                                      const float* vertexBufferPtr,
                                      const SbVec3f& size
                                  );



//
// INLINE IMPLEMENTATION OF TEMPLATE METHODS
//

// search a node by type
template<typename T>
T* 
searchNode(SoNode* scene)
{
  SoSearchAction* sa = new SoSearchAction;
  sa->setFind(SoSearchAction::TYPE);
  sa->setSearchingAll(true);
  sa->setType(T::getClassTypeId());
  sa->apply(scene);

  SoPath* path = sa->getPath();
  if ( !path )
  {
    std::cerr << T::getClassTypeId().getName().getString() << " not found" << std::endl;
    return(NULL);
  }
  return dynamic_cast<T*>(path->getTail());
}

// search a node by name
template<typename T>
T* 
searchName(SoNode* scene, SbName name)
{
  if (!scene)
  {
    T* node=(T*)SoNode::getByName(name);
    if ( node )
      return node;
  }
  else
  {
    SoSearchAction* sa = new SoSearchAction;
    sa->setFind(SoSearchAction::NAME);
    sa->setSearchingAll(true);
    sa->setType(T::getClassTypeId());
    sa->setName(name);
    sa->apply(scene);

    SoPath* path = sa->getPath();
    if ( path )
      return dynamic_cast<T*>(path->getTail());
  }

  std::cerr << T::getClassTypeId().getName().getString() << " not found" << std::endl;
  return(NULL);
}

