/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Pascal Estrade (Jul 2009)
**=======================================================================*/

/*----------------------------------------------------------------------------------------
Medical example program.
Purpose : High Quality rendering for medical data.
Description : Main
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoPhysicalMaterial.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoDataRange.h>
#include <Inventor/nodes/SoGradientBackground.h> 
#include <Inventor/nodes/SoRotationXYZ.h> 
#include <Inventor/nodes/SoImageBackground.h> 
#include <Inventor/nodes/SoTextProperty.h>
#include <Inventor/nodes/SoText2.h> 
#include <Inventor/nodes/SoCube.h> 
#include <Inventor/nodes/SoOrthographicCamera.h> 
#include <Inventor/nodes/SoPerspectiveCamera.h> 
#include <Inventor/nodes/SoTranslation.h> 
#include <Inventor/nodes/SoShadowGroup.h> 
#include <Inventor/nodes/SoFont.h> 
#include <Inventor/nodes/SoInteractiveComplexity.h>


#include "manageEvents.h"

// Data Set
#define FILENAME "$OIVHOME/examples/data/VolumeViz/medicalFoot.ldm"

// color map that can be edited in Avizo.
#define COLORMAPFILENAME "$OIVHOME/examples/data/VolumeViz/colormaps/bonesMuscles.col.am"



//main function
int main(int, char **argv)
{
	// Create the window
	Widget myWindow = SoXt::init(argv[0]);
	if (!myWindow) return 0;

	// ROOT
	SoSeparator *root = new SoSeparator;
	root->ref();    

	// Separator for data.
	SoSeparator *dataSep = new SoSeparator();

	// The camera for data.
	SoPerspectiveCamera *localCam = new SoPerspectiveCamera();

	// Increase the rendering quality when you stop to move the dataset.
	SoInteractiveComplexity* cplx = new SoInteractiveComplexity;
 	
	// Initialize of VolumeViz extension
	SoVolumeRendering::init();

	// All data under this group will be shadowed.
	SoShadowGroup *data = new SoShadowGroup(); 
	data->method.setValue(SoShadowGroup::VARIANCE_SHADOW_MAP);
	data->intensity.setValue(0.7f);
	data->smoothFactor.setValue(4);
	data->isActive = true;
	
	// Separator for the foot support.
	SoSeparator *suppportSep = new SoSeparator() ; 
	SoTranslation *suppTrans = new SoTranslation(); 
	suppTrans->translation.setValue( 10, 0.0, 0.0 );
	
	// The support.
	SoCube *support = new SoCube(); 
	support->depth.setValue( 50.0f );
	support->width.setValue( 50.0f );
	support->height.setValue( 0.05f );
	
	// Node to hold the volume data
	SoVolumeData* pVolData = new SoVolumeData();
	//Load the model in the SoVolumeData
	pVolData->fileName = FILENAME;

	// Load the colorMap from an Avizo colormap.
	SoTransferFunction* pTransFunc = new SoTransferFunction;
  pTransFunc->loadColormap(COLORMAPFILENAME);

	// Set data default range.
	SoDataRange *localDataRange = new SoDataRange;
	localDataRange->min = -868.93f;
	localDataRange->max = 2659.00f;

	// Property node which allows SoVolumeRender to draw High Quality volumes.  
	SoVolumeRenderingQuality *pVRVolQuality = new SoVolumeRenderingQuality;      
	pVRVolQuality->lighting = TRUE;
	pVRVolQuality->preIntegrated = TRUE;
	pVRVolQuality->edgeColoring = TRUE;
	pVRVolQuality->jittering = TRUE;
	pVRVolQuality->gradientQuality.setValue(SoVolumeRenderingQuality::MEDIUM);

	// Node in charge of drawing the volume
	SoVolumeRender* pVolRender = new SoVolumeRender;
	pVolRender->numSlicesControl = SoVolumeRender::MANUAL;
	pVolRender->samplingAlignment = SoVolumeRender::VIEW_ALIGNED;
	pVolRender->numSlices = 512;

	SoLightModel* lightModel = new SoLightModel;
	lightModel->model = SoLightModel::PHYSICALLY_BASED;

	//Set the parameters of the volume material
	SoPhysicalMaterial* volMaterial = new SoPhysicalMaterial;
	volMaterial->baseColor = SbColorRGBA(1.0f, 1.0f, 1.0f, 1.0f);
	volMaterial->specular = 1.0f;
	volMaterial->roughness = 0.4f;

	//Set the parameters of the support material
	SoPhysicalMaterial* supportMaterial = new SoPhysicalMaterial;
	supportMaterial->baseColor = SbColorRGBA(1.0f, 1.0f, 1.0f, 1.0f);
	supportMaterial->roughness = 1.0f;

	// Define background color
	SoGradientBackground *bg = new SoGradientBackground();
	
	SoRotationXYZ *rotX = new SoRotationXYZ();
	rotX->axis = SoRotationXYZ::X;
	rotX->angle.setValue(-3.14f/2);
	SoRotationXYZ *rotZ = new SoRotationXYZ();
	rotZ->axis = SoRotationXYZ::Z;
	rotZ->angle.setValue(-3.14f/2);
	
	// Mouse event callback to manage data range.
	SoEventCallback *mouseMoveEvent = new SoEventCallback();
	SoEventCallback *mouseKeyEvent = new SoEventCallback();

	// Text 2D to inform the user.
	SoSeparator *localAnnotation = new SoSeparator();
	// Text 2D font.
	SoFont *textFont = new SoFont();
	textFont->size.setValue(15);

	// Text 2D properties
	SoTextProperty* textProperty = new SoTextProperty;
	textProperty->alignmentV = SoTextProperty::TOP;
	textProperty->margin = 0.4f;
	
	SoTranslation *textTrans = new SoTranslation();
	textTrans->translation.setValue(-1.0f, 1.0f, 0.0f);
	
	SoText2 *instruction = new SoText2();
	instruction->justification = SoText2::LEFT;
	instruction->string.set1Value(0,"In selection mode :");
	instruction->string.set1Value(1,"  -> Drag horizontally with left mouse button pressed to change the min data range.");
	instruction->string.set1Value(2,"  -> Drag vertically with left mouse button pressed to change the max data range.");

	// Add Children
	root->addChild( dataSep );
	dataSep->addChild( localCam );

	// additional light source
	SoDirectionalLight* light = new SoDirectionalLight;
	light->direction = SbVec3f(0.4f, -0.25f, -0.87f);
	dataSep->addChild( light );
	
	dataSep->addChild( cplx ); 
	cplx->fieldSettings.set1Value(0, "SoVolumeRender numSlices 200 1000 1500");
	cplx->fieldSettings.set1Value(1, "SoVolumeRender lowScreenResolutionScale 2 1 -1");
	cplx->fieldSettings.set1Value(2, "SoVolumeRenderingQuality gradientQuality LOW MEDIUM");
	
	dataSep->addChild( lightModel );
	dataSep->addChild( volMaterial );
	dataSep->addChild( bg );
	dataSep->addChild( data ); 
	
	data->addChild( suppportSep );
	suppportSep->addChild( suppTrans );
	suppportSep->addChild( supportMaterial );
	suppportSep->addChild( support );
	
	data->addChild( rotX );
	data->addChild( rotZ );	
	data->addChild( pVolData );
	data->addChild( localDataRange );
	data->addChild( pTransFunc );
	data->addChild( pVRVolQuality );
	data->addChild( pVolRender );
	data->addChild( mouseMoveEvent );
	data->addChild( mouseKeyEvent );
	
	root->addChild( localAnnotation ); 
	// Camera for Text 2D.
	SoOrthographicCamera* annotCam = new SoOrthographicCamera;
	annotCam->viewportMapping = SoCamera::LEAVE_ALONE;
	localAnnotation->addChild(annotCam);
	localAnnotation->addChild(textFont);
	localAnnotation->addChild(textProperty);
	localAnnotation->addChild(textTrans);
	localAnnotation->addChild(instruction);
	
	// get mouse events to change the colormap ranges.
	mouseMoveEvent->addEventCallback( SoLocation2Event::getClassTypeId(), mouseMoveEventCB, localDataRange);
	mouseKeyEvent->addEventCallback( SoMouseButtonEvent::getClassTypeId(), mouseKeyEventCB, data );

	// Set up viewer:
	SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
	
	myViewer->setSceneGraph(root);
	myViewer->setCamera( localCam );
	myViewer->setDecoration(FALSE);
	myViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
	myViewer->setTitle("Bones & Muscles.");
	myViewer->show();
	myViewer->viewAll();
	localCam->viewAll(data,  myViewer->getViewportRegion());

	SoXt::show(myWindow);
	SoXt::mainLoop();
	delete myViewer;
	root->unref();
  SoVolumeRendering::finish();
	SoXt::finish();
	return 0;
}


