/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to create and use a color map
author : Jerome Hummel
August 2002
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <LDM/nodes/SoDataRange.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>


#include <Inventor/helpers/SbFileHelper.h>

//file name
SbString DataFile = SbFileHelper::expandString("$OIVHOME/examples/data/VolumeViz/virus.am");
SbString fileName = DataFile;

//Main function
int main(int /*argc*/, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init( argv[0] );
  if (!myWindow)return 0;
  
  // Initialize VolumeViz extension
  SoVolumeRendering::init();
  
  // Create volumeData node with a filename
  // Note: VolumeViz will try to open and load the data using one
  //       of the builtin file loaders (see SoVolumeReader).
  SoVolumeData* pVolData = new SoVolumeData();
  pVolData->fileName.setValue( fileName );

  // If necessary, specify the actual range of the data values.
  //    By default VolumeViz maps the entire range of the voxel data type
  //    (e.g. 0..65535 for unsigned short) into the colormap.  This works
  //    great for byte (8 bit) voxels, but not so well for 16 bit voxels
  //    and not at all for floating point voxels. So it's not actually
  //    necessary for this data set, but shown here for completeness.
  //    NOTE: Min/max values are stored in the header for LDM format
  //    files, but for other formats the getMinMax query can take a
  //    long time because VolumeViz has to examine every voxel.
  SoDataRange *pRange = new SoDataRange();
  int voxelSize = pVolData->getDataSize();
  if (voxelSize > 1) {
    double minval, maxval;
    pVolData->getMinMax( minval, maxval );
    pRange->min = minval;
    pRange->max = maxval;
  }
  
  // Transfer function description
  // Using a predefined color map would use for example this code :
  //   pTransFunc->predefColorMap = SoTransferFunction::GREY;
  //
  // But we want to create a new colorMap here.
  // For demo purposes, this code shows how to create the predefined
  // STANDARD colorMap with 256 entries.
  SoTransferFunction* pTransFunc = new SoTransferFunction;
  int numEntries = 256;
  pTransFunc->colorMap.setNum( 256 * 4 );
  for (int i = 0; i < numEntries; i++) {
    float f = (float)i / 255;
    SbColor rgb; 
    rgb.setHSVValue( 0.916f * f, 1, 1 );
    pTransFunc->colorMap.set1Value( 4*i+0, rgb[0] ); // red
    pTransFunc->colorMap.set1Value( 4*i+1, rgb[1] ); // green
    pTransFunc->colorMap.set1Value( 4*i+2, rgb[2] ); // blue
    pTransFunc->colorMap.set1Value( 4*i+3, 0.3f   ); // alpha
  }
  
  // Create the render node to draw the volume based on data and transfer function
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->samplingAlignment = SoVolumeRender::DATA_ALIGNED;
  
  // Assemble the sceneGraph :
  SoSeparator *root = new SoSeparator;
  root->ref();
  
  root->addChild( pVolData );
  root->addChild( pRange );
  root->addChild( pTransFunc );
  root->addChild( pVolRender );
  
  // Viewer
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer( myWindow );
  myViewer->setBackgroundColor( SbVec3f(.7f,.7f,.7f) );
  myViewer->setTitle( "Volume rendering" );
  myViewer->setSceneGraph(root);
  myViewer->show();
  
  // Loop forever
  SoXt::show( myWindow );
  SoXt::mainLoop();
  delete myViewer;
  root->unref();
  SoVolumeRendering::finish();
  SoXt::finish();

  return 0;
}


