/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (Apr 2003)
**=======================================================================*/

/* The class theDefaultLDMConverter is similar to the source code
* of the class SoVolumeConverter (but simplified).
* This may help you in writting a new volume converter.
*/

#include <iterator>

#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/converters/SoVolumeConverter.h>
#include <LDM/converters/SoConverterParameters.h>
#include <Inventor/helpers/SbDataTypeMacros.h>

class theDefaultLDMConverter : public SoVolumeConverter
{
public :
  // Constructor
  theDefaultLDMConverter() {};

  // destructor
  virtual ~theDefaultLDMConverter() {};

protected:

  // redefine from SoVolumeConverter
  virtual SoVolumeReader* getReader( const SbString& /*filename*/, const SbString& /*fileExt*/ )
  {
    // here we could return a custom reader 
    // if NULL then VolumeViz will search the best one depending on extension.
    return NULL;
  };

  // progress callback
  SoVolumeConverter::Abort
    progress( int /*numTilesGenerated*/, int /*numTilesToGenerate*/ )
  {
    return CVT_CONTINUE;
  };

  // custom sampleTile algorithm redefined from SoBaseLDMConverter
  virtual void sampleTile( 
    const SbVec3i32& tileDim, 
    int dataType, 
    int border, 
    const void* const octantTile[8], 
    const int octantExists[8], 
    void *parentTile )
  {

    for ( int octant = 0; octant < 8; octant++ )
    {
      if ( !octantExists[octant] )
        continue;

      SB_DATATYPE_CALL(sampleDecimation, (tileDim, octantTile[octant], parentTile, octant, border, octantExists), dataType);
    }
  }

  // Custom sample algo that use decimation algorithm. 
  // Just take one voxel out of two to fill the parent tile
  template <typename T>
  void 
    sampleDecimation( const SbVec3i32& tileDim,
    const void* const childTile,
    void *parentTile,
    int octant, int border, const int octantExists[8])
  {
    SbVec2i32 shiftParentOctant;
    SbVec3i32 halfTileDim;
    SbVec3f ratio;
    SbVec3i32 shiftOctant;

    getShiftAndHalfTileDim(shiftParentOctant, halfTileDim, tileDim, octant, border);
    getRatio(ratio, shiftOctant, tileDim, halfTileDim, octant, border, octantExists);

    int tileDim1 = tileDim[0];
    int tileDim2 = tileDim[0] * tileDim[1];

    int ijkshiftParent = shiftParentOctant[0];

    T* parentPtr = (T*)parentTile;
    T* octantPtr = (T*)childTile;

    int pLine, pSlice = 0;
    int iShift, jShift, kShift, ioctant, iparent = 0;

    for ( int k = 0; k < halfTileDim[2]*tileDim2; k+=tileDim2 )
    {
      pLine = 0;
      for ( int j = 0; j < halfTileDim[1]*tileDim1; j+=tileDim1 )
      {
        for ( int i = 0; i < halfTileDim[0]; i++ )
        {
          iparent = (k + j + i);
          iShift = (int)floor((i * ratio[0] ) + 0.5);
          jShift = (int)floor((pLine * ratio[1]) + 0.5) * tileDim1;
          kShift = (int)floor((pSlice * ratio[2]) + 0.5) * tileDim2;
          ioctant = ( iShift + shiftOctant[0] ) + ( jShift + tileDim1*shiftOctant[1]) + ( kShift + tileDim2*shiftOctant[2]);

          iparent += ijkshiftParent;
          parentPtr[iparent] = octantPtr[ioctant];
        }

        // If the octant next to current does not exist,
        // fill parent tile with proper value to avoid bad interpolation

        // 1 - Fill X axis with the last value of the line
        // only for even octant
        if ( !(octant % 2) &&  !octantExists[octant+1] )
        {
          int baseOffset = ijkshiftParent + (pSlice * tileDim2) + (tileDim1 * pLine) + halfTileDim[0];
          std::fill(parentPtr + baseOffset, parentPtr + baseOffset + halfTileDim[0], parentPtr[iparent]);
        }
        pLine++;
      }

      // 2 - Fill Y axis with the last line of the octant
      if ( (octant == 0 || octant == 1 || octant == 4 || octant == 5) && !octantExists[octant+2] )
      {
        int dstOffset = ( halfTileDim[0] * (octant%2)) + (tileDim2 * pSlice) + tileDim[0] * halfTileDim[1];
        int srcOffset = ( halfTileDim[0] * (octant%2)) + (tileDim2 * pSlice) + (tileDim[0] * (halfTileDim[1]-1));

        if ( octant == 4 || octant == 5 )
          dstOffset += halfTileDim[2] * tileDim2;

        for ( int y = 0; y < halfTileDim[0]; y++)
        {
          std::copy(parentPtr + srcOffset, parentPtr + srcOffset + halfTileDim[0], parentPtr + dstOffset);

          dstOffset += tileDim[0];
        }
      }

      // 3 - Fill other part of the parent tile
      if ( (octant == 0 ||octant == 4) && !octantExists[octant+3] )
      {
        int dstOffset = halfTileDim[0] + (halfTileDim[1] * tileDim[0] ) + (pSlice * tileDim2);

        if ( octant == 4 )
          dstOffset += halfTileDim[2] * tileDim2;

        int neededByte = ((halfTileDim[1]-1) * tileDim[0]) + halfTileDim[0];
        for ( int i = 0; i < halfTileDim[0]; i++)
        {
          std::fill(parentPtr + dstOffset, parentPtr + dstOffset + halfTileDim[0], parentPtr[neededByte]);
          dstOffset += tileDim[0];
        }
      }
      pSlice++;
    }
  }

private:
  // used by our custom sample tile algo
  void getShiftAndHalfTileDim(SbVec2i32& shiftParentOctant,
    SbVec3i32& halfTileDim,
    const SbVec3i32& tileDim,
    int octant, int border) const
  {
    SbVec3i32 halfTile2 = tileDim / 2;
    SbVec3i32 halfTile1 = tileDim - halfTile2;
    int tileDim1 = tileDim[0];
    int tileDim2 = tileDim[0] * tileDim[1];

    halfTileDim = halfTile1;

    shiftParentOctant = SbVec2i32(0, 0);
    if ( octant & 1 )
    {
      shiftParentOctant[0] = halfTile1[0];
      shiftParentOctant[1] = 1;
      halfTileDim[0] = halfTile2[0];
    }

    if ( octant & 2 )
    {
      shiftParentOctant[0] += tileDim1 * halfTile1[1];
      shiftParentOctant[1] += tileDim1;
      halfTileDim[1] = halfTile2[1];
    }

    if ( octant & 4 )
    {
      shiftParentOctant[0] += tileDim2 * halfTile1[2];
      shiftParentOctant[1] += tileDim2;
      halfTileDim[2] = halfTile2[2];
    }

    if ( border == 0 )
      shiftParentOctant[1] = 0;
  }

  // used by our custom sample tile algo
  void getRatio(SbVec3f &ratio, SbVec3i32 &shiftOctant, SbVec3i32 tileDim, SbVec3i32 halfTileDim, int octant, int border, const int octantExists[8] ) const
  {

    shiftOctant = SbVec3i32(0, 0, 0);

    // Border == 1 is a special case
    if ( border == 1 )
    {
      ratio = SbVec3f(2.0,2.0,2.0);
      if ( octant % 2 )
        shiftOctant[0] = border;

      if  ( octant != 0 && octant != 1 && octant != 4 && octant != 5 )
        shiftOctant[1] = border;

      if ( octant >= 4)
        shiftOctant[2] = border;

      return;
    }

    int leftBorder = int (border / 2);
    int rightBorder = border - leftBorder;

    // Compute the ratio for X axis
    if ( octant % 2 )
    {
      ratio[0] = (float)(tileDim[0] - rightBorder) / (float)halfTileDim[0];
      shiftOctant[0] = leftBorder;
    }
    else
    {
      if ( octantExists[octant+1] )
        ratio[0] = (float)(tileDim[0] - leftBorder) / (float) halfTileDim[0];
      else
        ratio[0] = (float)tileDim[0] / (float) halfTileDim[0];

      shiftOctant[0] = 0;
    }

    // Compute ratio for Y axis
    if ( octant == 0 || octant == 1 || octant == 4 || octant == 5  )
    {
      if ( octantExists[octant+2] )
        ratio[1] = (float)(tileDim[1] - leftBorder) / (float) halfTileDim[1];
      else
        ratio[1] = (float)tileDim[1]/ (float) halfTileDim[1];

      shiftOctant[1] = 0;
    }
    else
    {
      ratio[1] = (float)(tileDim[1] - rightBorder) / (float)halfTileDim[1];
      shiftOctant[1] = leftBorder;
    }

    // Compute ratio for Z axis
    if ( octant < 4 )
    {
      if ( octantExists[octant+4] )
        ratio[2] = (float)(tileDim[2] - leftBorder) / (float) halfTileDim[2];
      else
        ratio[2] = (float)tileDim[2] / (float) halfTileDim[2];

      shiftOctant[2] = 0;
    }
    else
    {
      ratio[2] = (float)(tileDim[2] - rightBorder) / (float)halfTileDim[2];
      shiftOctant[2] = leftBorder;
    }
  }

};

int
main(int argc, char ** argv)
{
  // Initialize VolumeViz (core Open Inventor is automatically initialized)
  // No viewer is needed since we don't render anything.
  SoVolumeRendering::init();

  int ret = 1;
  {
    theDefaultLDMConverter myConverter;
    SoConverterParameters* myConverterParameters = SoConverterParameters::create(argc,argv);
    if (myConverterParameters)
    {
      myConverterParameters->setTileDim(128);
      ret = myConverter.convert( myConverterParameters );
      delete myConverterParameters;
    }
  }
  SoVolumeRendering::finish();
  return ret;
}


