/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate retrieving custom tags from LDM file header
Author  : MHeck
February 2005

Expected output is:

  TileSize:(null)
    U:64
    V:64
    W:64
  MyCustomTags:(null)
    CustomTag1:Some custom info
    CustomTag2:More custom info
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>

#include <LDM/readers/SoVRLdmFileReader.h> 
#include <Inventor/helpers/SbFileHelper.h>

//main function
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;
    
  // Initialize VolumeViz extension
  SoVolumeRendering::init();

  // Node to hold the volume data
  SoVolumeData* pVolData = new SoVolumeData();
  pVolData->ref();

  // Set file to be loaded
  pVolData->fileName = SbFileHelper::expandString("$OIVHOME/examples/source/VolumeViz/customTags/syn_64_custom.ldm");

  // Get a pointer to the volume reader that VolumeViz selected.
  // This will also trigger loading of (at least) the file header.
  SoVolumeReader *pReader = pVolData->getReader();
  if (pReader == NULL) {
    fprintf(stderr,"Error: No reader!");
  }
  else {
    SoVolumeReader::ReaderType rtype = pReader->getReaderType();
    if (rtype != SoVolumeReader::LDM) {
      fprintf(stderr,"Error: Not an LDM reader!\n");
    }
    else {
      // Convert ptr to LDMReader
      SoVRLdmFileReader *pLdmReader = (SoVRLdmFileReader *)pReader;

      // First try to get one of the standard LDM header tags
      const char *tag1 = "TileSize";
      SbXmlTag myTagSection = pLdmReader->getXmlTag( tag1 );
      if (myTagSection.exist()) {
        printf( "%s:%s\n", myTagSection.getName(),
          myTagSection.getText() ? myTagSection.getText() : "<no text>" );
        for (SbXmlTag t = myTagSection.getFirstChildTag();
            t.exist();
            t = t.getNextSiblingTag())
          printf( "  %s:%s\n", t.getName(), t.getText() );
      }
      else {
        fprintf( stderr, "Error: Tag '%s' not found!\n", tag1 );
      }

      // Now try to get the custom tags
      const char *tag2 = "MyCustomTags";
      myTagSection = pLdmReader->getXmlTag( tag2 );
      if (myTagSection.exist()) {
        printf( "%s:%s\n", myTagSection.getName(), 
          myTagSection.getText() ? myTagSection.getText() : "<no text>" );
        for (SbXmlTag t = myTagSection.getFirstChildTag();
            t.exist();
            t = t.getNextSiblingTag())
          printf( "  %s:%s\n", t.getName(), t.getText() );
      }
      else {
        fprintf( stderr, "Error: Tag '%s' not found!\n", tag2 );
      }
    }
  }

// Not available for AutomaticTestTool
//#ifdef WIN32
  // On Windows, if program started by double clicking, shell window will go away
  //puts("Press enter to continue");
  //int c = getchar();
//#endif

  pVolData->unref();
  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}


