/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Thibaut Andrieu (May 2011)
**=======================================================================*/


#include "utils.h"

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <DialogViz/SoDialogVizAll.h>
#include <LDM/nodes/SoDataSet.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <Inventor/manips/SoTransformerManip.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/drawers/SoPolyLineScreenDrawer.h>
#include <Inventor/misc/SbExtrusionGenerator.h>
#include <Inventor/elements/SoViewVolumeElement.h>
#include <Inventor/actions/SoHandleEventAction.h>
#include <Inventor/nodes/SoShape.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <VolumeViz/details/SoVolumeRenderDetail.h>
#include <Inventor/SoPickedPoint.h>
#include <Inventor/events/SoLocation2Event.h>
#include <Inventor/nodes/SoFaceSet.h>



// Global variables
#define DIALOG_FILENAME "$OIVHOME/examples/source/VolumeViz/editing/editingExtrusion/interface.iv"
#define DEFAULT_SCENE "$OIVHOME/examples/source/VolumeViz/editing/editingExtrusion/scene.iv"

// root of scene graph
SoSeparator* g_root = NULL;

// dataset
SoDataSet* g_dataSet = NULL;

// value slider
SoDialogRealSlider* g_value = NULL;

// list of transaction ids that are currently applied
std::vector<int> g_editions;

// list of transaction ids that are currently undone
std::vector<int> g_undoneEditions;

// shapes generated by extrusion
SoSwitch* g_generatedShapes;

//------------------------------------------------------------------------------
// This function will be called each time a line created by the line drawer will be finalized.
// It retrieve points of the line in camera space, extrude it and write it in dataSet (g_dataSet->writeSolidShape)
void
lineDrawerFinalizeCallback( SoPolyLineScreenDrawer::EventArg& eventArg )
{
  SoPolyLineScreenDrawer* lineDrawer = eventArg.getSource();
  SoHandleEventAction* action = eventArg.getAction();

  // If less than 1 point, shape cannot be generated.
  if ( lineDrawer->point.getNum() < 1 )
  {
    lineDrawer->clear();
    return;
  }

  // retrieve points of line in cam space
  std::vector<SbVec2f> lineInCam( lineDrawer->point.getNum() );
  for ( unsigned int i = 0; i < lineInCam.size(); ++i )
    lineInCam[i].setValue( lineDrawer->point[i][0], lineDrawer->point[i][1] );

  //------------------------------------------------------------------------------
  // create a new extruded shape :

  // retrieve path of dataset
  SoSearchAction searchAction;
  searchAction.setNode( g_dataSet );
  searchAction.apply( g_root );
  SoPath* pathToExtrudedShapeRoot = searchAction.getPath();

  // retrieve bbox of volumeData node.
  SbBox3f bbox = getBbox( g_dataSet );

  // create an extruded shape from specified line. Line is extruded along view 
  // direction between bounding box enclosing planes.
  SoShape* extrudedShape = SbExtrusionGenerator::createFrom2DPoints( lineInCam, 
    pathToExtrudedShapeRoot, 
    SoViewVolumeElement::get(action->getState()), 
    bbox );
  if ( extrudedShape == NULL )
  {
    lineDrawer->clear();
    return;
  }

  //------------------------------------------------------------------------------
  // write shape to dataset
  extrudedShape->ref();
  int editionId = 0;
  g_dataSet->startEditing( editionId );
  g_dataSet->editSolidShape( extrudedShape, g_value->value.getValue() );
  g_dataSet->finishEditing( editionId );
  g_editions.push_back( editionId );

  if ( g_generatedShapes != NULL )
    g_generatedShapes->addChild(extrudedShape);

  extrudedShape->unref();

  // don't forget to clear line
  lineDrawer->clear();

  action->setHandled();
}


//------------------------------------------------------------------------------
class ButtonAuditor : public SoDialogPushButtonAuditor
{
  virtual void dialogPushButton(SoDialogPushButton* cpt)
  {
    // undo
    if ( cpt->auditorID.getValue() == "UNDO_BUTTON" )
    {
      if ( g_editions.empty() )
      {
        SoError::post("%s", "Nothing to undo.");
        return;
      }
      g_dataSet->undoEditing( g_editions.back() );
      g_undoneEditions.push_back( g_editions.back() );
      g_editions.pop_back();
      
    }

    // redo
    else if ( cpt->auditorID.getValue() == "REDO_BUTTON" )
    {
      if ( g_undoneEditions.empty() )
      {
        SoError::post("%s", "Nothing to redo.");
        return;
      }
      g_dataSet->redoEditing( g_undoneEditions.back() );
      g_editions.push_back( g_undoneEditions.back() );
      g_undoneEditions.pop_back();
    }

  }
};

//------------------------------------------------------------------------------
int main(int argc, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]); 
  if (myWindow == NULL)
    exit(1);

  // Initialize of VolumeViz extension
  SoDialogViz::init();
  SoVolumeRendering::init();

  SoSeparator* viewerRoot = new SoSeparator;
  viewerRoot->ref();


  //------------------------------------------------------------------------------
  // load scene

  SbString filename;
  if ( argc == 1 )
    filename = DEFAULT_SCENE;
  else
    filename = SbString(argv[1]);

  SoInput myInput;
  if (! myInput.openFile(filename) ) 
  {
    std::cerr << filename.getString() << std::endl;
    exit(1);
  }

  g_root = SoDB::readAll(&myInput);
  if (g_root == NULL) {
    exit(1);
  }
 viewerRoot->addChild( g_root );
 g_dataSet = find<SoDataSet>(g_root, "EditedDataSet");

 SoPolyLineScreenDrawer* lineDrawer = find<SoPolyLineScreenDrawer>( g_root );
 if( lineDrawer != NULL )
   lineDrawer->onFinish.add( lineDrawerFinalizeCallback );

  g_generatedShapes = find<SoSwitch>(g_root, "GeneratedShapes");

  //------------------------------------------------------------------------------
  // Build dialog.

  SoInput myInputInterface;
  if ( !myInputInterface.openFile(DIALOG_FILENAME) ) 
  {
    std::cerr << DIALOG_FILENAME << std::endl;
    exit(1);
  }

  SoGroup *myGroup = SoDB::readAll( &myInputInterface );
  if ( !myGroup )
  {  
    std::cerr << "Cannot create interface." << std::endl;
    exit(1);
  }
  myGroup->ref();

  SoTopLevelDialog* myTop = find<SoTopLevelDialog>( myGroup );
  myTop->ref();

  SoDialogCustom* myCustom = find<SoDialogCustom>( myTop, "DIALOG_CUSTOM" );

  // link buttons auditor
  ButtonAuditor* buttonAuditor = new ButtonAuditor;
  SoDialogPushButton* undoButton = find<SoDialogPushButton>( myTop, "UNDO_BUTTON" );
  undoButton->addAuditor( buttonAuditor );

  SoDialogPushButton* redoButton = find<SoDialogPushButton>( myTop, "REDO_BUTTON" );
  redoButton->addAuditor( buttonAuditor );

  // value slider
  g_value = find<SoDialogRealSlider>( myTop, "VALUE" );

  // show renderer combobox
  SoDialogComboBox* comboBox = find<SoDialogComboBox>( myTop, "SHOW_SELECT" );
  SoSeparator* renderer = find<SoSeparator>(g_root, "Renderer");
  SoSwitch* selectRenderer = find<SoSwitch>(renderer);
  for ( int i = 0; i < selectRenderer->getNumChildren(); ++i )
  {
    SoNode* child = selectRenderer->getChild(i);
    comboBox->items.set1Value(i, child->getName().getString());
  }
  selectRenderer->whichChild.connectFrom( &comboBox->selectedItem );


  myTop->buildDialog(myWindow, TRUE);
  myTop->show();

  SoXtExaminerViewer* myViewer = new SoXtExaminerViewer(myCustom->getWidget());

  myViewer->setSceneGraph(viewerRoot);
  viewerRoot->unref();

  // EACH_FRAME avoid having a small lag when starting drawing lasso
  myViewer->setFastEditSavePolicy( SoGLRenderAction::EACH_FRAME );

  myViewer->viewAll();
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();

  delete buttonAuditor;
  delete myViewer;
  myTop->unref();
  myGroup->unref();


  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();

  return 0;
}


