/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Thibaut Andrieu (May 2011)
**=======================================================================*/


#include "utils.h"

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <DialogViz/SoDialogVizAll.h>
#include <LDM/nodes/SoDataSet.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <Inventor/manips/SoTransformerManip.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/drawers/SoPolyLineScreenDrawer.h>
#include <Inventor/misc/SbExtrusionGenerator.h>
#include <Inventor/elements/SoViewVolumeElement.h>
#include <Inventor/actions/SoHandleEventAction.h>
#include <Inventor/nodes/SoShape.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <VolumeViz/details/SoVolumeRenderDetail.h>
#include <Inventor/SoPickedPoint.h>
#include <Inventor/events/SoLocation2Event.h>
#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/drawers/SoLassoScreenDrawer.h>
#include <Inventor/drawers/SoPolygonScreenDrawer.h>


// Global variables
#define DIALOG_FILENAME "$OIVHOME/examples/source/VolumeViz/editing/editingFeature/interface.iv"
#define DEFAULT_SCENE "$OIVHOME/examples/source/VolumeViz/editing/editingFeature/sceneMask.iv"
//#define DEFAULT_SCENE "$OIVHOME/examples/source/VolumeViz/editing/editingFeature/sceneInMemory.iv"

#if !defined isNaN
#define isNaN(x_) (x_ != x_)
#endif

// root of scene graph
SoSeparator* g_root = NULL;

// dataset
SoVolumeData* g_dataSet = NULL;

// subVolume to edit
SoSeparator* g_subVolume = NULL;

// thickness slider
SoDialogIntegerSlider* g_thickness = NULL;

// value slider
SoDialogIntegerSlider* g_value = NULL;

// shapes generated by lasso or polygon.
SoSwitch* g_generatedShapes;

// list of transaction ids that are currently applied
std::vector<int> g_editions;

// list of transaction ids that are currently undone
std::vector<int> g_undoneEditions;

//------------------------------------------------------------------------------
// Convert a bbox expressed in world space to a bbox expressed in dataSet space.
// bbox and dataSet must be in same space (not necessary the a world space, but the same one).
SbBox3i32 worldToDS( const SbBox3f bbox, SoDataSet* dataSet )
{
  SbVec3f dsSizeInWorld = dataSet->extent.getValue().getSize();
  SbVec3i32 dsSize = dataSet->getDimension();

  float ratioX = (float)dsSize[0] / dsSizeInWorld[0];
  float ratioY = (float)dsSize[1] / dsSizeInWorld[1];
  float ratioZ = (float)dsSize[2] / dsSizeInWorld[2];

  SbVec3f min = bbox.getMin() - dataSet->extent.getValue().getMin();
  SbVec3f max = bbox.getMax() - dataSet->extent.getValue().getMin();

  SbVec3f minInDS( ratioX * min[0], ratioY * min[1], ratioZ * min[2] );
  SbVec3f maxInDS( ratioX * max[0], ratioY * max[1], ratioZ * max[2] );

  return SbBox3i32( (int)floorf(minInDS[0]), (int)floorf(minInDS[1]), (int)floorf(minInDS[2]), (int)ceilf(maxInDS[0]), (int)ceilf(maxInDS[1]), (int)ceilf(maxInDS[2]) );
}


//------------------------------------------------------------------------------
// Called when lasso is finished.
// Retrieve points of the line in camera space, extrude it and edit dataset.
void
lassoFinishCallback( SoPolyLineScreenDrawer::EventArg& eventArg )
{
  SoPolyLineScreenDrawer* lineDrawer = eventArg.getSource();
  SoHandleEventAction* action = eventArg.getAction();

  // If less than 1 point, shape cannot be generated.
  if ( lineDrawer->point.getNum() < 1 )
  {
    lineDrawer->clear();
    return;
  }

  // retrieve points of line in cam space
  std::vector<SbVec2f> lineInCam( lineDrawer->point.getNum() );
  for ( unsigned int i = 0; i < lineInCam.size(); ++i )
    lineInCam[i].setValue( lineDrawer->point[i][0], lineDrawer->point[i][1] );

  //------------------------------------------------------------------------------
  // create a new extruded shape :

  // retrieve path of dataset
  SoSearchAction searchAction;
  searchAction.setNode( g_dataSet );
  searchAction.apply( g_root );
  SoPath* pathToExtrudedShapeRoot = searchAction.getPath();

  // retrieve bbox of volumeData node.
  SbBox3f bbox = getBbox( g_dataSet );

  // create an extruded shape from specified line. Line is extruded along view 
  // direction between bounding box enclosing planes.
  SoShape* extrudedShape = SbExtrusionGenerator::createFrom2DPoints( lineInCam, 
    pathToExtrudedShapeRoot, 
    SoViewVolumeElement::get(action->getState()), 
    bbox );
  if ( extrudedShape == NULL )
  {
    lineDrawer->clear();
    return;
  }

  //------------------------------------------------------------------------------
  // write shape to dataset
  extrudedShape->ref();
  int editionId = 0;
  g_dataSet->startEditing( editionId );
  g_dataSet->editSolidShape( extrudedShape, g_value->value.getValue() );
  g_dataSet->finishEditing( editionId );
  g_editions.push_back( editionId );

  if ( g_generatedShapes != NULL )
    g_generatedShapes->addChild(extrudedShape);
  extrudedShape->unref();

  // don't forget to clear line
  lineDrawer->clear();

  action->setHandled();
}

//------------------------------------------------------------------------------
// Called when the polygon is finalized. Compute intersection of each polygon point
// with volume, convert to data space, generate a surface and edit dataset.
void
polygonFinishCallback( SoPolyLineScreenDrawer::EventArg& eventArg )
{
  SoPolyLineScreenDrawer* drawer = eventArg.getSource();
  SoHandleEventAction* action = eventArg.getAction();

  // generated surface
  SoFaceSet* surface = new SoFaceSet;
  surface->ref();
  SoVertexProperty* vertexProperty = new SoVertexProperty;
  surface->vertexProperty.setValue( vertexProperty );

  // for each point of polygon
  for ( int i = 0; i < drawer->point.getNum(); ++i )
  {
    const SbVec2f point = *drawer->point.getValues(i);

    // retrieve clicked point
    SoRayPickAction pickaction = SoRayPickAction( action->getViewportRegion() );
    pickaction.setNormalizedPoint( 0.5 * (point + SbVec2f(1.0f, 1.0f)) ); // [-1, 1] to [0, 1]
    pickaction.setSceneManager( action->getSceneManager() );
    pickaction.apply( g_root );

    SoPickedPoint* pickedPoint = pickaction.getPickedPoint();
    if ( !pickedPoint )
    {
      SoError::post("Polygon point %d is not on dataset. Skip it.", i);
      continue;
    }

    vertexProperty->vertex.set1Value( vertexProperty->vertex.getNum(), pickedPoint->getPoint() );

  }

  // Surface needs at least 3 points.
  if ( vertexProperty->vertex.getNum() < 3 )
  {
    SoError::post("%s", "Polygon needs at least 3 points");
    drawer->clear();
    surface->unref();
    return;
  }

  surface->numVertices.set1Value( 0, vertexProperty->vertex.getNum() );

  // Edit dataset with generated surface.
  int editionId = 0;
  g_dataSet->startEditing( editionId );
  g_dataSet->editSurfaceShape( surface, (float)g_thickness->value.getValue(), g_value->value.getValue() );
  g_dataSet->finishEditing( editionId );
  g_editions.push_back( editionId );

  if ( g_generatedShapes != NULL )
    g_generatedShapes->addChild( surface );
  surface->unref();
  drawer->clear();

  action->setHandled();
}

//------------------------------------------------------------------------------
// Create a buffer representing a sphere. Returned buffer must be manually deleted.
SoBufferObject*
createBuffer( const SbVec3i32& buffSize, SbDataType dataType )
{
  SbDataType::DataValue valueToWrite = dataType.cast(g_value->value.getValue());

  // generate buffer
  SoCpuBufferObject* buffer = new SoCpuBufferObject;
  buffer->setSize( buffSize[0] * buffSize[1] * buffSize[2] * dataType.getSize() );

  unsigned char* ptr = (unsigned char*)buffer->map( SoCpuBufferObject::SET );
  for ( int k = 0; k < buffSize[2]; ++k )
  {
    for ( int j = 0; j < buffSize[1]; ++j )
    {
      for ( int i = 0; i < buffSize[0]; ++i )
      {
        float x = 2.0f * (float)(i - (buffSize[0] / 2)) / (float)buffSize[0];
        float y = 2.0f * (float)(j - (buffSize[1] / 2)) / (float)buffSize[1];
        float z = 2.0f * (float)(k - (buffSize[2] / 2)) / (float)buffSize[2];

        if ( x*x + y*y + z*z < 1 )
          memcpy( ptr, &valueToWrite, dataType.getSize() );
        else
          memset( ptr, 0, dataType.getSize() );

        ptr += dataType.getSize();
      }
    }
  }
  buffer->unmap();

  return buffer;
}

//------------------------------------------------------------------------------
// callback called during save editing.
// Display percent of saving process.
SbBool saveEditingCallback( SoDataSet* /*dataSet*/,
                           SoVolumeWriter* /*writer*/, 
                           int tilesRemaining, 
                           int totalTiles )
{
  if ( tilesRemaining == totalTiles )
  {
    std::cout << "Save editing" << std::endl;
  }
  else if ( tilesRemaining > 0 )
  {
    std::cout << (int)(100.0f * (1.0f - (float)tilesRemaining/(float)totalTiles)) << "%" << "\r";
  }
  else if ( tilesRemaining == 0 )
  {
    std::cout << "100%" << std::endl;
    std::cout << "done" << std::endl;
  }

  return TRUE;
}

//------------------------------------------------------------------------------
// Auditor used to manage buttons
class ButtonAuditor : public SoDialogPushButtonAuditor
{
  virtual void dialogPushButton(SoDialogPushButton* cpt)
  {
    // edit
    if ( cpt->auditorID.getValue() == "EDIT_BUTTON" )
    {
      int editionId = 0;
      g_dataSet->startEditing( editionId );

      SoSwitch* editors = find<SoSwitch>(g_root, "Editors");
      // SubVolume
      if ( editors->whichChild.getValue() == 0 )
      {
        // retrieve bbox of subVolume and convert it to DataSet space.
        SoSeparator* subVolume = find<SoSeparator>(editors, "SubVolume");
        SbViewportRegion fakeRegion(1, 1);
        SoGetBoundingBoxAction gba( fakeRegion );
        gba.apply( subVolume );
        SbBox3f bbox = gba.getBoundingBox();
        SbBox3i32 bboxInDS = worldToDS( bbox, g_dataSet );
        g_dataSet->editSubVolume( bboxInDS, g_value->value.getValue() );
      }
      // SubVolumeBuffer
      else if ( editors->whichChild.getValue() == 1 )
      {
        // retrieve bbox of subVolume and convert it to DataSet space.
        SoSeparator* subVolume = find<SoSeparator>(editors, "SubVolumeBuffer");
        SbViewportRegion fakeRegion(1, 1);
        SoGetBoundingBoxAction gba( fakeRegion );
        gba.apply( subVolume );
        SbBox3f bbox = gba.getBoundingBox();
        SbBox3i32 bboxInDS = worldToDS( bbox, g_dataSet );
        SoRef<SoBufferObject> buff = createBuffer( bboxInDS.getSize() + SbVec3i32(1, 1, 1), g_dataSet->getDataType() );
        g_dataSet->editSubVolume( bboxInDS, buff.ptr() );
      }

      g_dataSet->finishEditing( editionId );
      g_editions.push_back( editionId );
    }

    // undo
    else if ( cpt->auditorID.getValue() == "UNDO_BUTTON" )
    {
      if ( g_editions.empty() )
      {
        SoError::post("%s", "Nothing to undo.");
        return;
      }
      g_dataSet->undoEditing( g_editions.back() );
      g_undoneEditions.push_back( g_editions.back() );
      g_editions.pop_back();
      
    }

    // redo
    else if ( cpt->auditorID.getValue() == "REDO_BUTTON" )
    {
      if ( g_undoneEditions.empty() )
      {
        SoError::post("%s", "Nothing to redo.");
        return;
      }
      g_dataSet->redoEditing( g_undoneEditions.back() );
      g_editions.push_back( g_undoneEditions.back() );
      g_undoneEditions.pop_back();
    }

    // save
    else if ( cpt->auditorID.getValue() == "SAVE_BUTTON" )
    {
      g_dataSet->saveEditing(TRUE, std::vector<char*>(), saveEditingCallback);
      g_editions.clear();
      g_undoneEditions.clear();
    }
  }

};

//------------------------------------------------------------------------------
int main(int argc, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]); 
  if (myWindow == NULL)
    exit(1);

  // Initialize of VolumeViz extension
  SoDialogViz::init();
  SoVolumeRendering::init();

  SoSeparator* viewerRoot = new SoSeparator;
  viewerRoot->ref();

  SoLDMGlobalResourceParameters::setMaxMainMemory(16);
  SoLDMGlobalResourceParameters::setMaxTexMemory(16);


  //------------------------------------------------------------------------------
  // load scene

  SbString filename;
  if ( argc == 1 )
    filename = DEFAULT_SCENE;
  else
    filename = SbString(argv[1]);

  SoInput myInput;
  if (! myInput.openFile(filename) ) 
  {
    std::cerr << filename.getString() << std::endl;
    exit(1);
  }

  g_root = SoDB::readAll(&myInput);
  if (g_root == NULL) {
    exit(1);
  }
 viewerRoot->addChild( g_root );
 g_dataSet = find<SoVolumeData>(g_root, "EditedDataSet");

 g_generatedShapes = find<SoSwitch>(g_root, "GeneratedShapes");

 SoLassoScreenDrawer* lassoDrawer = find<SoLassoScreenDrawer>( g_root );
 if( lassoDrawer != NULL )
   lassoDrawer->onFinish.add( lassoFinishCallback );

 SoPolygonScreenDrawer* polygonDrawer = find<SoPolygonScreenDrawer>( g_root );
 if( polygonDrawer != NULL )
   polygonDrawer->onFinish.add( polygonFinishCallback );


  //------------------------------------------------------------------------------
  // Build dialog.

  SoInput myInputInterface;
  if ( !myInputInterface.openFile(DIALOG_FILENAME) ) 
  {
    std::cerr << DIALOG_FILENAME << std::endl;
    exit(1);
  }

  SoGroup *myGroup = SoDB::readAll( &myInputInterface );
  if ( !myGroup )
  {  
    std::cerr << "Cannot create interface." << std::endl;
    exit(1);
  }
  myGroup->ref();

  SoTopLevelDialog* myTop = find<SoTopLevelDialog>( myGroup );
  myTop->ref();

  SoDialogCustom* myCustom = find<SoDialogCustom>( myTop, "DIALOG_CUSTOM" );

  // fill content of tools radio button
  SoSwitch* editors = find<SoSwitch>( g_root, "Editors" );
  SoDialogRadioButtons* toolsRadioButton = find<SoDialogRadioButtons>( myTop, "TOOL_SELECT" );
  for ( int i = 0; i < editors->getNumChildren(); ++i )
  {
    SoNode* node = editors->getChild(i);
    toolsRadioButton->addItem( node->getName().getString() );
  }
  editors->whichChild.connectFrom( &toolsRadioButton->selectedItem );


  // link buttons auditor
  ButtonAuditor* buttonAuditor = new ButtonAuditor;
  SoDialogPushButton* editButton = find<SoDialogPushButton>( myTop, "EDIT_BUTTON" );
  editButton->addAuditor( buttonAuditor );

  SoDialogPushButton* undoButton = find<SoDialogPushButton>( myTop, "UNDO_BUTTON" );
  undoButton->addAuditor( buttonAuditor );

  SoDialogPushButton* redoButton = find<SoDialogPushButton>( myTop, "REDO_BUTTON" );
  redoButton->addAuditor( buttonAuditor );

  SoDialogPushButton* saveButton = find<SoDialogPushButton>( myTop, "SAVE_BUTTON" );
  saveButton->addAuditor( buttonAuditor );

  // thickness 
  g_thickness = find<SoDialogIntegerSlider>( myTop, "THICKNESS" );

  // value
  g_value = find<SoDialogIntegerSlider>( myTop, "VALUE" );


  myTop->buildDialog(myWindow, TRUE);
  myTop->show();

  SoXtExaminerViewer* myViewer = new SoXtExaminerViewer(myCustom->getWidget());
  myViewer->setFastEditSavePolicy(SoGLRenderAction::EACH_FRAME);

  myViewer->setSceneGraph(viewerRoot);
  viewerRoot->unref();


  myViewer->viewAll();
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();

  delete buttonAuditor;
  delete myViewer;
  myTop->unref();
  myGroup->unref();


  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();

  return 0;
}


