/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Thibaut Andrieu (Mar 2011)
**=======================================================================*/

#include <Inventor/nodes/SoNode.h>
#include <DialogViz/dialog/SoTopLevelDialog.h>
#include <Inventor/actions/SoSearchPathAction.h>
#include <VolumeViz/nodes/SoVolumeData.h>


//------------------------------------------------------------------------------
// convenient function to find first specified node in specified graph.
template <typename NodeType>
NodeType* 
find( SoNode* root, const SbString& nodeName = SbString() )
{
  if ( root == NULL )
    return NULL;

  root->ref();
  SoSearchAction* spa = NULL;
  if ( !nodeName.isEmpty() )
  {
    spa = new SoSearchPathAction;
    ((SoSearchPathAction*)spa)->setSearchString( nodeName.getString() );
  }
  else
  {
    spa = new SoSearchAction;
  }

  spa->setSearchingAll( TRUE );
  spa->setType( NodeType::getClassTypeId() );
  spa->apply( root );
  if ( spa->getPath() == NULL )
  {
    root->unref();
    delete spa;
    std::cerr << "Cannot find \"" << nodeName << "\" of type \"" << NodeType::getClassTypeId().getName() << "\" in scene graph." << std::endl;
    return NULL;
  }

  NodeType* node = dynamic_cast<NodeType*>( spa->getPath()->getTail() );
  if ( node == NULL )
  {
    root->unref();
    delete spa;
    std::cerr << "Cannot find \"" << nodeName << "\" of type \"" << NodeType::getClassTypeId().getName() << "\" in scene graph." << std::endl;
    return NULL;
  }

  root->unref();
  delete spa;
  return node;
}

//------------------------------------------------------------------------------
// convenient function to find first specified node in specified GUI graph.
template <typename NodeType>
NodeType* 
find( SoTopLevelDialog* root, const SbString& auditorId = SbString(), bool exitIfNotFound = true)
{
  if ( root == NULL )
    return NULL;

  NodeType* node = dynamic_cast<NodeType*>( root->searchForAuditorId(auditorId) );
  if ( node == NULL )
  {
    std::cerr << "Cannot find " << auditorId << " of type " << NodeType::getClassTypeId().getName() << " in interface." << std::endl;
    if ( exitIfNotFound )
      exit(1);
  }

  return node;
}

//------------------------------------------------------------------------------
// convenient function to retrieve boundingBox of specified node. 
// If node contains SoVolumeData, use SoVolumeData extent.
SbBox3f getBbox( SoNode* node )
{
  // retrieve root node of volume data and compute its boundingBox.
  // Try to find an SoVolumeData in specified volumeData. If SoVolumeData found, 
  // use its extend as bbox. Else, apply an SoGetBoundingBoxAction on g_volumeDataRoot.
  SoVolumeData* volumeData = find<SoVolumeData>( node );
  if ( volumeData != NULL )
  {
    return volumeData->extent.getValue();
  }
  else
  {
    // Use fake viewport for SoGetBoundingBoxAction. see SoGetBoundingBoxAction 
    // constructor for details.
    SbViewportRegion fakeViewportRegion(1, 1);
    SoGetBoundingBoxAction gba( fakeViewportRegion );
    gba.apply( node );
    return gba.getBoundingBox();
  }
}


