/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Thibaut Andrieu (May 2011)
**=======================================================================*/


#include "utils.h"

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <DialogViz/SoDialogVizAll.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <Inventor/manips/SoTransformerManip.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/drawers/SoPolyLineScreenDrawer.h>
#include <Inventor/misc/SbExtrusionGenerator.h>
#include <Inventor/elements/SoViewVolumeElement.h>
#include <Inventor/actions/SoHandleEventAction.h>
#include <Inventor/nodes/SoShape.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <VolumeViz/details/SoVolumeRenderDetail.h>
#include <Inventor/SoPickedPoint.h>
#include <Inventor/events/SoLocation2Event.h>
#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/drawers/SoPolygonScreenDrawer.h>
#include <VolumeViz/details/SoOrthoSliceDetail.h>
#include <VolumeViz/details/SoObliqueSliceDetail.h>


// Global variables
#define DIALOG_FILENAME "$OIVHOME/examples/source/VolumeViz/editing/editingPicking/interface.iv"
#define DEFAULT_SCENE "$OIVHOME/examples/source/VolumeViz/editing/editingPicking/scene.iv"

#if !defined isNaN
#define isNaN(x_) (x_ != x_)
#endif

// root of scene graph
SoSeparator* g_root = NULL;

// dataset
SoVolumeData* g_dataSet = NULL;

// polygon drawn during mouse displacement
SoPolygonScreenDrawer* g_polygon = NULL;

// thickness slider
SoDialogIntegerSlider* g_thickness = NULL;

// list of transaction ids that are currently applied
std::vector<int> g_editions;

// list of transaction ids that are currently undone
std::vector<int> g_undoneEditions;

// shapes generated by extrusion
SoSwitch* g_generatedShapes;


//------------------------------------------------------------------------------
// Called when the polygon is finalized. Compute intersection of each polygon point
// with volume, convert to data space, generate a surface and voxelize it.
void
finishCallback( SoPolyLineScreenDrawer::EventArg& eventArg )
{
  SoPolyLineScreenDrawer* drawer = eventArg.getSource();
  SoHandleEventAction* action = eventArg.getAction();

  // generated surface
  SoFaceSet* surface = new SoFaceSet;
  surface->ref();
  SoVertexProperty* vertexProperty = new SoVertexProperty;
  surface->vertexProperty.setValue( vertexProperty );

  // value of first clicked point
  double firstValue = std::numeric_limits<double>::quiet_NaN();

  // for each point of polygon
  for ( int i = 0; i < drawer->point.getNum(); ++i )
  {
    const SbVec2f point = *drawer->point.getValues(i);

    // retrieve clicked point
    SoRayPickAction pickaction = SoRayPickAction( action->getViewportRegion() );
    pickaction.setNormalizedPoint( 0.5 * (point + SbVec2f(1.0f, 1.0f)) ); // [-1, 1] to [0, 1]
    pickaction.setSceneManager( action->getSceneManager() );
    pickaction.apply( g_root );

    SoPickedPoint* pickedPoint = pickaction.getPickedPoint();
    if ( !pickedPoint )
    {
      SoError::post("Polygon point %d is not on dataset. Skip it.", i);
      continue;
    }

    vertexProperty->vertex.set1Value( vertexProperty->vertex.getNum(), pickedPoint->getPoint() );

    // If we still not have 1st point value, compute it.
    SbVec3i32 firstValuePos;
		if ( isNaN(firstValue) )
    {
      // depending on what was clicked, retrieve value of clicked voxel if possible.
      const SoDetail* detail = pickedPoint->getDetail();
      if ( detail != NULL && detail->getTypeId().isDerivedFrom(SoVolumeDetail::getClassTypeId()) )
      {
        ((SoVolumeDetail*) detail)->getFirstNonTransparentValue(firstValue, firstValuePos);
      }
      else if ( detail != NULL && detail->getTypeId().isDerivedFrom(SoOrthoSliceDetail::getClassTypeId()) )
      {
        firstValue = ((SoOrthoSliceDetail*) detail)->getValueD();
      }
      else if ( detail != NULL && detail->getTypeId().isDerivedFrom(SoObliqueSliceDetail::getClassTypeId()) )
      {
        firstValue = ((SoObliqueSliceDetail*) detail)->getValueD();
      }
      else
      {
        SoError::post("%s", "Cannot retrieve value of clicked point using current rendering mode.");
        drawer->clear();
        surface->unref();
        action->setHandled();
        return;
      }
    }

  }

  // Surface needs at least 3 points.
  if ( vertexProperty->vertex.getNum() < 3 )
  {
    SoError::post("%s", "Polygon needs at least 3 points");
    drawer->clear();
    surface->unref();
    action->setHandled();
    return;
  }

  surface->numVertices.set1Value( 0, vertexProperty->vertex.getNum() );

  // Edit dataset with generated surface.
  int editionId = 0;
  g_dataSet->startEditing( editionId );
  g_dataSet->editSurfaceShape( surface, (float)g_thickness->value.getValue(), firstValue );
  g_dataSet->finishEditing( editionId );
  g_editions.push_back( editionId );

  if ( g_generatedShapes != NULL )
    g_generatedShapes->addChild(surface);

  surface->unref();
  drawer->clear();

  action->setHandled();
}

//------------------------------------------------------------------------------
class ButtonAuditor : public SoDialogPushButtonAuditor
{
  virtual void dialogPushButton(SoDialogPushButton* cpt)
  {
    // undo
    if ( cpt->auditorID.getValue() == "UNDO_BUTTON" )
    {
      if ( g_editions.empty() )
      {
        SoError::post("%s", "Nothing to undo.");
        return;
      }
      g_dataSet->undoEditing( g_editions.back() );
      g_undoneEditions.push_back( g_editions.back() );
      g_editions.pop_back();
      
    }

    // redo
    else if ( cpt->auditorID.getValue() == "REDO_BUTTON" )
    {
      if ( g_undoneEditions.empty() )
      {
        SoError::post("%s", "Nothing to redo.");
        return;
      }
      g_dataSet->redoEditing( g_undoneEditions.back() );
      g_editions.push_back( g_undoneEditions.back() );
      g_undoneEditions.pop_back();
    }

  }
};

//------------------------------------------------------------------------------
int main(int argc, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]); 
  if (myWindow == NULL)
    exit(1);

  // Initialize of VolumeViz extension
  SoDialogViz::init();
  SoVolumeRendering::init();

  SoSeparator* viewerRoot = new SoSeparator;
  viewerRoot->ref();


  //------------------------------------------------------------------------------
  // load scene

  SbString filename;
  if ( argc == 1 )
    filename = DEFAULT_SCENE;
  else
    filename = SbString(argv[1]);

  SoInput myInput;
  if (! myInput.openFile(filename) ) 
  {
    std::cerr << filename.getString() << std::endl;
    exit(1);
  }

  g_root = SoDB::readAll(&myInput);
  if (g_root == NULL) {
    exit(1);
  }
  viewerRoot->addChild( g_root );
  g_dataSet = find<SoVolumeData>(g_root, "EditedDataSet");

  SoPolygonScreenDrawer* polylineDrawer = find<SoPolygonScreenDrawer>(g_root);
  polylineDrawer->onFinish.add( finishCallback );

  g_generatedShapes = find<SoSwitch>(g_root, "GeneratedShapes");


  //------------------------------------------------------------------------------
  // Build dialog.

  SoInput myInputInterface;
  if ( !myInputInterface.openFile(DIALOG_FILENAME) ) 
  {
    std::cerr << DIALOG_FILENAME << std::endl;
    exit(1);
  }

  SoGroup *myGroup = SoDB::readAll( &myInputInterface );
  if ( !myGroup )
  {  
    std::cerr << "Cannot create interface." << std::endl;
    exit(1);
  }
  myGroup->ref();

  SoTopLevelDialog* myTop = find<SoTopLevelDialog>( myGroup );
  myTop->ref();

  SoDialogCustom* myCustom = find<SoDialogCustom>( myTop, "DIALOG_CUSTOM" );

  // thickness slider
  g_thickness = find<SoDialogIntegerSlider>( myTop, "THICKNESS");

  // link buttons auditor
  ButtonAuditor* buttonAuditor = new ButtonAuditor;
  SoDialogPushButton* undoButton = find<SoDialogPushButton>( myTop, "UNDO_BUTTON" );
  undoButton->addAuditor( buttonAuditor );

  SoDialogPushButton* redoButton = find<SoDialogPushButton>( myTop, "REDO_BUTTON" );
  redoButton->addAuditor( buttonAuditor );

  // show renderer combobox
  SoDialogComboBox* comboBox = find<SoDialogComboBox>( myTop, "SHOW_SELECT" );
  SoSeparator* renderer = find<SoSeparator>(g_root, "Renderer");
  SoSwitch* selectRenderer = find<SoSwitch>(renderer);
  for ( int i = 0; i < selectRenderer->getNumChildren(); ++i )
  {
    SoNode* child = selectRenderer->getChild(i);
    comboBox->items.set1Value(i, child->getName().getString());
  }
  selectRenderer->whichChild.connectFrom( &comboBox->selectedItem );

  myTop->buildDialog(myWindow, TRUE);
  myTop->show();

  SoXtExaminerViewer* myViewer = new SoXtExaminerViewer(myCustom->getWidget());

  myViewer->setSceneGraph(viewerRoot);
  viewerRoot->unref();

  // EACH_FRAME avoid having a small lag when starting drawing lasso
  myViewer->setFastEditSavePolicy( SoGLRenderAction::EACH_FRAME );

  myViewer->viewAll();
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();

  delete buttonAuditor;
  delete myViewer;
  myTop->unref();
  myGroup->unref();


  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();

  return 0;
}


