/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to use the asynchronous getData API from LDM
author : Thomas Arcila
September 2007
----------------------------------------------------------------------------------------*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>

#include <LDM/manips/SoROIManip.h>
#include <LDM/SoLDMResourceManager.h>

#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoVertexProperty.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoRotation.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/sensors/SoNodeSensor.h>
#include <Inventor/sensors/SoTimerSensor.h>


#define DISPLAY_DATASET 1

enum MenuPosition {
  MENU_BASE_INSTRUCTIONS,
  MENU_ASYNCHRONOUS_LOAD,
  MENU_SYNCHRONOUS_LOAD,
  MENU_BUILDING_DATASET
};

bool  toggleViewData = true;
bool  updateData = false;
int   resolution = 0;

SoXtExaminerViewer* viewer = 0;
SoVolumeData* readDataSet = 0;
SoVolumeData* displayedDataSet = 0;
SoSwitch* activeDataSet = 0;
SoRotation* rotation = 0;
SoSwitch* menuSwitch = 0;

bool doSynchronousLoading = false;
bool doAsynchronousLoading = false;
bool doDisplayDataSet = false;

int requestId = 0;
bool requestDone = false;
SoRef<SoCpuBufferObject> cpuBuffer = NULL;

class MyAsynchronousLDMDataAccess : public SoLDMDataAccess
{
  virtual void endRequest(int /*requestId*/)
  {
    requestDone = true;
  }
};

MyAsynchronousLDMDataAccess *asyncDataAccess;


void
resetDataSet()
{
  if ( readDataSet )
  {
    readDataSet->unref();
    readDataSet = 0;
  }

  // Dataset used by LDMDataAccess
  if ( ! readDataSet )
  {
    readDataSet = new SoVolumeData();
    readDataSet->ref();
  }

  readDataSet->ldmResourceParameters.getValue()->loadPolicy = SoVolumeData::SoLDMResourceParameters::NEVER;
  readDataSet->fileName = "$OIVHOME/examples/data/VolumeViz/3DHEAD.ldm";

  if (displayedDataSet)
  {
    static unsigned char onVoxel[8];
    memset(onVoxel, 0, 8*sizeof(unsigned char));
    displayedDataSet->data.setValue( SbVec3i32(2,2,2), SbDataType::UNSIGNED_BYTE, 0, (void*)&onVoxel, SoSFArray::NO_COPY );
  }
}


void
keyboardCB(void *, SoEventCallback *eventCB)
{
  const SoEvent *event = eventCB->getEvent();

  if ( doSynchronousLoading || doAsynchronousLoading )
    return;

  // do an asynchronous loading
  if ( SO_KEY_PRESS_EVENT(event, A) )
 { 
    menuSwitch->whichChild = MENU_ASYNCHRONOUS_LOAD;
    doAsynchronousLoading = true;
  }

  // do a synchronous loading
  else if ( SO_KEY_PRESS_EVENT(event, S) )
  {
    menuSwitch->whichChild = MENU_SYNCHRONOUS_LOAD;
    doSynchronousLoading = true;
  }

  else if ( SO_KEY_PRESS_EVENT(event, R) )
  {
    resetDataSet();
  }
}

void
timerCallback(void* /*data*/, SoSensor* /*sensor*/)
{
  if ( !readDataSet ) // source dataset has not been created
    return;

  asyncDataAccess->setDataSet(readDataSet);

  SbRotation currentRotation = rotation->rotation.getValue();
  currentRotation = SbRotation(SbVec3f(0.0f, 0.0f, 1.0f), (float)(M_PI/90.0f)) * currentRotation;
  rotation->rotation.setValue(currentRotation);

  if ( !doSynchronousLoading && !doAsynchronousLoading && !doDisplayDataSet )
    return;

  // get data informations
  SbBox3i32 box(SbVec3i32(0, 0, 0), readDataSet->data.getSize() - SbVec3i32(1, 1, 1));
  SoVolumeData::LDMDataAccess::DataInfoBox info;
  info = readDataSet->getLdmDataAccess().getData(0, box, (SoBufferObject*)(NULL));

  if ( doSynchronousLoading )
  {
    cpuBuffer = new SoCpuBufferObject();
    cpuBuffer->setSize((size_t)info.bufferSize);
    SoLDMDataAccess& dataAccess(readDataSet->getLdmDataAccess());

    SoLDMDataAccess::DataInfoBox info(dataAccess.getData(0, box, cpuBuffer.ptr()));
    if (info.errorFlag == SoVolumeData::LDMDataAccess::CORRECT) {
#if DISPLAY_DATASET
      menuSwitch->whichChild = MENU_BUILDING_DATASET;
      doDisplayDataSet = true;
#endif
    }
    else
      SoError::post("Error while synchronous request");

    doSynchronousLoading = false;
  }
  else if ( doAsynchronousLoading )
  {
    if ( requestId == 0 )
    { // start request
      cpuBuffer= new SoCpuBufferObject();
      cpuBuffer->setSize((size_t)info.bufferSize);

      requestId = asyncDataAccess->requestData(0, box, cpuBuffer.ptr());

      if ( requestId < 0 )
      { // request was synchronous (already in memory)
        SoLDMDataAccess::DataInfoBox info;
        asyncDataAccess->getRequestedData(-requestId, info);
        if (info.errorFlag == SoVolumeData::LDMDataAccess::CORRECT) {
#if DISPLAY_DATASET
        menuSwitch->whichChild = MENU_BUILDING_DATASET;
        doDisplayDataSet = true;
#endif
        }
        else
          SoError::post("Error while asynchronous request");
        requestId = 0;
        doAsynchronousLoading = false;
        requestDone = false;
      }
    }
    else
    { // already started request
      if ( requestDone )
      {
        SoLDMDataAccess::DataInfoBox info;
        asyncDataAccess->getRequestedData(requestId, info);
        if (info.errorFlag == SoVolumeData::LDMDataAccess::CORRECT) {
#if DISPLAY_DATASET
          menuSwitch->whichChild = MENU_BUILDING_DATASET;
          doDisplayDataSet = true;
#endif
        }
        else
          SoError::post("Error while asynchronous request");

        requestId = 0;
        requestDone = false;

        doAsynchronousLoading = false;
      }
    }
  }
#if DISPLAY_DATASET
  else if ( doDisplayDataSet )
  {
    SoVolumeData::DataType dataType = readDataSet->getDataType();
    SoMemoryObject *memObj = new SoMemoryObject(cpuBuffer.ptr(),SbDataType(dataType), false);
    displayedDataSet->data.setValue(info.bufferDimension, 0, memObj, SoSFArray::NO_COPY );
    displayedDataSet->extent.setValue( SbBox3f(SbVec3f(-.5f, -.5f, -.5f), SbVec3f(.5f, .5f, .5f) ) );
    activeDataSet->whichChild = SO_SWITCH_ALL;
    menuSwitch->whichChild = MENU_BASE_INSTRUCTIONS;
    doDisplayDataSet = false;
  }
#endif

}

SoSeparator*
buildScene()
{
  SoSeparator* scene = new SoSeparator;
  scene->ref();

  // Track the keyboard events
  SoEventCallback *eventCB = new SoEventCallback;
  eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), keyboardCB, NULL );
  scene->addChild(eventCB);


  rotation = new SoRotation;

  SoTimerSensor *schedulingSensor = new SoTimerSensor(timerCallback, rotation);
  schedulingSensor->setInterval(.02f);
  schedulingSensor->schedule();

  scene->addChild(rotation);


  // bounding box
  SoSeparator* cubeSeparator = new SoSeparator;
  SoDrawStyle* cubeStyle = new SoDrawStyle;
  cubeStyle->style = SoDrawStyle::LINES;
  cubeSeparator->addChild(cubeStyle);
  SoLightModel* cubeLightModel = new SoLightModel;
  cubeLightModel->model = SoLightModel::BASE_COLOR;
  scene->addChild(cubeLightModel);
  SoCube* cube = new SoCube;
  cube->width = cube->height = cube->depth = 1.f;
  cubeSeparator->addChild(cube);
  scene->addChild(cubeSeparator);

  // dataAccess build dataset
  activeDataSet = new SoSwitch;
  activeDataSet->whichChild = SO_SWITCH_NONE;

  displayedDataSet = new SoVolumeData;
  displayedDataSet ->ldmResourceParameters.getValue()->loadPolicy = SoVolumeData::SoLDMResourceParameters::ALWAYS;
  activeDataSet->addChild(displayedDataSet);

  scene->addChild(activeDataSet);

  // Use a predefined colorMap with the SoTransferFunction
  SoTransferFunction *pTransFunc = new SoTransferFunction;
  pTransFunc->predefColorMap = SoTransferFunction::SEISMIC;
  pTransFunc->minValue = 15;
  pTransFunc->maxValue = 255;
  scene->addChild(pTransFunc);

  SoVolumeRender* volumeRender = new SoVolumeRender;
  volumeRender->samplingAlignment = SoVolumeRender::DATA_ALIGNED;
  scene->addChild(volumeRender);
  scene->unrefNoDelete();

  return scene;
}

SoSeparator*
buildTextMenu()
{
  SoSeparator* menuSep = new SoSeparator;
  menuSep->ref();

  SoFont *font = new SoFont;
  font->name = "Courier New";
  font->size = 12;
  menuSep->addChild(font);

  SoBaseColor *menuColor = new SoBaseColor;
  menuColor->rgb.setValue( 1, 1, 0);
  menuSep->addChild( menuColor );

  SoTranslation *menuPos = new SoTranslation;
  menuPos->translation.setValue( -0.95f, 0.95f, 0. );
  menuSep->addChild( menuPos );

  menuSwitch = new SoSwitch;

  SoText2 *menu = new SoText2;
  menu->string.set1Value( 0, "While the volume bounding box is rotating press:" );
  menu->string.set1Value( 1, " 'a' to asynchronously load volume data and then display it" );
  menu->string.set1Value( 2, " 's' to do synchronous load of volume data before displaying it" );
  menu->string.set1Value( 3, " 'r' to trash all the previously loaded data and force reload" );
  menuSwitch->addChild( menu );

  menu = new SoText2;
  menu->string.set1Value( 0, "Data loading asynchronously...");
  menuSwitch->addChild( menu );

  menu = new SoText2;
  menu->string.set1Value( 0, "Data loading synchronously...");
  menuSwitch->addChild( menu );

  menu = new SoText2;
  menu->string.set1Value( 0, "Data loaded, building volume data...");
  menuSwitch->addChild( menu );

  menuSwitch->whichChild = MENU_BASE_INSTRUCTIONS;

  menuSep->addChild(menuSwitch);
  menuSep->unrefNoDelete();

  return menuSep;
}

int
main (int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if ( !myWindow )
    return 0;

  // Initialize of VolumeViz extension
  SoVolumeRendering::init();
  asyncDataAccess = new MyAsynchronousLDMDataAccess();

  // Assemble the scene graph
  SoSeparator *root = new SoSeparator;
  root->ref();
  root->addChild(buildTextMenu());
  SoPerspectiveCamera* camera = new SoPerspectiveCamera;
  root->addChild(new SoPerspectiveCamera);

  SoSeparator* scene = buildScene();

  root->addChild(scene);

  viewer = new SoXtExaminerViewer(myWindow);
  viewer->setSceneGraph(root);
  viewer->setTitle("Volume rendering");
  viewer->show();

  camera->viewAll(scene, viewer->getViewportRegion());
  viewer->viewAll();
  viewer->saveHomePosition();

  resetDataSet();

  //
  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete viewer;
  root->unref();
  SoVolumeRendering::finish();
  SoXt::finish();

  return 0;
}


