/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to use the getDAta API from VolumeViz
author : Jerome Hummel
January 2004
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <LDM/manips/SoROIManip.h>
#include <LDM/SoLDMResourceManager.h>
#include <LDM/SoLDMTopoOctree.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoVertexProperty.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/sensors/SoNodeSensor.h>
#include <Inventor/devices/SoCpuBufferObject.h>

bool  toggleViewData = true;
bool  updateData     = false;
int   resolution     = 0;

SoXtExaminerViewer* myViewer2 = NULL;
SoSeparator*             root = NULL;
SoSeparator*            root2 = NULL;
SoSeparator*           boxSep = NULL;
SoVolumeData*        pVolData = NULL;
SoROIManip*         pROIManip = NULL;
SoVolumeData*         volData = NULL;
SoSwitch      *switchViewData = NULL;
SoText2*                menu  = NULL;

////////////////////////////////////////////////////////////////////////
//
void
EventCB (void *, SoEventCallback *eventCB)
{
  const SoEvent *event = eventCB->getEvent();

  // Toggle data display
  if (SO_KEY_PRESS_EVENT(event, D))
  {
    toggleViewData = !toggleViewData;
    updateData = true;
  }

  // Update data
  else if (SO_KEY_PRESS_EVENT(event, U))
  {
    updateData = true;
  }

  // increase resolution
  else if (SO_KEY_PRESS_EVENT(event, UP_ARROW))
  {
    if (resolution)
      resolution--;
    updateData = true;
  }

  // Change Data Access Mode
  else if (SO_KEY_PRESS_EVENT(event, M))
  {
    int mode = pVolData->getLdmDataAccess().getGetDataMode();
    mode = (mode + 1)%3;
    pVolData->getLdmDataAccess().setGetDataMode(SoLDMDataAccess::GetDataMode(mode));
    updateData = true;
  }

  // decrease resolution
  else if (SO_KEY_PRESS_EVENT(event, DOWN_ARROW))
  {
    if (resolution < pVolData->getLDMTopoOctree()->getLevelMax())
      resolution++;
    updateData = true;
  }

  else {
    return;
  }

  pROIManip->touch(); // to force viewer2 to be updated
}

////////////////////////////////////////////////////////////////////////
//
void
updateView2( void* /*userData*/ )
{
  //do not display while moving
//  if (switchViewData)
//    switchViewData->whichChild = SO_SWITCH_NONE;

  // pVolData is the volume in the main window
  // Get dimensions of volume, volume size and extent of ROI
  SbBox3f   volSize = pVolData->extent.getValue();
  SbBox3i32 box(pROIManip->subVolume.getValue().getMin(), pROIManip->subVolume.getValue().getMax());

  // Call with NULL buffer (default 3rd parameter) to get number of bytes required to hold data
  SoVolumeData::LDMDataAccess::DataInfoBox info;

  // First call to getData with no buffer to retrieve needed information
  info = pVolData->getLdmDataAccess().getData( resolution, box );

  bool firstTime = (boxSep == NULL);
  if (firstTime)
  {
    root = new SoSeparator;
    root->ref();

    SoEventCallback *eventCB = new SoEventCallback;
    eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), EventCB, NULL );
    root->addChild( eventCB );

    SoLightModel *lightModel = new SoLightModel;
    lightModel->model = SoLightModel::BASE_COLOR;
    root->addChild( lightModel );

    { // menu
      SoSeparator* menuRoot = new SoSeparator;
      SoFont *font = new SoFont;
      font->name = "Courier New";
      font->size = 12;
      menuRoot->addChild( font);

      SoBaseColor *menuColor = new SoBaseColor;
      menuColor->rgb.setValue( 1, 1, 0);
      menuRoot->addChild( menuColor );

      SoTranslation *menuPos = new SoTranslation;
      menuPos->translation.setValue( -0.95f, 0.95f, 0. );
      menuRoot->addChild( menuPos );

      menu = new SoText2;
      menu->setName("menu");
      menu->string.set1Value( 0, "Adjust ROI in main window, then" );
      menu->string.set1Value( 1, "Press 'U' to display that data here." );
      menu->string.set1Value( 2, "U       : Update data" );
      menu->string.set1Value( 3, "UP      : Increase resolution" );
      menu->string.set1Value( 4, "DOWN    : Decrease resolution" );
      menu->string.set1Value( 5, "M       : Toggle DataAccess mode" );


      menuRoot->addChild( menu );
      root->addChild(menuRoot);
    }

    root2 = new SoSeparator;
    root2->setName( "root2" );
    root->addChild( root2 );
    root2->addChild( new SoPerspectiveCamera );

    boxSep = new SoSeparator;
    boxSep->setName( "boxsep" );
    root2->addChild( boxSep );

    myViewer2->setSceneGraph( root );
  }

  boxSep->removeAllChildren();

  switch (pVolData->getLdmDataAccess().getGetDataMode() )
  {
  case SoLDMDataAccess::CACHE:
    menu->string.set1Value( 5, "M       : Toggle DataAccess mode (CACHE)" );
    pVolData->ldmResourceParameters.getValue()->maxMainMemory.setValue(20);
    break;
  case SoLDMDataAccess::DIRECT:
    menu->string.set1Value( 5, "M       : Toggle DataAccess mode (DIRECT)" );
    pVolData->ldmResourceParameters.getValue()->maxMainMemory.setValue(0);
    break;
  case SoLDMDataAccess::DIRECT_AND_PREFETCH:
    pVolData->ldmResourceParameters.getValue()->maxMainMemory.setValue(0);
    menu->string.set1Value( 5, "M       : Toggle DataAccess mode (DIRECT_AND_PREFETCH" );
    break;
  }

  SoDrawStyle *lineStyle = new SoDrawStyle;
  lineStyle->style = SoDrawStyle::LINES;
  boxSep->addChild( lineStyle );

  float volXsize, volYsize, volZsize;
  volSize.getSize( volXsize, volYsize, volZsize );
  SbVec3f volCenter = volSize.getCenter();

  // Box representing the entire volume
  SoSeparator *volSep = new SoSeparator;
  boxSep->addChild( volSep );
  {
    SoTranslation *volTr = new SoTranslation;
    volTr->translation = volCenter;
    volSep->addChild( volTr );

    SoBaseColor *volColor = new SoBaseColor;
    volColor->rgb.setValue( 0.9f, 0.9f, 0.9f );
    volSep->addChild( volColor );

    SoCube *vol = new SoCube;
    vol->width  = volXsize;
    vol->height = volYsize;
    vol->depth  = volZsize;
    volSep->addChild( vol );
  }

  // Box representing the current ROI
  // Note: ROI is in voxel coords and we need XYZ coords
  SbVec3i32 roiMin = box.getMin();
  SbVec3i32 roiMax = box.getMax();
  SbVec3f roiMinf = pVolData->getLdmDataAccess().voxelToXYZ( roiMin );
  SbVec3f roiMaxf = pVolData->getLdmDataAccess().voxelToXYZ( roiMax );
  SoSeparator *roiSep = new SoSeparator;
  boxSep->addChild( roiSep );
  {
    SbVec3f roiDimf = roiMaxf - roiMinf;

    SoTranslation *volTr = new SoTranslation;
    volTr->translation = roiMinf + (roiDimf / 2);
    roiSep->addChild( volTr );

    SoBaseColor *roiColor = new SoBaseColor;
    roiColor->rgb.setValue( 1, 0, 0 );
    roiSep->addChild( roiColor );

    SoCube *roi = new SoCube;
    roi->width  = roiDimf[0];
    roi->height = roiDimf[1];
    roi->depth  = roiDimf[2];
    roiSep->addChild( roi );
  }

  if (updateData) 
  {
    updateData = false;

    // Create volume data, colormap and render nodes if necessary
    if (!volData)
    {
      switchViewData = new SoSwitch;
      switchViewData->whichChild = SO_SWITCH_ALL;
      root2->addChild( switchViewData );
      {
        SoDrawStyle *volStyle = new SoDrawStyle;
        volStyle->style = SoDrawStyle::FILLED;
        switchViewData->addChild( volStyle );

        volData = new SoVolumeData;
        volData->ldmResourceParameters.getValue()->loadPolicy = SoVolumeData::SoLDMResourceParameters::ALWAYS;
        switchViewData->addChild( volData );

        SoTransferFunction* transFunc = new SoTransferFunction;
        transFunc->predefColorMap = SoTransferFunction::GLOW;
        transFunc->minValue = 20;
		    transFunc->maxValue = 255;
        switchViewData->addChild( transFunc );
        
        SoVolumeRender* volRend = new SoVolumeRender;
        switchViewData->addChild( volRend );
      }
    }

    // Fetch the data contained in the ROI in the original volume
    SoMemoryObject* memObj;

    if (info.bufferSize) 
    {
      {
        SoRef<SoCpuBufferObject> cpuBufferObject = new SoCpuBufferObject;
        
        cpuBufferObject->setSize((size_t)info.bufferSize);
        info = pVolData->getLdmDataAccess().getData( resolution, box, cpuBufferObject.ptr() );
        SoVolumeData::DataType dataType = pVolData->getDataType();
        memObj = new SoMemoryObject(cpuBufferObject.ptr(), (SbDataType)dataType);
      }

      if (info.errorFlag == SoVolumeData::LDMDataAccess::CORRECT) 
      {
        volData->data.setValue( info.bufferDimension, 0, memObj, SoSFArray::NO_COPY );
      }

      // Set the volume bounding box to the geometric extent of the ROI
      volData->extent.setValue( SbBox3f(roiMinf,roiMaxf) );

      if (switchViewData)
        switchViewData->whichChild = SO_SWITCH_ALL;

      myViewer2->scheduleRedraw();
    }
  }

  if (firstTime)
  {
    myViewer2->setCameraSceneGraph( root2 );
    myViewer2->viewAll();
    myViewer2->saveHomePosition();
    firstTime = false;
  }
}

////////////////////////////////////////////////////////////////////////
//
// main function
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  // Initialize of VolumeViz extension
  SoVolumeRendering::init();  

  // Node to hold the volume data
  pVolData = new SoVolumeData();
  pVolData->fileName = "$OIVHOME/examples/data/VolumeViz/3DHEAD.ldm";
  
  //getting data is faster in ALWAYS mode
  pVolData->ldmResourceParameters.getValue()->loadPolicy = SoVolumeData::SoLDMResourceParameters::ALWAYS;

  // Track the keyboard events
  SoEventCallback *eventCB = new SoEventCallback;
  eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), EventCB, NULL );

  // Use a predefined colorMap with the SoTransferFunction
  SoTransferFunction *pTransFunc = new SoTransferFunction;
  pTransFunc->predefColorMap = SoTransferFunction::SEISMIC;
  pTransFunc->minValue = 15;
  pTransFunc->maxValue = 255;

  // ROI / ROIManip
  // Initialize both ROI box and subvolume to be the entire volume.
  // Constrain the ROIManip to stay inside the volume.
   SbVec3i32 dimensions = pVolData->data.getSize();

   //pVolData->getVolumeData(dimensions, datatemp, type);
  pVolData->ldmResourceParameters.getValue()->loadPolicy = SoVolumeData::SoLDMResourceParameters::ALWAYS;
  pROIManip = new SoROIManip();
  pROIManip->box.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  pROIManip->subVolume.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  pROIManip->constrained = TRUE;
  pROIManip->boxOn = FALSE;

  // The sensor must be attached before setting the scene graph, in order
  // for the interface to be initialized.
  // It's a behavior change since 9.6.1, where the interface initialization
  // callback was scheduled during the first frame render.
  SoNodeSensor *roiSensor = new SoNodeSensor((SoSensorCB *)&updateView2, NULL);
  roiSensor->attach(pROIManip);

  // Node in charge of drawing the volume
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->samplingAlignment = SoVolumeRender::VIEW_ALIGNED;

  // Assemble the scene graph
  // Note: SoVolumeRender must appear after the SoVolumeData node.
  SoSeparator *root = new SoSeparator;
  root->ref();
  root->addChild( eventCB );
  root->addChild( pVolData );
  root->addChild( pROIManip );
  root->addChild( pTransFunc );
  root->addChild( pVolRender );

  // Set up viewer 1:
  SoXtExaminerViewer *myViewer1 = new SoXtExaminerViewer(myWindow);
  myViewer1->setSceneGraph(root);
  myViewer1->setTitle("Volume rendering");
  myViewer1->show();

  // Set up viewer 2:
  myViewer2 = new SoXtExaminerViewer(myWindow, "", FALSE);
  myViewer2->setTitle("GetData( box )");
  myViewer2->setBackgroundColor( SbColor(0.4f,0.4f,0.4f) );
  myViewer2->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();

  SoNode* viewer2Scene = NULL;
  if (myViewer2->getSceneGraph())
    viewer2Scene = myViewer2->getSceneGraph();
  
  delete myViewer1;
  delete myViewer2;
  
  root->unref();
  if (viewer2Scene)
    viewer2Scene->unref();

  SoVolumeRendering::finish();
  SoXt::finish();

  return 0;
}


