/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to use LDMDataAccess API in multi threaded application
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <LDM/manips/SoROIManip.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoText2.h>

#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoBaseColor.h>

#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoEventCallback.h>

// CpuBufferObject header
#include <Inventor/devices/SoBufferObject.h>
#include <Inventor/devices/SoCpuBufferObject.h>

// Some header for thread synchronisation 
#include <Inventor/threads/SbThreadBarrier.h> 
#include <Inventor/threads/SbThreadSemaphore.h> 
#include <Inventor/sensors/SoTimerSensor.h> 

// Define the number of thread to launch for the demo
#define NB_THREADS 2

bool  updateData     = false;
int   resolution     = 0;
SbThread *threadHandle[NB_THREADS];

SoXtExaminerViewer*      myViewer2 = NULL;
SoSeparator*                boxSep = NULL;
SoVolumeData*             pVolData = NULL;
SoROIManip*              pROIManip = NULL;
SoVolumeData*              volData = NULL;
SoSwitch*           switchViewData = NULL;
SoSeparator*                 root2 = NULL;
SoCpuBufferObject *cpuBufferObject = NULL;
SoText2*                     menu  = NULL;

SoVolumeData::LDMDataAccess::DataInfoBox info;
SbVec3f roiMinf;
SbVec3f roiMaxf;

SbThreadBarrier *m_threadBarrier   = NULL;
SbThreadSemaphore *m_semStatus     = NULL;
bool threadShouldRun               = TRUE;
static unsigned int m_threadStatus = 0;
SoLDMDataAccess::GetDataMode getDataMode = SoLDMDataAccess::CACHE;

typedef struct s_userData
{
  int threadIndex;
  SbBox3i32 threadBox;
  int resolution;
  SoCpuBufferObject* destBuffer;
  int zPerThread;
} t_userData;

t_userData threadData[NB_THREADS];

////////////////////////////////////////////////////////////////////////
// Call when thread loaded the data
static void notifyMainThread(void* , SoSensor*)
{
  if ( m_threadStatus == (1 << NB_THREADS) -1 )
    // Wait for all thread to have finished before proceeding to rendering
  {
    SoVolumeData::DataType dataType = pVolData->getDataType();
    SoMemoryObject* memObj = new SoMemoryObject(cpuBufferObject, (SbDataType)dataType);

    volData->data.setValue( info.bufferDimension, 0, memObj, SoSFArray::NO_COPY );

    // Set the volume bounding box to the geometric extent of the ROI
    volData->extent.setValue( SbBox3f(roiMinf,roiMaxf) );

    if ( switchViewData )
      switchViewData->whichChild = SO_SWITCH_ALL;

    myViewer2->scheduleRedraw();

    m_semStatus->wait();
    {
      m_threadStatus = 0;
    }
    m_semStatus->release();
  }
}

////////////////////////////////////////////////////////////////////////
// Thread routine
void*
getDataThreadRoutine( void* userData )
{
  t_userData *data = (t_userData*)userData;
  SoVolumeData::LDMDataAccess::DataInfoBox info;
  SoRef<SoCpuBufferObject> threadCpuBuff = new SoCpuBufferObject();

  int threadId = data->threadIndex;

  while ( threadShouldRun )
  {
    // Wait for main thread to signal data are requested
    m_threadBarrier->enter();
    pVolData->getLdmDataAccess().setGetDataMode(getDataMode);

    // First call to getData with no buffer to retrieve needed information
    info = pVolData->getLdmDataAccess().getData( threadData[threadId].resolution, threadData[threadId].threadBox);
    threadCpuBuff->setSize((size_t)info.bufferSize);
  
    info = pVolData->getLdmDataAccess().getData( threadData[threadId].resolution, threadData[threadId].threadBox, threadCpuBuff.ptr() );
    int dstOffset = threadId * (threadData[threadId].zPerThread+1) * (threadData[threadId].threadBox.getMax()[0] - threadData[threadId].threadBox.getMin()[0]+1) * (threadData[threadId].threadBox.getMax()[1] - threadData[threadId].threadBox.getMin()[1]+1) ;
  
    if ( info.errorFlag == SoLDMDataAccess::CORRECT )
    {
        threadData[threadId].destBuffer->memcpy(threadCpuBuff.ptr(), dstOffset * pVolData->getDataSize(), 0, (size_t)info.bufferSize);

      // Take the semaphore before changing the thread status 
      m_semStatus->wait();
      {
        m_threadStatus |= (1 << threadId);
      }
      m_semStatus->release();
    }
  }

  return NULL;
}

////////////////////////////////////////////////////////////////////////
//
void
EventCB (void *, SoEventCallback *eventCB)
{
  const SoEvent *event = eventCB->getEvent();

  // Update data
  if (SO_KEY_PRESS_EVENT(event, U))
  {
    updateData = true;
  }
  // Change Data Access Mode
  else if (SO_KEY_PRESS_EVENT(event, M))
  {
    getDataMode = pVolData->getLdmDataAccess().getGetDataMode();
    getDataMode = (SoLDMDataAccess::GetDataMode)((getDataMode + 1)%3);
    pVolData->getLdmDataAccess().setGetDataMode(getDataMode);
  }
  else 
  {
    return;
  }

  pROIManip->touch(); // to force viewer2 to be updated
}

////////////////////////////////////////////////////////////////////////
//
void
updateView2( void* /*userData*/ )
{
  // pVolData is the volume in the main window
  // Get dimensions of volume, volume size and extent of ROI
  SbBox3f   volSize = pVolData->extent.getValue();
  SbBox3i32 box(pROIManip->subVolume.getValue().getMin(), pROIManip->subVolume.getValue().getMax());

  // First call to getData with no buffer to retrieve needed information
  info = pVolData->getLdmDataAccess().getData( resolution, box );

  bool firstTime = (boxSep == NULL);
  if (firstTime)
  {
    SoSeparator* root = new SoSeparator;
    root->ref();

    SoEventCallback *eventCB = new SoEventCallback;
    eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), EventCB, NULL );
    root->addChild( eventCB );

    SoLightModel *lightModel = new SoLightModel;
    lightModel->model = SoLightModel::BASE_COLOR;
    root->addChild( lightModel );

    { // menu
      SoSeparator* menuRoot = new SoSeparator;
      SoFont *font = new SoFont;
      font->name = "Courier New";
      font->size = 12;
      menuRoot->addChild( font);

      SoBaseColor *menuColor = new SoBaseColor;
      menuColor->rgb.setValue( 1, 1, 0);
      menuRoot->addChild( menuColor );

      SoTranslation *menuPos = new SoTranslation;
      menuPos->translation.setValue( -0.95f, 0.95f, 0. );
      menuRoot->addChild( menuPos );

      menu = new SoText2;
      menu->setName("menu");
      menu->string.set1Value( 0, "Adjust ROI in main window, then" );
      menu->string.set1Value( 1, "Press 'U' to display that data here." );
      menu->string.set1Value( 2, "M       : Toggle DataAccess mode" );

      menuRoot->addChild( menu );
      root->addChild(menuRoot);
    }

    root2 = new SoSeparator;
    root2->setName( "root2" );
    root->addChild( root2 );
    root2->addChild( new SoPerspectiveCamera );

    boxSep = new SoSeparator;
    boxSep->setName( "boxsep" );
    root2->addChild( boxSep );

    myViewer2->setSceneGraph( root );

    // If this is the first time, launch the needed threads to retrieve data
    for ( int i = 0; i < NB_THREADS; i++ )
    {
      threadData[i].threadIndex = i;
      threadHandle[i] = SbThread::create(getDataThreadRoutine, (void*)&threadData[i]);
    }
  }

  boxSep->removeAllChildren();

    switch (pVolData->getLdmDataAccess().getGetDataMode() )
  {
  case SoLDMDataAccess::CACHE:
    menu->string.set1Value( 2, "M       : Toggle DataAccess mode (CACHE)" );
    pVolData->ldmResourceParameters.getValue()->maxMainMemory = -1;
    break;
  case SoLDMDataAccess::DIRECT:
    menu->string.set1Value( 2, "M       : Toggle DataAccess mode (DIRECT)" );
    pVolData->ldmResourceParameters.getValue()->maxMainMemory= 1;
    break;
  case SoLDMDataAccess::DIRECT_AND_PREFETCH:
    pVolData->ldmResourceParameters.getValue()->maxMainMemory= 1;
    menu->string.set1Value( 2, "M       : Toggle DataAccess mode (DIRECT_AND_PREFETCH" );
    break;
  }

  SoDrawStyle *lineStyle = new SoDrawStyle;
  lineStyle->style = SoDrawStyle::LINES;
  boxSep->addChild( lineStyle );

  float volXsize, volYsize, volZsize;
  volSize.getSize( volXsize, volYsize, volZsize );
  SbVec3f volCenter = volSize.getCenter();

  // Box representing the entire volume
  SoSeparator *volSep = new SoSeparator;
  boxSep->addChild( volSep );
  {
    SoTranslation *volTr = new SoTranslation;
    volTr->translation = volCenter;
    volSep->addChild( volTr );

    SoBaseColor *volColor = new SoBaseColor;
    volColor->rgb.setValue( 0.9f, 0.9f, 0.9f );
    volSep->addChild( volColor );

    SoCube *vol = new SoCube;
    vol->width  = volXsize;
    vol->height = volYsize;
    vol->depth  = volZsize;
    volSep->addChild( vol );
  }

  // Box representing the current ROI
  // Note: ROI is in voxel coords and we need XYZ coords
  SbVec3i32 roiMin = box.getMin();
  SbVec3i32 roiMax = box.getMax();
  roiMinf = pVolData->getLdmDataAccess().voxelToXYZ( roiMin );
  roiMaxf = pVolData->getLdmDataAccess().voxelToXYZ( roiMax );
  SoSeparator *roiSep = new SoSeparator;
  boxSep->addChild( roiSep );
  {
    SbVec3f roiDimf = roiMaxf - roiMinf;

    SoTranslation *volTr = new SoTranslation;
    volTr->translation = roiMinf + (roiDimf / 2);
    roiSep->addChild( volTr );

    SoBaseColor *roiColor = new SoBaseColor;
    roiColor->rgb.setValue( 1, 0, 0 );
    roiSep->addChild( roiColor );

    SoCube *roi = new SoCube;
    roi->width  = roiDimf[0];
    roi->height = roiDimf[1];
    roi->depth  = roiDimf[2];
    roiSep->addChild( roi );
  }

  if ( updateData ) 
  {
    updateData = false;

    // Create volume data, colormap and render nodes if necessary
    if ( !volData )
    {
      switchViewData = new SoSwitch;
      switchViewData->whichChild = SO_SWITCH_ALL;
      root2->addChild( switchViewData );
      {
        SoDrawStyle *volStyle = new SoDrawStyle;
        volStyle->style = SoDrawStyle::FILLED;
        switchViewData->addChild( volStyle );

        volData = new SoVolumeData;
        volData->ldmResourceParameters.getValue()->loadPolicy = SoVolumeData::SoLDMResourceParameters::ALWAYS;
        switchViewData->addChild( volData );

        SoTransferFunction* transFunc = new SoTransferFunction;
        transFunc->predefColorMap = SoTransferFunction::SEISMIC;
        transFunc->minValue = 15;
		    transFunc->maxValue = 255;
        switchViewData->addChild( transFunc );

        switchViewData->addChild( new SoVolumeRender );
      }
    }

    if (info.bufferSize) 
    {
      cpuBufferObject = new SoCpuBufferObject;
      cpuBufferObject->setSize((size_t)info.bufferSize);

      // Clean the buffer to avoid dummy value
      void* tmpBuff = cpuBufferObject->map(SoBufferObject::SET);
      memset(tmpBuff, 0, (size_t)info.bufferSize);
      cpuBufferObject->unmap();

      // Set data for each thread....
      int xmin, ymin, zmin;
      int xmax, ymax, zmax;
      box.getBounds(xmin, ymin, zmin, xmax, ymax, zmax);

      int dimZ = zmax-zmin + 1;
      int zPerThread = (dimZ / NB_THREADS);

      SbBox3i32 reducedBox;
      int startingZ = zmin;

      // Compute Bbox for each thread
      for (int i = 0; i < NB_THREADS; i++)
      {
        threadData[i].resolution = resolution;
        threadData[i].destBuffer = cpuBufferObject;
        threadData[i].zPerThread = zPerThread;

        if ( i == NB_THREADS-1 )
          reducedBox = SbBox3i32(xmin, ymin, startingZ, xmax, ymax, zmax);  
        else
          reducedBox = SbBox3i32(xmin, ymin, startingZ, xmax, ymax, startingZ+zPerThread);

        threadData[i].threadBox = reducedBox;
        startingZ += zPerThread + 1;
      }

      //and ask them to start
      if ( !firstTime )
        m_threadBarrier->enter();
    }
  }

  if (firstTime)
  {
    myViewer2->setCameraSceneGraph( root2 );
    myViewer2->viewAll();
    myViewer2->saveHomePosition();
    firstTime = false;
  }
}

////////////////////////////////////////////////////////////////////////
//
// main function
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  // Initialize of VolumeViz extension
  SoVolumeRendering::init();  
  SoTimerSensor *checkSensor = new SoTimerSensor (notifyMainThread,  NULL);
  checkSensor ->setBaseTime(SbTime(0.0));
  checkSensor ->setInterval(SbTime(0.5));
  checkSensor ->schedule();

  m_threadBarrier  = new SbThreadBarrier(NB_THREADS + 1 );
  m_semStatus = new SbThreadSemaphore(1);

  // Node to hold the volume data
  pVolData = new SoVolumeData();
  pVolData->fileName = "$OIVHOME/examples/data/VolumeViz/3DHEAD.ldm";
  //pVolData->fileName = "$OIVHOME/examples/data/VolumeViz/colt-float.ldm";
  
  pVolData->ldmResourceParameters.getValue()->loadPolicy = SoLDMResourceParameters::ALWAYS;

  // Track the keyboard events
  SoEventCallback *eventCB = new SoEventCallback;
  eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), EventCB, NULL );
  
  // Use a predefined colorMap with the SoTransferFunction
  SoTransferFunction *pTransFunc = new SoTransferFunction;
  pTransFunc->predefColorMap = SoTransferFunction::SEISMIC;
  pTransFunc->minValue = 15;
  pTransFunc->maxValue = 255;

  // ROI / ROIManip
  // Initialize both ROI box and subvolume to be the entire volume.
  // Constrain the ROIManip to stay inside the volume.
   SbVec3i32 dimensions = pVolData->data.getSize();

  pROIManip = new SoROIManip();
  pROIManip->box.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  pROIManip->subVolume.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  pROIManip->constrained = TRUE;
  pROIManip->boxOn = FALSE;

  // The sensor must be attached before setting the scene graph, in order
  // for the interface to be initialized.
  // It's a behavior change since 9.6.1, where the interface initialization
  // callback was scheduled during the first frame render.
  SoNodeSensor *roiSensor = new SoNodeSensor((SoSensorCB *)&updateView2, NULL);
  roiSensor->attach(pROIManip);

  // Node in charge of drawing the volume
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->samplingAlignment = SoVolumeRender::VIEW_ALIGNED;

  // Assemble the scene graph
  // Note: SoVolumeRender must appear after the SoVolumeData node.
  SoSeparator *root = new SoSeparator;
  root->ref();
  root->addChild( eventCB );
  root->addChild( pVolData );
  root->addChild( pROIManip );
  root->addChild( pTransFunc );
  root->addChild( pVolRender );

  // Set up viewer 1:
  SoXtExaminerViewer *myViewer1 = new SoXtExaminerViewer(myWindow);
  myViewer1->setSceneGraph(root);
  myViewer1->setTitle("Volume rendering");
  myViewer1->show();

  // Set up viewer 2:
  myViewer2 = new SoXtExaminerViewer(myWindow, "", FALSE);
  myViewer2->setTitle("GetData( box )");
  myViewer2->setBackgroundColor( SbColor(0.4f,0.4f,0.4f) );
  myViewer2->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();

  // Kill running thread
  for ( int i = 0; i < NB_THREADS; i++ )
    SbThread::kill(threadHandle[i]);

  delete myViewer1;
  delete myViewer2;
  root->unref();
  SoVolumeRendering::finish();
  SoXt::finish();
  
  return 0;
}


