/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to use the getDAta API from VolumeViz
author : Jerome Hummel
January 2004
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <LDM/manips/SoROIManip.h>
#include <LDM/SoLDMResourceManager.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoVertexProperty.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/sensors/SoNodeSensor.h>
#include <Inventor/draggers/SoJackDragger.h>

bool  toggleViewData = true;
int   resolution = 0;
float amplitude = 1.0f;

SoXtExaminerViewer* myViewer2 = NULL;
SoSeparator*           root2  = NULL;
SoVolumeData*        pVolData = NULL;
SoROIManip*         pROIManip = NULL;
SoJackDragger*   planeDragger = NULL;
SoText2*                 menu = NULL;

void
EventCB (void *, SoEventCallback *eventCB)
{
  const SoEvent *event = eventCB->getEvent();

  // Toggle data display
  if (SO_KEY_PRESS_EVENT(event, D)) {
    toggleViewData = !toggleViewData;
  }

  // increase resolution
  else if (SO_KEY_PRESS_EVENT(event, UP_ARROW)) {
    if (resolution)
      resolution--;
  }

  // decrease resolution
  else if (SO_KEY_PRESS_EVENT(event, DOWN_ARROW)) {
    resolution++;
  }

  // increase amplitude
  else if (SO_KEY_PRESS_EVENT(event, PAGE_UP)) {
    amplitude *= 1.3f;
  }

  // decrease amplitude
  else if (SO_KEY_PRESS_EVENT(event, PAGE_DOWN)) {
    amplitude /= 1.3f;
  }

  // YZ plane
  else if (SO_KEY_PRESS_EVENT(event, X)) {
    planeDragger->rotation = SbRotation( SbVec3f(0,0,1), 1.570796325f );
  }

  // XZ plane
  else if (SO_KEY_PRESS_EVENT(event, Y)) {
    planeDragger->rotation = SbRotation( SbVec3f(0,1,0), 1.570796325f );
  }

  // XY plane
  else if (SO_KEY_PRESS_EVENT(event, Z)) {
    planeDragger->rotation = SbRotation( SbVec3f(1,0,0), 1.570796325f );
  }
 
  // Change Data Access Mode
  else if (SO_KEY_PRESS_EVENT(event, M))
  {
    int mode = pVolData->getLdmDataAccess().getGetDataMode();
    mode = (mode + 1)%3;
    pVolData->getLdmDataAccess().setGetDataMode(SoLDMDataAccess::GetDataMode(mode));
  }

  else {
    return;
  }

  pROIManip->touch(); // to force viewer2 to be updated
}


void
updateView2( void* /*userData*/ )
{
  //SbVec3i32 dim = pVolData->getDimension();
  SbVec3i32 dim = pVolData->data.getSize();

  SbLine line;
  { // get line
    SbVec3f linePosition = planeDragger->translation.getValue();
    // rotate the plane's normal by the dragger rotation
    SbRotation rotation = planeDragger->rotation.getValue();
    SbVec3f lineDirection;
    rotation.multVec( SbVec3f(0,1,0), lineDirection );
    SbLine lineXYZ( linePosition, linePosition + lineDirection );

    // line in data coordinates
    SbMatrix mx;
    mx.makeIdentity();
    SbVec3f volmin, volmax;
    pVolData->extent.getValue().getBounds( volmin, volmax );
    mx[0][0] = dim[0]/(volmax[0]-volmin[0]); mx[3][0] = -mx[0][0]*volmin[0];
    mx[1][1] = dim[1]/(volmax[1]-volmin[1]); mx[3][1] = -mx[1][1]*volmin[1];
    mx[2][2] = dim[2]/(volmax[2]-volmin[2]); mx[3][2] = -mx[2][2]*volmin[2];
    mx.multLineMatrix( lineXYZ, line );
  }

  SbBox3i32 box(pROIManip->subVolume.getValue().getMin(), pROIManip->subVolume.getValue().getMax());
  SoVolumeData::LDMDataAccess::DataInfoLine info;
  
  // First call to getData with no buffer to retrieve needed information
  info = pVolData->getLdmDataAccess().getData( resolution, box, line );

  bool firstTime = (root2 == NULL);
  if (firstTime) {

    SoSeparator *root = new SoSeparator;
    root->ref();

    SoEventCallback *eventCB = new SoEventCallback;
    eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), EventCB, NULL );
    root->addChild( eventCB );

    SoLightModel *lightModel = new SoLightModel;
    lightModel->model = SoLightModel::BASE_COLOR;
    root->addChild( lightModel );

    { // menu
      SoFont *font = new SoFont;
      font->name = "Courier New";
      font->size = 12;
      root->addChild( font);

      SoBaseColor *menuColor = new SoBaseColor;
      menuColor->rgb.setValue( 1, 1, 0);
      root->addChild( menuColor );

      SoTranslation *menuPos = new SoTranslation;
      menuPos->translation.setValue( -0.95f, 0.95f, 0. );
      root->addChild( menuPos );

      menu = new SoText2;
      menu->string.set1Value( 0, "D       : Toggle data" );
      menu->string.set1Value( 1, "UP      : Increase resolution" );
      menu->string.set1Value( 2, "DOWN    : Decrease resolution" );
      menu->string.set1Value( 3, "PAGEUP  : Increase amplitude" );
      menu->string.set1Value( 4, "PAGEDOWN: Decrease amplitude" );
      menu->string.set1Value( 5, "XYZ     : Axis alignment" );
      menu->string.set1Value( 6, "M       : Toggle DataAccess mode" );
      root->addChild( menu );
    }

    root->addChild( new SoPerspectiveCamera );

    root2 = new SoSeparator;
    root->addChild( root2 );

    myViewer2->setSceneGraph( root );
  }

  root2->removeAllChildren();

  switch (pVolData->getLdmDataAccess().getGetDataMode() )
  {
  case SoLDMDataAccess::CACHE:
    menu->string.set1Value( 6, "M       : Toggle DataAccess mode (CACHE)" );
    pVolData->ldmResourceParameters.getValue()->maxMainMemory.setValue(20);
    break;
  case SoLDMDataAccess::DIRECT:
    menu->string.set1Value( 6, "M       : Toggle DataAccess mode (DIRECT)" );
    pVolData->ldmResourceParameters.getValue()->maxMainMemory.setValue(0);
    break;
  case SoLDMDataAccess::DIRECT_AND_PREFETCH:
    pVolData->ldmResourceParameters.getValue()->maxMainMemory.setValue(0);
    menu->string.set1Value( 6, "M       : Toggle DataAccess mode (DIRECT_AND_PREFETCH" );
    break;
  }


  SoDrawStyle *lineStyle = new SoDrawStyle;
  lineStyle->style = SoDrawStyle::LINES;
  root2->addChild( lineStyle );

  SoSeparator *volSep = new SoSeparator;
  root2->addChild( volSep );
  {
    SoTranslation *volTr = new SoTranslation;
    volTr->translation = SbVec3f( (float)(dim[0]/2.), (float)(dim[1]/2.), (float)(dim[2]/2.) );
    volSep->addChild( volTr );

    SoBaseColor *volColor = new SoBaseColor;
    volColor->rgb.setValue( 0.9f, 0.9f, 0.9f );
    volSep->addChild( volColor );

    SoCube *vol = new SoCube;
    vol->width  = (float)dim[0];
    vol->height = (float)dim[1];
    vol->depth  = (float)dim[2];
    volSep->addChild( vol );
  }

  SoSeparator *roiSep = new SoSeparator;
  root2->addChild( roiSep );
  {
    SbVec3i32 roiMin = box.getMin();
    SbVec3i32 roiDim = box.getMax() - roiMin + SbVec3i32(1,1,1);

    SoTranslation *volTr = new SoTranslation;
    volTr->translation = SbVec3f( (float)(roiMin[0] + roiDim[0]/2.), (float)(roiMin[1] + roiDim[1]/2.), (float)(roiMin[2] + roiDim[2]/2.) );
    roiSep->addChild( volTr );

    SoBaseColor *roiColor = new SoBaseColor;
    roiColor->rgb.setValue( 1, 0, 0 );
    roiSep->addChild( roiColor );

    SoCube *roi = new SoCube;
    roi->width  = (float)roiDim[0];
    roi->height = (float)roiDim[1];
    roi->depth  = (float)roiDim[2];
    roiSep->addChild( roi );
  }

  if (info.errorFlag == SoVolumeData::LDMDataAccess::CORRECT) {

    SbVec3f vsegment[2];
    SbVec3f vline[2];
    for (int i = 0; i < 2; i++)
    {
      vsegment[i][0] = (float)info.lineCoord[i][0];
      vsegment[i][1] = (float)info.lineCoord[i][1];
      vsegment[i][2] = (float)info.lineCoord[i][2];
    }
    vline[0] = vsegment[0] - (vsegment[1] - vsegment[0]);
    vline[1] = vsegment[1] + (vsegment[1] - vsegment[0]);

    SoBaseColor *lineColor = new SoBaseColor;
    lineColor->rgb.setValue( 0.3f, 0.7f, 0.5f );
    root2->addChild( lineColor );

    SoDrawStyle *segmentWidthStyle = new SoDrawStyle;
    segmentWidthStyle->style = SoDrawStyle::LINES;
    segmentWidthStyle->lineWidth = 5.f;
    root2->addChild( segmentWidthStyle );

    SoLineSet *segment = new SoLineSet;
    SoVertexProperty *vp = new SoVertexProperty;
    vp->vertex.setValues( 0, 2, vsegment );
    segment->vertexProperty = vp;
    segment->numVertices.set1Value( 0, 2 );
    root2->addChild( segment );

    SoDrawStyle *lineWidthStyle = new SoDrawStyle;
    lineWidthStyle->style = SoDrawStyle::LINES;
    root2->addChild( lineWidthStyle );

    SoLineSet *line = new SoLineSet;
    vp = new SoVertexProperty;
    vp->vertex.setValues( 0, 2, vline );
    line->vertexProperty = vp;
    line->numVertices.set1Value( 0, 2 );
    root2->addChild( line );
  }

  if (toggleViewData && info.bufferSize > 0) 
  {
    int i;
    SoRef<SoCpuBufferObject> cpuBufferObject = new SoCpuBufferObject();
    cpuBufferObject->setSize((size_t)info.bufferSize);
    info = pVolData->getLdmDataAccess().getData( resolution, box, line, cpuBufferObject.ptr() );

    // the curve will be drawn onto the plane built against the line and a upvector
    SbVec3f vsegment[2];
    for (i = 0; i < 2; i++) 
    {
      vsegment[i][0] = (float)info.lineCoord[i][0];
      vsegment[i][1] = (float)info.lineCoord[i][1];
      vsegment[i][2] = (float)info.lineCoord[i][2];
    }
    SbVec3f dsegment = vsegment[1] - vsegment[0];

    SbVec3f axis = dsegment.getClosestAxis();
    SbVec3f upvector;
    if (axis[0] != 0)
    {
      if (dsegment[0] > 0)
        upvector.setValue( -dsegment[1], dsegment[0], 0 );
      else
        upvector.setValue( dsegment[1], -dsegment[0], 0 );
    }
    else if (axis[1] != 0)
    {
      if (dsegment[1] > 0)
        upvector.setValue( dsegment[1], -dsegment[0], 0 );
      else
        upvector.setValue( -dsegment[1], dsegment[0], 0 );
    }
    else
    {
      if (dsegment[2] > 0)
        upvector.setValue( 0, dsegment[2], -dsegment[1] );
      else
        upvector.setValue( 0, -dsegment[2], dsegment[1] );
    }
    upvector.normalize();

    SoLineSet *curve = new SoLineSet;
    SoVertexProperty *vp = new SoVertexProperty;

    int n = info.bufferDimension;
    if (cpuBufferObject->getSize()>0)
    {
      void *buffer = cpuBufferObject->map(SoCpuBufferObject::READ_ONLY);
      for (i = 0; i < n; i++)
      {
        double value = 0.;
        switch (pVolData->getDataType()) 
        {
        case SoVolumeData::UNSIGNED_BYTE:  value = (double)(((unsigned char  *)buffer)[i]) /       0xff; break;
        case SoVolumeData::UNSIGNED_SHORT: value = (double)(((unsigned short *)buffer)[i]) /     0xffff; break;
        case SoVolumeData::UNSIGNED_INT32: value = (double)(((unsigned int   *)buffer)[i]) / 0xffffffff; break;
        case SoVolumeData::SIGNED_BYTE:  value = (double)(((char  *)buffer)[i]) /       0xff + 0.5; break;
        case SoVolumeData::SIGNED_SHORT: value = (double)(((short *)buffer)[i]) /     0xffff + 0.5; break;
        case SoVolumeData::SIGNED_INT32: value = (double)(((int   *)buffer)[i]) / 0xffffffff + 0.5; break;
        case SoVolumeData::FLOAT: value = (double)(((float *)buffer)[i]) / 50000. + 0.5; break;
        default: break;
        }
        float u = (float) i / (n-1);
        float v = 5.f + amplitude/10.f * (float)value * 250.f;
        SbVec3f p = vsegment[0] +  u * dsegment + v * upvector;
        vp->vertex.set1Value( i, p );
      }
    cpuBufferObject->unmap();
    }
    curve->vertexProperty = vp;
    curve->numVertices.set1Value( 0, n );
    root2->addChild( curve );
  }

  if (firstTime)
  {
    myViewer2->viewAll();
    firstTime = false;
  }
}


//main function
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  // Initialize of VolumeViz extension
  SoVolumeRendering::init();

  // Node to hold the volume data
  pVolData = new SoVolumeData();
  pVolData->fileName = "$OIVHOME/examples/data/VolumeViz/3DHEAD.ldm";
  //getting data is faster in ALWAYS mode
  pVolData->ldmResourceParameters.getValue()->loadPolicy = SoVolumeData::SoLDMResourceParameters::ALWAYS;

  // Track the keyboard events
  SoEventCallback *eventCB = new SoEventCallback;
  eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), EventCB, NULL );

  // Plane manipulator
  planeDragger = new SoJackDragger;
  planeDragger->rotation = SbRotation( SbVec3f(0,0,1), 1.570796325f );

  planeDragger->addMotionCallback((SoDraggerCB *)&updateView2, NULL);

  // Use a predefined colorMap with the SoTransferFunction
  SoTransferFunction *pTransFunc = new SoTransferFunction;
  pTransFunc->predefColorMap = SoTransferFunction::SEISMIC;
  pTransFunc->minValue = 15;
  pTransFunc->maxValue = 255;

  // ROI / ROIManip
  // Initialize both ROI box and subvolume to be the entire volume.
  // Constrain the ROIManip to stay inside the volume.
  SbVec3i32 dimensions = pVolData->data.getSize();

  pROIManip = new SoROIManip();
  pROIManip->box.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  pROIManip->subVolume.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  pROIManip->constrained = TRUE;
  pROIManip->boxOn = FALSE;

  // The sensor must be attached before setting the scene graph, in order
  // for the interface to be initialized.
  // It's a behavior change since 9.6.1, where the interface initialization
  // callback was scheduled during the first frame render.
  SoNodeSensor *roiSensor = new SoNodeSensor((SoSensorCB *)&updateView2, NULL);
  roiSensor->attach(pROIManip);

  // Node in charge of drawing the volume
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->samplingAlignment = SoVolumeRender::VIEW_ALIGNED;

  // Assemble the scene graph
  // Note: SoVolumeRender must appear after the SoVolumeData node.
  SoSeparator *root = new SoSeparator;
  root->ref();
  root->addChild( eventCB );
  root->addChild( planeDragger );
  root->addChild( pVolData );
  root->addChild( pROIManip );
  root->addChild( pTransFunc );
  root->addChild( pVolRender );

  // Set up viewer 1:
  SoXtExaminerViewer *myViewer1 = new SoXtExaminerViewer(myWindow);
  myViewer1->setSceneGraph(root);
  myViewer1->setTitle("Volume rendering");
  myViewer1->show();

  // Set up viewer 2:
  myViewer2 = new SoXtExaminerViewer(myWindow, "", FALSE);
  myViewer2->setTitle("GetData( line )");
  myViewer2->setBackgroundColor( SbColor(0.4f,0.4f,0.4f) );
  myViewer2->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete myViewer1;
  delete myViewer2;
  root->unref();

  SoVolumeRendering::finish();
  SoXt::finish();

  return 0;
}


