/*==============================================================================
***   THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),    ***
***             AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                ***
***                                                                          ***
*** REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS  ***
*** SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR  ***
*** WRITTEN AUTHORIZATION OF FEI S.A.S.                                      ***
***                                                                          ***
***                       RESTRICTED RIGHTS LEGEND                           ***
*** USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS ***
*** WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN ***
*** SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT ***
*** CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN ***
*** TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.        ***
***                                                                          ***
***                  COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                   ***
***                           MERIGNAC, FRANCE                               ***
***                         ALL RIGHTS RESERVED                              ***
==============================================================================*/

#pragma once

#include <Inventor/Axis.h>
#include <Inventor/draggers/SoDragPointDragger.h>
#include <Inventor/draggers/SoJackDragger.h>
#include <Inventor/engines/SoCompose.h>
#include <Inventor/manips/SoJackManip.h>
#include <Inventor/nodes/SoBBox.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPhysicalMaterial.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoScale.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoFragmentShader.h>
#include <VolumeViz/draggers/SoOrthoSliceDragger.h>
#include <VolumeViz/nodes/SoHeightFieldGeometry.h>
#include <VolumeViz/nodes/SoHeightFieldRender.h>
#include <VolumeViz/nodes/SoMultiDataSeparator.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeShader.h>
#include <VolumeViz/nodes/SoVolumeSkin.h>

class SceneGraph
{
public:
  SceneGraph()
  {
    init();
  }

  SceneGraph(const char* volumeDataPath, const char* horizonGeomPath)
  : SceneGraph()
  {
    setVolumeData(volumeDataPath);
    setHorizonGeometry(horizonGeomPath);
    setSliceToCenter();
  }

  SoNode* getSceneGraph()
  {
    init();
    return m_sceneGraph.ptr();
  }

  void setSliceToCenter()
  {
    m_orthoSlice->sliceNumber = m_volumeData->getDimension()[m_orthoSlice->axis.getValue()] / 2;
  }

  SoOrthoSlice::Axis getSliceAxis() const
  {
    return static_cast<SoOrthoSlice::Axis>(m_orthoSlice->axis.getValue());
  }

  void setSliceAxis(SoOrthoSlice::Axis sliceAxis)
  {
    m_orthoSlice->axis = sliceAxis;
    setSliceToCenter();
  }

  SoOrthoSlice::ClippingSide getClippingSide() const
  {
    return static_cast<SoOrthoSlice::ClippingSide>(m_orthoSlice->clippingSide.getValue());
  }

  void setClippingSide(SoOrthoSlice::ClippingSide clippingSide)
  {
    m_orthoSlice->clippingSide = clippingSide;
  }

  float getSkinOpacity() const
  {
    return getOpacity(m_skinMaterial->transparency, 1.0f);
  }

  void setSkinOpacity(float opacityValue)
  {
    setOpacity(m_skinMaterial->transparency, opacityValue, 1.0f);
  }

  float getVolumeOpacity() const
  {
    return getOpacity(m_volumeMaterial->transparency, 16.0f);
  }

  void setVolumeOpacity(float opacityValue)
  {
    setOpacity(m_volumeMaterial->transparency, opacityValue, 16.0f);
  }

  void setVolumeData(const char* dataSetPath)
  {
    init();
    m_volumeData->fileName = (dataSetPath == nullptr) ? "" : dataSetPath;
    adjustExtent(m_volumeData);
  }

  SoVolumeData* getVolumeData() const
  {
    return m_volumeData;
  }

  void displayHorizon(bool value)
  {
    displaySwitch(m_horizonSwitch, value);
  }

  bool isHorizonDisplayed() const
  {
    return isSwitchDisplayed(m_horizonSwitch);
  }

  void displaySlice(bool value)
  {
    displaySwitch(m_sliceSwitch, value);
  }

  bool isSliceDisplayed() const
  {
    return isSwitchDisplayed(m_sliceSwitch);
  }

  void displayVolumeRender(bool value)
  {
    displaySwitch(m_volumeRenderSwitch, value);
  }

  bool isVolumeRenderDisplayed() const
  {
    return isSwitchDisplayed(m_volumeRenderSwitch);
  }

  void displayVolumeSkin(bool value)
  {
    displaySwitch(m_volumeSkinSwitch, value);
  }

  bool isVolumeSkinDisplayed() const
  {
    return isSwitchDisplayed(m_volumeSkinSwitch);
  }

  void setHorizonGeometry(const char* horizonGeomPath)
  {
    init();
    m_heightFieldGeom->fileName = (horizonGeomPath == nullptr) ? "" : horizonGeomPath;
    adjustExtent(m_heightFieldGeom);
  }

  SoHeightFieldGeometry* getHorizonGeometry() const
  {
    return m_heightFieldGeom;
  }

  void setHorizonAxis(openinventor::inventor::Axis::Type axis)
  {
    m_horizonAxis = axis;

    SoDragPointDragger* dragPtDragger = static_cast<SoJackDragger*>(m_horizonManip->getDragger())->getPart<SoDragPointDragger>("translator");

    switch (axis)
    {
    case openinventor::inventor::Axis::X:
      m_horizonAxisRotation->rotation = SbRotation(SbVec3f(0.0f, 1.0f, 0.0f), static_cast<float>(M_PI_2));
      dragPtDragger->showDraggerSet(SoDragPointDragger::X_AXIS);
      break;
    case openinventor::inventor::Axis::Y:
      m_horizonAxisRotation->rotation = SbRotation(SbVec3f(-1.0f, 0.0f, 0.0f), static_cast<float>(M_PI_2));
      dragPtDragger->showDraggerSet(SoDragPointDragger::Y_AXIS);
      break;
    default:
    case openinventor::inventor::Axis::Z:
      m_horizonAxisRotation->rotation = SbRotation();
      dragPtDragger->showDraggerSet(SoDragPointDragger::Z_AXIS);
      break;
    }
  }

  openinventor::inventor::Axis::Type getHorizonAxis() const
  {
    return m_horizonAxis;
  }

  SoTransform* getHorizonUserTransform() const
  {
    return m_horizonUserTransform;
  }

  // Normalize biggest dimension to 1 in order to avoid
  // big size differences between Volume Data and Horizon
  void adjustExtent(SoDataSet* dataSet)
  {
    SbBox3f extent = dataSet->extent.getValue();

    const bool isHeightField = dataSet->isOfType<SoHeightFieldGeometry>();
    if (isHeightField)
    {
      double dataMin = 0.0;
      double dataMax = 1.0;
      if (!SbDataType((SbDataType::DataType) dataSet->getDataType()).isInteger())
        dataSet->getMinMax(dataMin, dataMax);

      SbBox3f newBBox = m_horizonBBox->boundingBox.getValue();
      newBBox.getMin()[2] = dataMin;
      newBBox.getMax()[2] = dataMax;
      m_horizonBBox->boundingBox = newBBox;

      extent.getMin()[2] = static_cast<float>(dataMin);
      extent.getMax()[2] = static_cast<float>(dataMax);
    }
    SbVec3f scale = extent.getMax() - extent.getMin();
    float maxScale = scale[0];
    for (int i = 1; i < 3; i++)
    {
      if (scale[i] > maxScale)
        maxScale = scale[i];
    }

    const float factor = 2.0f / maxScale;
    scale *= factor;
    const SbVec3f newMin = -0.5f * scale;
    const SbVec3f newMax = newMin + scale;
    if (isHeightField)
    {
      m_horizonTransform->translation = SbVec3f(0.0f, 0.0f, -0.5f * factor * (extent.getMin()[2] + extent.getMax()[2]));
      m_horizonTransform->scaleFactor = SbVec3f(1.0f, 1.0f, factor);
    }
    dataSet->extent = SbBox3f(newMin, newMax);
  }

  void init()
  {
    if (m_sceneGraph.ptr() == nullptr)
      m_sceneGraph = buildSceneGraph();
  }

  void clear()
  {
    m_sceneGraph = nullptr;
  }

private:
  static void displaySwitch(SoSwitch* switchNode, bool value)
  {
    switchNode->whichChild = value ? SO_SWITCH_ALL : SO_SWITCH_NONE;
  }

  static bool isSwitchDisplayed(SoSwitch* switchNode)
  {
    return (switchNode->whichChild.getValue() == SO_SWITCH_ALL);
  }

  static float getOpacity(const SoMFFloat& transparencyField, float power)
  {
    return 1.0f - std::pow(transparencyField.getValues(0)[0], power);
  }

  static void setOpacity(SoMFFloat& transparencyField, float opacityValue, float power)
  {
    transparencyField = std::pow(1.0f - opacityValue, 1.0f / power);
  }

  SoNode* buildSceneGraph()
  {
    SoMultiDataSeparator* root = new SoMultiDataSeparator;

    m_volumeData = new SoVolumeData;
    m_volumeData->dataSetId = 1;

    SoTransferFunction* transferFunction = new SoTransferFunction;
    transferFunction->predefColorMap = SoTransferFunction::STANDARD;
    root->addChild(transferFunction);

    // Horizon
    {
      m_horizonSwitch = new SoSwitch;
      m_horizonSwitch->setName("HEIGHT_FIELD_RENDER");
      m_horizonSwitch->whichChild = SO_SWITCH_ALL;
      root->addChild(m_horizonSwitch);

      SoSeparator* horizonSep = new SoSeparator;
      m_horizonSwitch->addChild(horizonSep);

      // Volume Data
      horizonSep->addChild(m_volumeData);

      // Manip
      {
        m_horizonManip = new SoJackManip;

        // Remove scaler and rotator parts
        SoJackDragger* dragger = static_cast<SoJackDragger*>(m_horizonManip->getDragger());
        dragger->setPart("scaler", nullptr);
        dragger->setPart("rotator", nullptr);

        m_horizonManip->scaleFactor = SbVec3f(0.5f, 0.5f, 0.5f);
        horizonSep->addChild(m_horizonManip);
      }

      m_horizonAxisRotation = new SoRotation;
      horizonSep->addChild(m_horizonAxisRotation);

      m_horizonUserTransform = new SoTransform;
      horizonSep->addChild(m_horizonUserTransform);

      // Horizon Transform
      m_horizonTransform = new SoTransform;
      horizonSep->addChild(m_horizonTransform);

      // Fake BBox to make the manip smaller
      {
        m_horizonBBox = new SoBBox;
        m_horizonBBox->mode = SoBBox::USER_DEFINED;
        m_horizonBBox->boundingBox = SbBox3f(-0.5f, -0.5f, 0.0f, 0.5f, 0.5f, 1.0f);
        horizonSep->addChild(m_horizonBBox);
      }

      SoLightModel* horizonLightModel = new SoLightModel;
      horizonLightModel->model = SoLightModel::PHYSICALLY_BASED;
      horizonSep->addChild(horizonLightModel);

      SoPhysicalMaterial* horizonMaterial = new SoPhysicalMaterial;
      horizonMaterial->baseColor = SbColorRGBA(1.0f, 1.0f, 1.0f, 1.0f);
      horizonMaterial->specular = 1.0f;
      horizonMaterial->roughness = 0.3f;
      horizonSep->addChild(horizonMaterial);

      m_heightFieldGeom = new SoHeightFieldGeometry;
      m_heightFieldGeom->dataSetId = 2;
      horizonSep->addChild(m_heightFieldGeom);

      // Combine Shader
      {
        m_shaderSwitch = new SoSwitch;
        m_shaderSwitch->setName("COMBINE_SHADER");
        m_shaderSwitch->whichChild = SO_SWITCH_ALL;
        horizonSep->addChild(m_shaderSwitch);

        SoFragmentShader* combineShader = new SoFragmentShader;
        combineShader->sourceProgram = "$OIVHOME/examples/source/VolumeViz/horizonInVolume/combine_frag.glsl";

        SoShaderParameter1i* volDataParam = new SoShaderParameter1i;
        volDataParam->name = "volData";
        volDataParam->value.connectFrom(&m_volumeData->dataSetId);
        combineShader->parameter.addShaderParameter(volDataParam);

        SoShaderParameter1i* hfGeomParam = new SoShaderParameter1i;
        hfGeomParam->name = "hfGeom";
        hfGeomParam->value.connectFrom(&m_heightFieldGeom->dataSetId);
        combineShader->parameter.addShaderParameter(hfGeomParam);

        // Setup data combine shader in DATA_COMBINE_FUNCTION slot
        SoVolumeShader* volumeShader = new SoVolumeShader;
        volumeShader->shaderObject.set1Value(SoVolumeShader::DATA_COMBINE_FUNCTION, combineShader);
        m_shaderSwitch->addChild(volumeShader);
      }

      // Disable picking for performances
      {
        SoPickStyle* pickStyle = new SoPickStyle;
        pickStyle->style = SoPickStyle::UNPICKABLE;
        horizonSep->addChild(pickStyle);
      }

      // Finally, add the render node
      horizonSep->addChild(new SoHeightFieldRender);
    }

    // Volume Rendering
    {
      SoSeparator* volumeRenderingSep = new SoSeparator;
      root->addChild(volumeRenderingSep);

      volumeRenderingSep->addChild(m_volumeData);

      SoVolumeSkin* skin = new SoVolumeSkin;

      // Volume Box
      SoSwitch* skinBoxSwitch = new SoSwitch;
      skinBoxSwitch->setName("VOLUME_BOX");
      skinBoxSwitch->whichChild = SO_SWITCH_ALL;
      {
        SoSeparator* boxSep = new SoSeparator;

        SoPickStyle* boxPickStyle = new SoPickStyle;
        boxPickStyle->style = SoPickStyle::UNPICKABLE;
        boxSep->addChild(boxPickStyle);

        SoComplexity* boxComplexity = new SoComplexity;
        boxComplexity->type = SoComplexity::BOUNDING_BOX;
        boxSep->addChild(boxComplexity);

        SoLightModel* lightModel = new SoLightModel;
        lightModel->model = SoLightModel::BASE_COLOR;
        boxSep->addChild(lightModel);

        SoDrawStyle* drawStyle = new SoDrawStyle;
        drawStyle->style = SoDrawStyle::LINES;
        boxSep->addChild(drawStyle);

        SoMaterial* boxMaterial = new SoMaterial;
        boxMaterial->diffuseColor = SbColor(0.0f, 0.0f, 0.0f);
        boxSep->addChild(boxMaterial);

        boxSep->addChild(skin);

        skinBoxSwitch->addChild(boxSep);
      }
      volumeRenderingSep->addChild(skinBoxSwitch);

      // Slices
      m_sliceSwitch = new SoSwitch;
      m_sliceSwitch->setName("ORTHO_SLICE");
      m_sliceSwitch->whichChild = SO_SWITCH_ALL;
      {
        SoGroup* sliceAndDraggerGroup = new SoGroup;

        SoGroup* sliceGroup = new SoGroup;
        sliceAndDraggerGroup->addChild(sliceGroup);

        m_orthoSlice = new SoOrthoSlice;
        m_orthoSlice->axis = SoOrthoSlice::Y;
        m_orthoSlice->clipping = TRUE;
        m_orthoSlice->clippingSide = SoOrthoSlice::FRONT;
        sliceGroup->addChild(m_orthoSlice);

        SoPath* slicePath = new SoPath(sliceGroup);
        slicePath->append(m_orthoSlice);

        m_orthoSliceDragger = new SoOrthoSliceDragger;
        m_orthoSliceDragger->orthoSlicePath = slicePath;
        sliceAndDraggerGroup->addChild(m_orthoSliceDragger);

        m_sliceSwitch->addChild(sliceAndDraggerGroup);
      }
      volumeRenderingSep->addChild(m_sliceSwitch);

      // Volume Skin
      m_volumeSkinSwitch = new SoSwitch;
      m_volumeSkinSwitch->setName("VOLUME_SKIN");
      m_volumeSkinSwitch->whichChild = SO_SWITCH_ALL;
      {
        SoSeparator* skinSep = new SoSeparator;

        SoPickStyle* skinPickStyle = new SoPickStyle;
        skinPickStyle->style = SoPickStyle::UNPICKABLE;
        skinSep->addChild(skinPickStyle);

        m_skinMaterial = new SoMaterial;
        m_skinMaterial->transparency = 0.6f;
        skinSep->addChild(m_skinMaterial);

        skin->faceMode = SoVolumeSkin::FRONT_AND_BACK;
        skinSep->addChild(skin);

        m_volumeSkinSwitch->addChild(skinSep);
      }
      volumeRenderingSep->addChild(m_volumeSkinSwitch);

      // Volume Render
      m_volumeRenderSwitch = new SoSwitch;
      m_volumeRenderSwitch->setName("VOLUME_RENDER");
      m_volumeRenderSwitch->whichChild = SO_SWITCH_NONE;
      {
        SoSeparator* volumeRenderSep = new SoSeparator;

        SoPickStyle* volumeRenderPickStyle = new SoPickStyle;
        volumeRenderPickStyle->style = SoPickStyle::UNPICKABLE;
        volumeRenderSep->addChild(volumeRenderPickStyle);

        m_volumeMaterial = new SoMaterial;
        m_volumeMaterial->transparency = 0.985f;
        volumeRenderSep->addChild(m_volumeMaterial);

        SoVolumeRenderingQuality* vrq = new SoVolumeRenderingQuality;
        vrq->deferredLighting = FALSE;
        volumeRenderSep->addChild(vrq);

        SoVolumeRender* volumeRender = new SoVolumeRender;
        volumeRenderSep->addChild(volumeRender);

        m_volumeRenderSwitch->addChild(volumeRenderSep);
      }
      volumeRenderingSep->addChild(m_volumeRenderSwitch);
    }

    return root;
  }

  SoRef<SoNode> m_sceneGraph;
  SoVolumeData* m_volumeData;
  SoHeightFieldGeometry* m_heightFieldGeom;
  SoOrthoSliceDragger* m_orthoSliceDragger;
  SoOrthoSlice* m_orthoSlice;
  SoMaterial* m_skinMaterial;
  SoMaterial* m_volumeMaterial;
  SoSwitch* m_horizonSwitch;
  SoRotation* m_horizonAxisRotation;
  SoTransform* m_horizonTransform;
  SoBBox* m_horizonBBox;
  SoTransform* m_horizonUserTransform;
  SoJackManip* m_horizonManip;
  openinventor::inventor::Axis::Type m_horizonAxis;
  SoSwitch* m_sliceSwitch;
  SoSwitch* m_volumeRenderSwitch;
  SoSwitch* m_volumeSkinSwitch;
  SoSwitch* m_shaderSwitch;
};
