/*==============================================================================
***   THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),    ***
***             AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                ***
***                                                                          ***
*** REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS  ***
*** SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR  ***
*** WRITTEN AUTHORIZATION OF FEI S.A.S.                                      ***
***                                                                          ***
***                       RESTRICTED RIGHTS LEGEND                           ***
*** USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS ***
*** WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN ***
*** SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT ***
*** CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN ***
*** TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.        ***
***                                                                          ***
***                  COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                   ***
***                           MERIGNAC, FRANCE                               ***
***                         ALL RIGHTS RESERVED                              ***
==============================================================================*/

#include <QApplication>
#include <QGridLayout>
#include <QMainWindow>
#include <GUIWidget.h>
#include <Inventor/SoInteraction.h>
#include <Inventor/Qt/SbQtHelper.h>
#include <Inventor/nodes/SoCamera.h>
#include <Inventor/nodes/SoFile.h>
#include <Inventor/ViewerComponents/Qt/RenderAreaOrbiter.h>
#include <Inventor/ViewerComponents/nodes/SoViewingCube.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include "SceneGraph.h"

int
main(int argc, char** argv)
{
  SbQtHelper::addPlatformPluginsPath();

  // Create window
  QApplication app(argc, argv);
  QMainWindow* window = new QMainWindow(NULL);
  window->setWindowTitle("Horizon In Volume");

  SoInteraction::init();
  SoVolumeRendering::init();

  static const char* defaultVolumeData = "$OIVHOME/examples/data/VolumeViz/colt-float.ldm";
  static const char* defaultHorizonGeom = "$OIVHOME/examples/data/VolumeViz/horizons/horizon.ldm";
  SceneGraph scene(defaultVolumeData, defaultHorizonGeom);

  // Special parameters for these datasets
  scene.setHorizonAxis(openinventor::inventor::Axis::Y);
  scene.getHorizonUserTransform()->translation = SbVec3f(0.0f, 0.0f, 0.3f);
  scene.getHorizonUserTransform()->scaleFactor = SbVec3f(1.0f, 1.0f, 0.5f);
  scene.setClippingSide(SoOrthoSlice::BACK);
  scene.getVolumeData()->extent.setValue(SbVec3f(-1.0f, -1.0f, -1.0f), SbVec3f(1.0f, 1.0f, 1.0f));

  // Setup widgets layout
  QWidget* centralWidget = new QWidget(window);
  QVBoxLayout* centralLayout = new QVBoxLayout;
  centralLayout->setContentsMargins(0, 0, 0, 0);
  centralLayout->setSpacing(0);
  centralWidget->setLayout(centralLayout);

  // Create GUI widget and setup callbacks to connect GUI and scenegraph changes
  {
    GUIWidget* guiWidget = new GUIWidget(window);

    // Connect Transparencies
    guiWidget->setVolumeOpacity((double) scene.getSkinOpacity());
    scene.setVolumeOpacity(guiWidget->getVolumeOpacity());
    QObject::connect(guiWidget, &GUIWidget::volumeOpacityChanged, [guiWidget, &scene]() {
      const double opacity = guiWidget->getVolumeOpacity();
      scene.setVolumeOpacity(opacity);
      scene.setSkinOpacity(opacity);
    });

    // Connect On/Off CheckBoxes
    {
      guiWidget->displayHorizon(scene.isHorizonDisplayed());
      QObject::connect(guiWidget, &GUIWidget::horizonDisplayChanged, [guiWidget, &scene]() {
        scene.displayHorizon(guiWidget->isHorizonDisplayed());
      });

      guiWidget->displaySlice(scene.isSliceDisplayed());
      QObject::connect(guiWidget, &GUIWidget::sliceDisplayChanged, [guiWidget, &scene]() {
        scene.displaySlice(guiWidget->isSliceDisplayed());
      });

      guiWidget->displayVolumeRender(scene.isVolumeRenderDisplayed());
      guiWidget->displayVolumeSkin(scene.isVolumeSkinDisplayed());
      QObject::connect(guiWidget, &GUIWidget::volumeDisplayChanged, [guiWidget, &scene]() {
        scene.displayVolumeRender(guiWidget->isVolumeRenderDisplayed());
        scene.displayVolumeSkin(guiWidget->isVolumeSkinDisplayed());
      });
    }

    // Connect Slice Axis
    guiWidget->setSliceAxis(static_cast<int>(scene.getSliceAxis()));
    QObject::connect(guiWidget, &GUIWidget::sliceAxisChanged, [guiWidget, &scene]() {
      scene.setSliceAxis(static_cast<SoOrthoSlice::Axis>(guiWidget->getSliceAxis()));
    });

    // Connect Clipping Side
    guiWidget->setClippingSide(static_cast<int>(scene.getClippingSide()));
    QObject::connect(guiWidget, &GUIWidget::clippingSideChanged, [guiWidget, &scene]() {
      scene.setClippingSide(static_cast<SoOrthoSlice::ClippingSide>(guiWidget->getClippingSide()));
    });

    centralLayout->addWidget(guiWidget);
  }

  // Setup Render Area
  RenderAreaOrbiter* renderArea = new RenderAreaOrbiter(window);
  renderArea->setClearColor(SbColorRGBA(1.0f, 1.0f, 1.0f, 1.0f));
  renderArea->setAntialiasingMode(SoSceneManager::SMAA);
  renderArea->setAntialiasingQuality(1.0f);
  SceneOrbiter* orbiter = renderArea->getSceneInteractor();

  SoViewingCube* vCube = orbiter->getViewingCube();
  vCube->edgeStyle = SoViewingCube::CORNER;
  vCube->size = SbVec2f(200.0f, 200.0f);
  vCube->opacityMin = 0.0f;
  vCube->opacityMax = 0.8f;
  SoFile* compass = new SoFile;
  compass->name = "$OIVHOME/examples/data/Inventor/ViewingCube/Compass/compass-northsouth-y.iv";
  vCube->compass = compass;
  vCube->facePosX = "$OIVHOME/examples/data/Inventor/ViewingCube/Faces/NSEW/xpos.png";
  vCube->faceNegX = "$OIVHOME/examples/data/Inventor/ViewingCube/Faces/NSEW/xneg.png";
  vCube->facePosY = "$OIVHOME/examples/data/Inventor/ViewingCube/Faces/NSEW/ypos.png";
  vCube->faceNegY = "$OIVHOME/examples/data/Inventor/ViewingCube/Faces/NSEW/yneg.png";
  vCube->facePosZ = "$OIVHOME/examples/data/Inventor/ViewingCube/Faces/NSEW/zpos.png";
  vCube->faceNegZ = "$OIVHOME/examples/data/Inventor/ViewingCube/Faces/NSEW/zneg.png";

  // Set default camera orientation
  orbiter->getCamera()->orientation = SbRotation(SbVec3f(-1.0f, 0.0f, 0.0f), 0.5f) * SbRotation(SbVec3f(0.0f, 1.0f, 0.0f), 0.4f);

  renderArea->setSceneGraph(scene.getSceneGraph());
  renderArea->viewAll(SbViewportRegion());
  centralLayout->addWidget(renderArea->getContainerWidget());

  window->setCentralWidget(centralWidget);
  window->resize(1024, 768);
  window->show();

  renderArea->getGLRenderAction()->setTransparencyType(SoGLRenderAction::SORTED_PIXEL);

  app.exec();

  scene.clear();
  delete window;

  SoVolumeRendering::finish();
  SoInteraction::finish();

  return EXIT_SUCCESS;
}
