/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#include <QApplication>
#include <QGridLayout>
#include <QMainWindow>
#include <GUIWidget.h>
#include <Inventor/Qt/SbQtHelper.h>
#include <Inventor/ViewerComponents/Qt/RenderAreaOrbiter.h>
#include <Inventor/Axis.h>
#include <Inventor/SoInteraction.h>
#include <Inventor/SoPickedPoint.h>
#include <Inventor/draggers/SoDragPointDragger.h>
#include <Inventor/draggers/SoScale1Dragger.h>
#include <Inventor/engines/SoCompose.h>
#include <Inventor/events/SoLocation2Event.h>
#include <Inventor/nodes/SoAlgebraicSphere.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoFontStyle.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoRotation.h>
#include <Inventor/nodes/SoScale.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoTessellationEvaluationShader.h>
#include <Inventor/sensors/SoFieldSensor.h>
#include <Inventor/ViewerComponents/nodes/SoViewingCube.h>
#include <VolumeViz/details/SoHeightFieldDetail.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoHeightFieldGeometry.h>
#include <VolumeViz/nodes/SoHeightFieldRender.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeShader.h>

SoHeightFieldGeometry* g_heightFieldGeom = nullptr;
SoTranslation* g_sphereTranslation = nullptr;
SoShaderParameter1f* g_heightScaleParam = nullptr;
SoSwitch* g_sphereSwitch = nullptr;

// Custom HeightFieldGeometry to overload voxelToXYZ and XYZToVoxel methods
// We overload these methods to get correct bounding boxes, tile loading and
// picking values when a shift is applied.
class SoCustomHeightFieldGeometry : public SoHeightFieldGeometry
{
public:
  // Used for bbox and tile loading
  // We apply the shift after calling the parent method
  virtual SbVec3f voxelToXYZ( const SbVec3f& dataPosition ) const
  {
    return vertexShift( SoHeightFieldGeometry::voxelToXYZ( dataPosition ) );
  }

  // Used for bbox and tile loading
  // We apply voxelToXYZ() to all 8 corners of the input box
  // then we compute the global min and max values for all 3 dimensions
  // and we create the converted box using this min and max.
  // This is an approximation because the projection is a non-linear
  // transformation, but it is a good enough heuristic here.
  virtual SbBox3f voxelToXYZ( const SbBox3f& box ) const
  {
    float xmin, ymin, zmin, xmax, ymax, zmax;
    box.getBounds( xmin, ymin, zmin, xmax, ymax, zmax );

    SbVec3f corners[8] = {
      SbVec3f( xmin, ymin, zmin ),
      SbVec3f( xmin, ymin, zmax ),
      SbVec3f( xmin, ymax, zmin ),
      SbVec3f( xmin, ymax, zmax ),
      SbVec3f( xmax, ymin, zmin ),
      SbVec3f( xmax, ymin, zmax ),
      SbVec3f( xmax, ymax, zmin ),
      SbVec3f( xmax, ymax, zmax ),
    };

    for ( int i = 0; i < 8; i++ )
    {
      corners[i] = voxelToXYZ( corners[i] );

      if ( i == 0 )
      {
        xmin = xmax = corners[i][0];
        ymin = ymax = corners[i][1];
        zmin = zmax = corners[i][2];
      }
      else
      {
        xmin = SbMathHelper::Min( xmin, corners[i][0] );
        ymin = SbMathHelper::Min( ymin, corners[i][1] );
        zmin = SbMathHelper::Min( zmin, corners[i][2] );
        xmax = SbMathHelper::Max( xmax, corners[i][0] );
        ymax = SbMathHelper::Max( ymax, corners[i][1] );
        zmax = SbMathHelper::Max( zmax, corners[i][2] );
      }
    }

    return SbBox3f( xmin, ymin, zmin, xmax, ymax, zmax );
  }

  // Used during heightfield picking to fill IJK detail
  // We apply the inverse shift before calling the parent method
  virtual SbVec3f XYZToVoxel( const SbVec3f& dataPosition ) const
  {
    return SoHeightFieldGeometry::XYZToVoxel( vertexShiftInverse( dataPosition ) );
  }

  // Used for bbox and tile loading
  // We apply XYZToVoxel() to all 8 corners of the input box
  // then we compute the global min and max values for all 3 dimensions
  // and we create the converted box using this min and max.
  // This is an approximation because the projection is a non-linear
  // transformation, but it is a good enough heuristic here.
  virtual SbBox3f XYZToVoxel( const SbBox3f& box ) const
  {
    float xmin, ymin, zmin, xmax, ymax, zmax;
    box.getBounds( xmin, ymin, zmin, xmax, ymax, zmax );

    SbVec3f corners[8] = {
      SbVec3f( xmin, ymin, zmin ),
      SbVec3f( xmin, ymin, zmax ),
      SbVec3f( xmin, ymax, zmin ),
      SbVec3f( xmin, ymax, zmax ),
      SbVec3f( xmax, ymin, zmin ),
      SbVec3f( xmax, ymin, zmax ),
      SbVec3f( xmax, ymax, zmin ),
      SbVec3f( xmax, ymax, zmax ),
    };

    for ( int i = 0; i < 8; i++ )
    {
      corners[i] = XYZToVoxel( corners[i] );

      if ( i == 0 )
      {
        xmin = xmax = corners[i][0];
        ymin = ymax = corners[i][1];
        zmin = zmax = corners[i][2];
      }
      else
      {
        xmin = SbMathHelper::Min( xmin, corners[i][0] );
        ymin = SbMathHelper::Min( ymin, corners[i][1] );
        zmin = SbMathHelper::Min( zmin, corners[i][2] );
        xmax = SbMathHelper::Max( xmax, corners[i][0] );
        ymax = SbMathHelper::Max( ymax, corners[i][1] );
        zmax = SbMathHelper::Max( zmax, corners[i][2] );
      }
    }

    return SbBox3f( xmin, ymin, zmin, xmax, ymax, zmax );
  }

private:
  // CPU implementation of the vertex shift
  // This should be the same shift as the one defined by the
  // VVizTessVertexShift() function in the shift shader
  static SbVec3f vertexShift( const SbVec3f& position )
  {
    const SbVec3f& spherePosition = g_sphereTranslation->translation.getValue();
    SbVec2f XYorientation = SbVec2f( position ) - SbVec2f( spherePosition );
    const float angle = XYorientation.normalize() / spherePosition[2];
    XYorientation *= sinf( angle );
    const SbVec3f projectedFloor = spherePosition[2] * SbVec3f( XYorientation[0], XYorientation[1], 1.0f - cosf( angle ) );
    SbVec3f projectedFloorDir = projectedFloor - spherePosition;
    projectedFloorDir.normalize();
    return spherePosition + (fabs( spherePosition[2] ) + g_heightScaleParam->value.getValue() * position[2]) * projectedFloorDir;
  }

  // The inverse vertex shift
  // Given a shifted vertex, this method yields the original vertex position.
  static SbVec3f vertexShiftInverse( const SbVec3f& position )
  {
    const SbVec3f& spherePosition = g_sphereTranslation->translation.getValue();
    SbVec3f projectedFloorDir = position - spherePosition;
    const float sphereRadius = fabs( spherePosition[2] );
    const float height = (projectedFloorDir.normalize() - sphereRadius) / g_heightScaleParam->value.getValue();
    const float XYLength = sphereRadius * acosf( projectedFloorDir[2] );
    SbVec2f XYDir( projectedFloorDir );
    XYDir.normalize();
    XYDir *= XYLength;
    return SbVec3f( XYDir[0], XYDir[1], height );
  }
};

// Callback used to update picking info
void eventCallbackCB( void* data, SoEventCallback* node )
{
  SoText2* pickInfoText = static_cast<SoText2*>( data );

  const SoHeightFieldDetail* detail = nullptr;
  const SoPickedPoint* pickedPoint = node->getPickedPoint();
  if ( pickedPoint != nullptr )
    detail = dynamic_cast<const SoHeightFieldDetail*>(pickedPoint->getDetail());

  if ( detail != nullptr )
  {
    const SbVec3i32& ijk = detail->getIjkPos();
    pickInfoText->string.set1Value( 1, SbString( "IJK: (" ) + ijk[0] + ", " + ijk[1] + ", " + ijk[2] + ")" );
    pickInfoText->string.set1Value( 2, SbString( "Height: " ) + detail->getHeight() );
  }
  else
  {
    pickInfoText->string.set1Value( 1, "IJK: " );
    pickInfoText->string.set1Value( 2, "Height: " );
  }
}

// When the shift parameters change, the bounding boxes of the tiles may need to be updated
// but the VolumeData node has no way of knowing when to do the update.
// This callback is called when the shift parameters change and performs a trick
// to force the update of the tiles bounding boxes.
void projectionChangedCB( void* , SoSensor* )
{
  const SbBox3f& extentValue = g_heightFieldGeom->extent.getValue();
  SbVec3f extentMax = extentValue.getMax();

  // Modifying the extent of a SoDataSet triggers a re-computation of the tiles
  // bounding boxes. The Z component of the extent is unused for HeightFields so
  // we change it to trigger the re-computation without affecting the rendering.
  // (we set the Z max value to 0 or 1 depending on the previous value)
  extentMax[2] = (extentMax[2] > 0.0f) ? 0.0f : 1.0f;

  g_heightFieldGeom->extent.setValue( extentValue.getMin(), extentMax );
}

SoNode*
buildSceneGraph()
{
  SoSeparator* root = new SoSeparator;

  // Pick Info
  {
    SoSeparator* pickInfoSep = new SoSeparator;
    root->addChild( pickInfoSep );

    // Callback to update pick info
    SoEventCallback* eventCBNode = new SoEventCallback;
    pickInfoSep->addChild( eventCBNode );

    SoPickStyle* pickInfoPickStyle = new SoPickStyle;
    pickInfoPickStyle->style = SoPickStyle::UNPICKABLE;
    pickInfoSep->addChild( pickInfoPickStyle );

    SoOrthographicCamera* pickInfoCam = new SoOrthographicCamera;
    pickInfoCam->viewportMapping = SoCamera::LEAVE_ALONE;
    pickInfoCam->position = SbVec3f( 0.005f, 0.005f, 1.0f );
    pickInfoCam->height = 0.01f;
    pickInfoSep->addChild( pickInfoCam );

    SoMaterial* pickInfoMaterial = new SoMaterial;
    pickInfoMaterial->ambientColor = SbColor( 0.0f, 0.0f, 0.0f );
    pickInfoMaterial->diffuseColor = SbColor( 0.0f, 0.0f, 0.0f );
    pickInfoSep->addChild( pickInfoMaterial );

    SoTranslation* pickInfoTranslation = new SoTranslation;
    pickInfoTranslation->translation = SbVec3f( 0.0002f, 0.0094f, 0.0f );
    pickInfoSep->addChild( pickInfoTranslation );

    SoFontStyle* pickInfoFontStyle = new SoFontStyle;
    pickInfoFontStyle->size = 20.0f;
    pickInfoFontStyle->family = SoFontStyle::SANS;
    pickInfoSep->addChild( pickInfoFontStyle );

    SoText2* pickInfoText = new SoText2;
    pickInfoText->string.set1Value( 0, "Pick info:" );
    pickInfoText->string.set1Value( 1, "IJK: " );
    pickInfoText->string.set1Value( 2, "Height: " );
    pickInfoSep->addChild( pickInfoText );

    eventCBNode->addEventCallback<SoLocation2Event>( eventCallbackCB, pickInfoText );
  }

  // A node to represent the projection sphere translation
  // Its radius is defined as the absolute value of the Z component
  g_sphereTranslation = new SoTranslation;
  g_sphereTranslation->translation = SbVec3f( 0.0f, 0.0f, -1.0f );

  // a semi-transparent sphere to visualize the projection
  {
    g_sphereSwitch = new SoSwitch;
    g_sphereSwitch->whichChild = SO_SWITCH_ALL;
    root->addChild( g_sphereSwitch );

    SoSeparator* sphereSep = new SoSeparator;
    g_sphereSwitch->addChild( sphereSep );

    SoTranslation* sphereTranslation = new SoTranslation;
    sphereTranslation->translation.connectFrom( &g_sphereTranslation->translation );
    sphereSep->addChild( sphereTranslation );

    SoDecomposeVec3f* decomposeTranslation = new SoDecomposeVec3f;
    decomposeTranslation->vector.connectFrom( &g_sphereTranslation->translation );

    SoPickStyle* spherePickStyle = new SoPickStyle;
    spherePickStyle->style = SoPickStyle::UNPICKABLE;
    sphereSep->addChild( spherePickStyle );

    SoMaterial* sphereMaterial = new SoMaterial;
    sphereMaterial->transparency = 0.9f;
    sphereSep->addChild( sphereMaterial );

    SoAlgebraicSphere* sphere = new SoAlgebraicSphere;
    sphere->radius.connectFrom( &decomposeTranslation->z );
    sphereSep->addChild( sphere );
  }

  // VolumeViz nodes used to render the horizon
  {
    SoSeparator* horizonNodes = new SoSeparator;
    root->addChild( horizonNodes );

    // Horizon geometry data node
    // We use a custom derived SoHeightFieldGeometry
    // with some overloaded methods (see SoCustomHeightFieldGeometry class definition)
    {
      g_heightFieldGeom = new SoCustomHeightFieldGeometry;
      g_heightFieldGeom->fileName = "$OIVHOME/examples/data/VolumeViz/horizons/horizon.ldm";
      horizonNodes->addChild( g_heightFieldGeom );
    }

    // A transfer function to add some color
    {
      SoTransferFunction* transferFunction = new SoTransferFunction;
      transferFunction->predefColorMap = SoTransferFunction::STANDARD;
      horizonNodes->addChild( transferFunction );
    }

    // Horizon Z scale parameter
    {
      g_heightScaleParam = new SoShaderParameter1f;
      g_heightScaleParam->name = "heightScale";
      g_heightScaleParam->value = 0.2f;
    }

    // Projection sphere position parameter
    SoShaderParameter3f* spherePositionParam = new SoShaderParameter3f;
    {
      spherePositionParam->name = "spherePosition";
      spherePositionParam->value.connectFrom( &g_sphereTranslation->translation );
    }

    // When the shift parameters change, the bounding boxes of the tiles may need to be updated
    // but the VolumeData node has no way of knowing when to do the update, so we need a trick
    // to force the re-computation of those bounding boxes.
    // We use sensors to know when the shift parameters change.
    // The trick is done in projectionChangedCB()
    {
      SoFieldSensor* heightScaleSensor = new SoFieldSensor;
      heightScaleSensor->setFunction( projectionChangedCB );
      heightScaleSensor->attach( &g_heightScaleParam->value );

      SoFieldSensor* spherePositionSensor = new SoFieldSensor;
      spherePositionSensor->setFunction( projectionChangedCB );
      spherePositionSensor->attach( &spherePositionParam->value );
    }

    // Shift shader
    SoTessellationEvaluationShader* tessShader = new SoTessellationEvaluationShader;
    tessShader->sourceProgram = "$OIVHOME/examples/source/VolumeViz/horizonShiftProjection/shiftProjection_tess.glsl";

    // Send projection parameters to the shift shader
    tessShader->parameter.set1Value( 0, g_heightScaleParam );
    tessShader->parameter.set1Value( 1, spherePositionParam );

    // Setup shift shader in TESS_VERTEX_SHIFT slot
    SoVolumeShader* volumeShader = new SoVolumeShader;
    volumeShader->shaderObject.set1Value( SoVolumeShader::TESS_VERTEX_SHIFT, tessShader );
    horizonNodes->addChild( volumeShader );

    // Finally, add the render node
    horizonNodes->addChild( new SoHeightFieldRender );
  }

  return root;
}

int
main( int argc, char** argv )
{
  SbQtHelper::addPlatformPluginsPath();

  // Create window
  QApplication app( argc, argv );
  QMainWindow* window = new QMainWindow( NULL );
  window->setWindowTitle( "Horizon Shift Projection" );

  SoInteraction::init();
  SoVolumeRendering::init();

  SoRef<SoNode> root = buildSceneGraph();

  // Setup widgets layout
  QWidget* centralWidget = new QWidget( window );
  QVBoxLayout* centralLayout = new QVBoxLayout;
  centralLayout->setContentsMargins( 0, 0, 0, 0 );
  centralLayout->setSpacing( 0 );
  centralWidget->setLayout( centralLayout );

  // Create GUI widget and setup callbacks to connect GUI and scenegraph changes
  GUIWidget* guiWidget = new GUIWidget( window );
  guiWidget->setCurvature( abs(0.5 / (double) g_sphereTranslation->translation.getValue()[2]) );
  guiWidget->setHeightScale( (double) g_heightScaleParam->value.getValue() );
  QObject::connect( guiWidget, &GUIWidget::curvatureModified, [&]() mutable
  {
    const double curvature = guiWidget->getCurvature();
    float radius;
    if ( curvature == 0.0 )
    {
      // prevent division by zero and sphere rendering
      radius = 1000.0f;
      g_sphereSwitch->whichChild = SO_SWITCH_NONE;
    }
    else
    {
      radius = static_cast<float>(0.5 / curvature);
      g_sphereSwitch->whichChild = SO_SWITCH_ALL;
    }
    g_sphereTranslation->translation = SbVec3f( 0.0f, 0.0f, -radius );
  } );
  QObject::connect( guiWidget, &GUIWidget::heightScaleModified, [&]() mutable
  {
    g_heightScaleParam->value = guiWidget->getHeightScale();
  } );
  centralLayout->addWidget( guiWidget );

  // Setup Render Area
  RenderAreaOrbiter* renderArea = new RenderAreaOrbiter( window );
  renderArea->setClearColor( SbColorRGBA( 0.96f, 0.96f, 0.96f, 1.0f ) );
  renderArea->setAntialiasingMode( SoSceneManager::AUTO );
  renderArea->setAntialiasingQuality( 1 );
  SceneOrbiter* orbiter = renderArea->getSceneInteractor();
  orbiter->setUpAxis( openinventor::inventor::Axis::Z );
  orbiter->setRotationMethod( SceneOrbiter::TRACKBALL );
  renderArea->setSceneGraph( root.ptr() );
  renderArea->viewAll( SbViewportRegion() );
  centralLayout->addWidget( renderArea->getContainerWidget() );

  window->setCentralWidget( centralWidget );
  window->resize( 1024, 768 );
  window->show();

  renderArea->getGLRenderAction()->setTransparencyType( SoGLRenderAction::SORTED_PIXEL );

  app.exec();

  root = NULL;
  delete window;

  SoVolumeRendering::finish();
  SoInteraction::finish();

  return EXIT_SUCCESS;
}
