/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Mike Heck (Dec 2008)
**=======================================================================*/

/*----------------------------------------------------------------------

Example program:
Version 2 of the volume shader for multi-channel data.

This version adds support for the image enhancement features of the
SoVolumeRenderingQuality node, with the exception of lighting and
pre-integrated rendering.  This includes support for jittering,
high quality gradient computation, boundary opacity, edge coloring
and 2D edge detection.  This version has a tiny bit of user interface,
allowing a few of the main features to be enabled and disabled by
pressing the appropriate keys (with the viewer in selection mode).

This example could be easily extended to also support lighting. Follow
the same strategy, making a copy of the standard fragment main shader
and making essentially the same modifications, but start with the code
in "vvizvolumelight_frag.glsl".  Also modify the application program
to load the appropriate fragment main shader depending on whether
lighting is enabled or not.  Extending to support pre-integrated
rendering should also be possible, but more complicated.  Perhaps in
a future example...

This test data was created by segmenting the tiny data set SYN_64.vol
into 3 "channels".  Every combination of the three channels is visible
in the resulting rendering.

This example only shows volume rendering and assumes 3 channels of
data.  To do slice rendering and/or handle a different number of
channels requires a slightly different shader program and corresponding
different setup of the shader parameters.  It should be fairly clear
how to do after understanding this example.

----------------------------------------------------------------------*/

// Header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoFragmentShader.h>
#include <Inventor/nodes/SoSeparator.h>

#include <LDM/nodes/SoDataRange.h>
#include <LDM/nodes/SoMultiDataSeparator.h>

#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoVolumeShader.h>

#include <Inventor/events/SoKeyboardEvent.h>

#include <Inventor/helpers/SbFileHelper.h>

#include <iostream>
using namespace std;

#include "MakeBBox.h"

////////////////////////////////////////////////////////////////////////
// Forward declarations

void EventCB(void *userData, SoEventCallback *node );
SoVolumeRenderingQuality* makeVolumeShader();
SoTransferFunction *makeColorMap( SbColor color, float gamma, int id );


////////////////////////////////////////////////////////////////////////
// Multi-channel fragment shader for volume rendering
// Version 2: Supports jittering, boundary opacity, etc.
//            but not lighting or pre-integrated rendering.
//
// The basic main program comes from the file "vvizvolume_frag.glsl".
// The following changes were made:
//   - Add our uniform variables (volData1, etc) to get the colormap
//     and channel data volumes from VolumeViz.
//   - Remove the call to VVizCombineData.
//     This function is not needed because it is intended to combine
//     voxel values and we need to combine color values.
//   - Replace the call to VVizComputeVolumeRendering with a call to
//     our own function ComputeMultichannelRendering, which also returns
//     a computed color value.
//
// Notes
// 1) This version is for volume rendering.
//    (Must make a separate version for slice/skin rendering.)
// 2) This version is for 3 channels of data.
//    (Must make a separate version for 2 channels.)
// 3) This shader assumes that 3D textures are used.
// 4) It seems reasonable to sum the color (RGB) components.  But we
//    probably don't want the voxel to be more opaque simply because
//    values exist in multiple channels, so we'll use the highest
//    opacity value in any of the channels.  Does that make sense?

// We also override the VVizCombineData function.
//
// This is a function provided by the VolumeViz shader library. We don't
// actually use it for basic multichannel rendering, because we combine
// color values, not data values.  However we need it for some of the
// image enhancement techniques, like boundary opacity and edge
// coloring that require computation of the gradient vector.  If we
// just used the default CombineData function then the gradient would be
// computed using only the first channel (volume).

////////////////////////////////////////////////////////////////////////
//
// Make the volume shader node
//
// Builds the FragmentShader and VolumeShader objects and sets up
// the basic parameters.
//
// Note: This version assumes there are 3 channels of data and the
//       data will be rendered using SoVolumeRender.

SoVolumeRenderingQuality* makeVolumeShader()
{
  // Load fragment shader main program
  SoFragmentShader* fragmentShader = new SoFragmentShader();
  fragmentShader->sourceProgram.setValue( "$OIVHOME/examples/source/VolumeViz/multiChannel/MultiChannel_ex2_color_frag.glsl" );

  // Create uniform parameters allowing shader to access textures
  //
  // Volume data is in texture units 1, 2 and 3
  SoShaderParameter1i *paramTex0 = new SoShaderParameter1i;
  paramTex0->name = "voldata1";
  paramTex0->value.setValue(1);
  SoShaderParameter1i *paramTex1 = new SoShaderParameter1i;
  paramTex1->name = "voldata2";
  paramTex1->value.setValue(2);
  SoShaderParameter1i *paramTex2 = new SoShaderParameter1i;
  paramTex2->name = "voldata3";
  paramTex2->value.setValue(3);

  fragmentShader->parameter.set1Value(0, paramTex0);
  fragmentShader->parameter.set1Value(1, paramTex1);
  fragmentShader->parameter.set1Value(2, paramTex2);

  // Load replacement function for VVizCombineData
  SoFragmentShader* combineDataShader = new SoFragmentShader();
  combineDataShader->sourceProgram.setValue( "$OIVHOME/examples/source/VolumeViz/multiChannel/MultiChannel_ex2_combine_frag.glsl" );
  combineDataShader->parameter.set1Value(0, paramTex0);
  combineDataShader->parameter.set1Value(1, paramTex1);
  combineDataShader->parameter.set1Value(2, paramTex2);

  // Initialize the volume shader
  SoVolumeRenderingQuality* pVolShader = new SoVolumeRenderingQuality;
  pVolShader->deferredLighting.setValue( FALSE );
  pVolShader->preIntegrated.setValue( FALSE );
  pVolShader->shaderObject.set1Value(SoVolumeShader::FRAGMENT_COMPUTE_COLOR, fragmentShader);
  pVolShader->shaderObject.set1Value(SoVolumeShader::DATA_COMBINE_FUNCTION, combineDataShader);

  // Specify shader should be applied to SoVolumeRender nodes.
  // (Name is a little deceptive because you might think that FALSE means
  // for BOTH volumes and slices, but it's really one or the other.)
  pVolShader->forVolumeOnly = TRUE;

  return pVolShader;
}


////////////////////////////////////////////////////////////////////////
//
// Make a multi-channel colormap (transfer function)
//
// This function creates a colormap appropriate for (some) multi-channel
// volume data.  It contains a linear color ramp, i.e. the minimum data
// value is mapped to black and the maximum data value is mapped to the
// full color.  The alpha ramp is a gamma curve, so the higher the gamma
// value the more strongly low valued voxels are suppressed.
//
// The colormaps must have unique id numbers, usually matching the
// dataSetId of the associated volume.

SoTransferFunction *
makeColorMap( SbColor color, float gamma, int id )
{
  SoTransferFunction *pTF = new SoTransferFunction();
  pTF->transferFunctionId = id;

  // Color map will contain 256 RGBA values -> 1024 float values.
  // setNum pre-allocates memory so we can edit the field directly.
  pTF->colorMap.setNum( 256 * 4 );

  float R = color[0];
  float G = color[1];
  float B = color[2];

  // Get an edit pointer, assign values, then finish editing.
  float *p = pTF->colorMap.startEditing();
  for (int i = 0; i < 256; ++i) {
    float factor = (float)i/255;
    *p++ = R * factor;
    *p++ = G * factor;
    *p++ = B * factor;
    *p++ = pow(factor,gamma);
  }
  pTF->colorMap.finishEditing();
  return pTF;
}


////////////////////////////////////////////////////////////////////////
//main function
int main(int, char **argv)
{
  cout << "Press 'B' to toggle Boundary opacity  (def = off)\n";
  cout << "      'E' to toggle Edge coloring     (def = off)\n";
  cout << "      '2' to toggle 2D edge detection (def = off)\n";
  cout << "      'J' to toggle Jittering         (def = off)\n";
  cout << "      '-' and '+' to change gradient threshold (def = 0.3)\n";
  cout << "      '<-' and '->' to change edge threshold     (def = 0)\n";

  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  // Uncomment this line to get more info about compile and link
  // problems with GLSL shaders.
//  SoPreferences::setBool( "OIV_GLSL_DEBUG", 1 );

  // Initialize VolumeViz extension
  SoVolumeRendering::init();

  // Load volume data
  // Each channel is in a separate file.
  // Each volume must have a different dataSetId.
  SoVolumeData* pVolData0 = new SoVolumeData();
  pVolData0->fileName  = "$OIVHOME/examples/source/VolumeViz/multiChannel/syn_r_chan.vol";
  pVolData0->dataSetId = 1;

  SoVolumeData* pVolData1 = new SoVolumeData();
  pVolData1->fileName  = "$OIVHOME/examples/source/VolumeViz/multiChannel/syn_g_chan.vol";
  pVolData1->dataSetId = 2;

  SoVolumeData* pVolData2 = new SoVolumeData();
  pVolData2->fileName  = "$OIVHOME/examples/source/VolumeViz/multiChannel/syn_b_chan.vol";
  pVolData2->dataSetId = 3;

  // ---------- Map actual range of data values ----------
  // This test data only uses values 0..192, so we will not use all the
  // values in the colormap unless we explicitly set the data range.
  // Note:
  // 1. Each volume data node must have its own data range node.
  // 2. The dataRangeId must be the same as the dataSetId of the
  //    corresponding volume data node.
  //
  // In many cases we don't need data range nodes for 8-bit (byte) data
  // because all values 0..255 are used, but we do need them in most
  // cases for larger data types, e.g. 16 bit data sets often only
  // contain 12 bits of actual data.
  //
  // Note that the getMinMax query is "free" for LDM format data,
  // because the min and max values are stored in the file header.
  // Custom readers may also support this query.  But many of the
  // standard volume readers have to load the entire data set from
  // disk to answer this query and this can be very time consuming.
  // If using such a reader, when possible, preprocess the data and
  // store the min/max values in some sort of "info" file that the
  // application can load quickly and avoid calling getMinMax().
  double dataMin, dataMax;
  pVolData0->getMinMax( dataMin, dataMax );
  SoDataRange *pDataRange0 = new SoDataRange();
  pDataRange0->min = dataMin;
  pDataRange0->max = dataMax;
  pDataRange0->dataRangeId = 1;

  pVolData1->getMinMax( dataMin, dataMax );
  SoDataRange *pDataRange1 = new SoDataRange();
  pDataRange1->min = dataMin;
  pDataRange1->max = dataMax;
  pDataRange1->dataRangeId = 2;

  pVolData2->getMinMax( dataMin, dataMax );
  SoDataRange *pDataRange2 = new SoDataRange();
  pDataRange2->min = dataMin;
  pDataRange2->max = dataMax;
  pDataRange2->dataRangeId = 3;

  // Make color map for each channel
  // Typically for multichannel rendering we will create a colormap
  // with a linear color ramp, i.e. black -> full color, and a gamma
  // curve alpha ramp.
  // Note that each volume must have a unique transferFunctionId,
  // which generally should be the same as the corresponding volume
  // data node's dataSetId.
  SoTransferFunction* pTF0 = makeColorMap( SbColor(1,0,0), 2.39f, 1 );
  SoTransferFunction* pTF1 = makeColorMap( SbColor(0,1,0), 2.39f, 2 );
  SoTransferFunction* pTF2 = makeColorMap( SbColor(0,0,1), 2.39f, 3 );

  // Render node
  // Set numSlicesControl to automatic, so we can specify the number of
  // slices to draw (if necessary) and the viewer can reduce the number
  // of slices while moving.
  //
  // Note that because our test volume is so small (64 x 64 x 64), we
  // require VolumeViz to render 128 slices through the volume to
  // improve the image quality (effectively we are supersampling).
  // This is particularly important because the VolumeRenderingQuality
  // node forces rendering using view aligned slices.
  // --> Set this to an appropriate value for your data volume.
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->samplingAlignment = SoVolumeRender::DATA_ALIGNED;
  pVolRender->numSlicesControl = SoVolumeRender::AUTOMATIC;
  pVolRender->numSlices = 128;

  // Volume shader and rendering parameters
  // Notes:
  // This test volume isn't ideal for showing boundary opacity because
  // there is already a very sharp change from zero valued voxels to
  // non-zero voxels.  What the boundary opacity effect will appear to
  // do is darken the surfaces of the "object".  This is because it is
  // increasing the opacity of the zero valued voxels adjacent to the
  // "object" and those voxels are black (RGB = 0,0,0).  To reduce this
  // effect and because we know only large gradients are significant in
  // this data, we'll increase the gradientThreshold so that small
  // gradients are ignored.
  //
  // To emphasize the effect of edge coloring, we'll set the edgeColor
  // to full white.  The effect will appear to create a "halo" around
  // the object because the voxel color is changed to white where the
  // gradient vector is nearly perpendicular to the view vector.  Note
  // that this color applies to edge coloring and 2D edge detection.
  // Use values appropriate for your data.
  SoVolumeRenderingQuality *pVolShader = makeVolumeShader();
  pVolShader->gradientThreshold = 0.3f;        // Default is zero
  pVolShader->edgeColor = SbColor(1,1,1);   // Default is 0,0,0

  // Assemble the volume rendering scene graph.
  // Note that we use a MultiDataSeparator node to hold multiple data
  // sets that will be combined in a single rendering node.
  SoMultiDataSeparator *pVolSep = new SoMultiDataSeparator();
  pVolSep->addChild( pTF0 );
  pVolSep->addChild( pTF1 );
  pVolSep->addChild( pTF2 );
  pVolSep->addChild( pVolShader );
  pVolSep->addChild( pVolData0 );
  pVolSep->addChild( pVolData1 );
  pVolSep->addChild( pVolData2 );
  pVolSep->addChild( pDataRange0 );
  pVolSep->addChild( pDataRange1 );
  pVolSep->addChild( pDataRange2 );
  pVolSep->addChild( pVolRender );

  SoSeparator *root = new SoSeparator();
  root->ref();
  root->addChild( pVolSep );
  root->addChild( makeBBox( pVolData1->extent.getValue()) );

  // Setup callback node to detect key press events.
  // Pass volume shader node to callback function.
  SoEventCallback *pEvCB = new SoEventCallback;
  pEvCB->addEventCallback( SoKeyboardEvent::getClassTypeId(),
                           EventCB, (void*)pVolShader );
  root->insertChild( pEvCB, 0 );

  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setSceneGraph(root);
  myViewer->setTitle("Multichannel rendering");
  myViewer->setTransparencyType( SoGLRenderAction::OPAQUE_FIRST );
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete myViewer;
  root->unref();
  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}
////////////////////////////////////////////////////////////////////////
//
// Handles key press events

void EventCB(void *userData, SoEventCallback *node )
{
  // Get the event that triggered the callback
  const SoEvent *event = node->getEvent();
  SoVolumeRenderingQuality *pVRQ = (SoVolumeRenderingQuality *)userData;

  // If 'B' pressed, toggle boundary opacity
  if (SO_KEY_PRESS_EVENT(event, B)) {
    SbBool state = !pVRQ->boundaryOpacity.getValue();
    pVRQ->boundaryOpacity = state;
    cout << "Boundary opacity: " << (state ? "ON" : "Off") << endl;
  }
  // If 'E' pressed, toggle edge coloring
  else if (SO_KEY_PRESS_EVENT(event, E)) {
    SbBool state = !pVRQ->edgeColoring.getValue();
    pVRQ->edgeColoring = state;
    cout << "Edge coloring: " << (state ? "ON" : "Off") << endl;
    ;
  }
  // If '2' pressed, toggle 2D edge detection
  else if (SO_KEY_PRESS_EVENT(event, NUMBER_2) ||
           SO_KEY_PRESS_EVENT(event, PAD_2)) {
    SbBool state = !pVRQ->edgeDetect2D.getValue();
    pVRQ->edgeDetect2D = state;
    cout << "Edge detect 2D: " << (state ? "ON" : "Off") << endl;
    ;
  }
  // If 'J' pressed, toggle jittering
  else if (SO_KEY_PRESS_EVENT(event, J)) {
    SbBool state = !pVRQ->jittering.getValue();
    pVRQ->jittering = state;
    cout << "Jittering: " << (state ? "ON" : "Off") << endl;
    ;
  }

  // If '-' or '+' pressed, decrement or increment gradient threshold
  else if (SO_KEY_PRESS_EVENT(event, MINUS)) {
    float gradThr = pVRQ->gradientThreshold.getValue();
    gradThr -= 0.1f;
    if (gradThr < 1e-6) gradThr = 0;
    pVRQ->gradientThreshold = gradThr;
    cout << "gradientThreshold = " << gradThr << endl;
  }
  else if (SO_KEY_PRESS_EVENT(event, EQUAL)) {
    float gradThr = pVRQ->gradientThreshold.getValue();
    gradThr += 0.1f;
    pVRQ->gradientThreshold = gradThr;
    cout << "gradientThreshold = " << gradThr << endl;
  }

  // If '<-' or '->' pressed, decrement or increment edge threshold
  else if (SO_KEY_PRESS_EVENT(event, LEFT_ARROW)) {
    float edgeThr = pVRQ->edgeThreshold.getValue();
    edgeThr -= 0.1f;
    if (edgeThr < 1e-6) edgeThr = 0;
    pVRQ->edgeThreshold = edgeThr;
    cout << "edgeThreshold = " << edgeThr << endl;
  }
  else if (SO_KEY_PRESS_EVENT(event, RIGHT_ARROW)) {
    float edgeThr = pVRQ->edgeThreshold.getValue();
    edgeThr += 0.1f;
    pVRQ->edgeThreshold = edgeThr;
    cout << "edgeThreshold = " << edgeThr << endl;
  }
}

