/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Mike Heck (Dec 2008)
**=======================================================================*/

/*----------------------------------------------------------------------

Example program.
Shows a simple volume shader for multi-channel data.

This test data was created by segmenting the tiny data set SYN_64.vol
into 3 "channels".  Every combination of the three channels is visible
in the resulting rendering.

This example only shows volume rendering and assumes 3 channels of
data.  To do slice rendering and/or handle a different number of
channels requires a slightly different shader program and corresponding
different setup of the shader parameters.

This example completely replaces the default VolumeViz fragment shader
main program (to keep things as simple as possible).  As a result it
does not support VolumeViz advanced rendering options such as jittering,
edge color and boundary opacity.  There is a separate example for this.
----------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoFragmentShader.h>

#include <LDM/nodes/SoDataRange.h>
#include <LDM/nodes/SoMultiDataSeparator.h> 

#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeShader.h>

#include "MakeBBox.h"


////////////////////////////////////////////////////////////////////////
//
// Make the volume shader node
//
// Builds the FragmentShader and VolumeShader objects and sets up
// the basic parameters.
//
// Note: This version assumes there are 3 channels of data and the
//       data will be rendered using SoVolumeRender.

SoVolumeShader* makeVolumeShader()
{
  // Load fragment shader program
  SoFragmentShader* fragmentShader = new SoFragmentShader();
  fragmentShader->sourceProgram.setValue( "$OIVHOME/examples/source/VolumeViz/multiChannel/MultiChannel_ex1_color_frag.glsl" );

  // Create uniform parameters allowing shader to access textures
  //
  // Transfer function is in texture unit 0 (default)
  SoShaderParameter1i *paramTex0 = new SoShaderParameter1i;
  paramTex0->name = "transfer";
  paramTex0->value.setValue(0);

  // Volume data is in texture units 1, 2 and 3
  SoShaderParameter1i *paramTex1 = new SoShaderParameter1i;
  paramTex1->name = "voldata1";
  paramTex1->value.setValue(1);
  SoShaderParameter1i *paramTex2 = new SoShaderParameter1i;
  paramTex2->name = "voldata2";
  paramTex2->value.setValue(2);
  SoShaderParameter1i *paramTex3 = new SoShaderParameter1i;
  paramTex3->name = "voldata3";
  paramTex3->value.setValue(3);

  fragmentShader->parameter.set1Value(0, paramTex0);
  fragmentShader->parameter.set1Value(1, paramTex1);
  fragmentShader->parameter.set1Value(2, paramTex2);
  fragmentShader->parameter.set1Value(3, paramTex3);
  
  // Initialize the volume shader
  // Note in this version we completely replace the default VolumeViz
  // fragment shader main program (to keep the example simple).
  SoVolumeShader* pVolShader = new SoVolumeShader;
  pVolShader->shaderObject.set1Value(SoVolumeShader::FRAGMENT_COMPUTE_COLOR, fragmentShader);

  // Specify shader should be applied to SoVolumeRender nodes.
  // (Name is a little deceptive because you might think that FALSE means
  // for BOTH volumes and slices, but it's really one or the other.)
  pVolShader->forVolumeOnly = TRUE;

  return pVolShader;
}

////////////////////////////////////////////////////////////////////////
//
// Make a multi-channel colormap (transfer function)
//
// This function creates a colormap appropriate for (some) multi-channel
// volume data.  It contains a linear color ramp, i.e. the minimum data
// value is mapped to black and the maximum data value is mapped to the
// full color.  The alpha ramp is a gamma curve, so the higher the gamma
// value the more strongly low valued voxels are suppressed.
//
// The colormaps must have unique id numbers, starting at zero.

SoTransferFunction *
makeColorMap( SbColor color, float gamma, int id )
{
  SoTransferFunction *pTF = new SoTransferFunction();
  pTF->transferFunctionId = id;

  // Color map will contain 256 RGBA values -> 1024 float values.
  // setNum pre-allocates memory so we can edit the field directly.
  pTF->colorMap.setNum( 256 * 4 );

  float R = color[0];
  float G = color[1];
  float B = color[2];

  // Get an edit pointer, assign values, then finish editing.
  float *p = pTF->colorMap.startEditing();
  for (int i = 0; i < 256; ++i) {
    float factor = (float)i/255;
    *p++ = R * factor;
    *p++ = G * factor;
    *p++ = B * factor;
    *p++ = pow(factor,gamma);
  }
  pTF->colorMap.finishEditing();
  return pTF;
}

////////////////////////////////////////////////////////////////////////
//main function
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

//  SoPreferences::setBool( "OIV_GLSL_DEBUG", 1 );
    
  // Initialize VolumeViz extension
  SoVolumeRendering::init();

  // ---------- Load volume data ----------
  // Each channel is in a separate file.
  // Note that each volume must have a unique dataSetId.
  SoVolumeData* pVolData0 = new SoVolumeData();
  pVolData0->fileName  = "$OIVHOME/examples/source/VolumeViz/multiChannel/syn_r_chan.vol";
  pVolData0->dataSetId = 1;

  SoVolumeData* pVolData1 = new SoVolumeData();
  pVolData1->fileName  = "$OIVHOME/examples/source/VolumeViz/multiChannel/syn_g_chan.vol";
  pVolData1->dataSetId = 2;

  SoVolumeData* pVolData2 = new SoVolumeData();
  pVolData2->fileName  = "$OIVHOME/examples/source/VolumeViz/multiChannel/syn_b_chan.vol";
  pVolData2->dataSetId = 3;

  // ---------- Map actual range of data values ----------
  // This test data only uses values 0..192, so we will not use all the
  // values in the colormap unless we explicitly set the data range.
  // Note:
  // 1. Each volume data node must have its own data range node.
  // 2. The dataRangeId must be the same as the dataSetId of the
  //    corresponding volume data node.
  //
  // In many cases we don't need data range nodes for 8-bit (byte) data
  // because all values 0..255 are used, but we do need them in most
  // cases for larger data types, e.g. 16 bit data sets often only
  // contain 12 bits of actual data.
  //
  // Note that the getMinMax query is "free" for LDM format data,
  // because the min and max values are stored in the file header.
  // Custom readers may also support this query.  But many of the
  // standard volume readers have to load the entire data set from
  // disk to answer this query and this can be very time consuming.
  // If using such a reader, when possible, preprocess the data and
  // store the min/max values in some sort of "info" file that the
  // application can load quickly and avoid calling getMinMax().
  double dataMin, dataMax;
  pVolData0->getMinMax( dataMin, dataMax );
  SoDataRange *pDataRange0 = new SoDataRange();
  pDataRange0->min = dataMin;
  pDataRange0->max = dataMax;
  pDataRange0->dataRangeId = 1;

  pVolData1->getMinMax( dataMin, dataMax );
  SoDataRange *pDataRange1 = new SoDataRange();
  pDataRange1->min = dataMin;
  pDataRange1->max = dataMax;
  pDataRange1->dataRangeId = 2;

  pVolData2->getMinMax( dataMin, dataMax );
  SoDataRange *pDataRange2 = new SoDataRange();
  pDataRange2->min = dataMin;
  pDataRange2->max = dataMax;
  pDataRange2->dataRangeId = 3;

  // ---------- Make color map for each channel ----------
  // Typically for multichannel rendering we will create a colormap
  // with a linear color ramp, i.e. black -> full color, and a gamma
  // curve alpha ramp.
  // Note that each volume must have a unique transferFunctionId,
  // which generally should be the same as the corresponding volume
  // data node's dataSetId.
  SoTransferFunction* pTF0 = makeColorMap( SbColor(1,0,0), 2.39f, 1 );
  SoTransferFunction* pTF1 = makeColorMap( SbColor(0,1,0), 2.39f, 2 );
  SoTransferFunction* pTF2 = makeColorMap( SbColor(0,0,1), 2.39f, 3 );

  // Render node
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->samplingAlignment = SoVolumeRender::DATA_ALIGNED;
  pVolRender->numSlices = 128;  // To improve image quality

  // Volume shader node
  SoVolumeShader *pVolShader = makeVolumeShader();

  // Assemble the volume rendering scene graph.
  // Note that we use a MultiDataSeparator node to hold multiple data
  // sets that will be combined in a single rendering node.
  SoMultiDataSeparator *pVolSep = new SoMultiDataSeparator();
  pVolSep->addChild( pVolShader );
  pVolSep->addChild( pVolData0 );
  pVolSep->addChild( pVolData1 );
  pVolSep->addChild( pVolData2 );
  pVolSep->addChild( pDataRange0 );
  pVolSep->addChild( pDataRange1 );
  pVolSep->addChild( pDataRange2 );
  pVolSep->addChild( pTF0 );
  pVolSep->addChild( pTF1 );
  pVolSep->addChild( pTF2 );
  pVolSep->addChild( pVolRender );

  // Assemble the main scene graph.
  // Includes a representation of the volume's bounding box.
  SoSeparator *root = new SoSeparator();
  root->ref();
  root->addChild( pVolSep );
  root->addChild( makeBBox( pVolData1->extent.getValue()) );

  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setSceneGraph(root);
  myViewer->setTitle("Multichannel rendering");
  myViewer->setTransparencyType( SoGLRenderAction::OPAQUE_FIRST );
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete myViewer;
  root->unref();
  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}


