#include <Inventor/ViewerComponents/Qt/QtHelper.h>
#include <Inventor/ViewerComponents/Qt/RenderAreaOrbiter.h>
#include <Inventor/ViewerComponents/nodes/SoViewingCube.h>
#include <Inventor/draggers/SoTabBoxDragger.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoFile.h>
#include <Inventor/nodes/SoFragmentShader.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoTransformSeparator.h>
#include <Inventor/nodes/SoTranslation.h>
#include <LDM/nodes/SoMultiDataSeparator.h>
#include <QApplication>
#include <QCheckBox>
#include <QDebug>
#include <QGroupBox>
#include <QLabel>
#include <QMainWindow>
#include <QRadioButton>
#include <QSlider>
#include <QVBoxLayout>
#include <QWidget>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeShader.h>
#include "VolumeViz/nodes/SoVolumeRenderingQuality.h"

// Forward declarations
SoVolumeData* makeVelocityVolume( const SbVec3i32& volDim );
SoSeparator* createBBox( const SbBox3f& bbox );
void updateTfOpacity( SoTransferFunction* tf, float opacity );
void manageVolumeRender( SoVolumeRender* vr, SoSwitch* sw, QCheckBox* chbAmplitude, QCheckBox* chbVelocity );

static const int32_t AMPLITUDE_ID = 1;
static const int32_t VELOCITY_ID = 2;

int
main( int argc, char** argv )
{
  QtHelper::addPlatformPluginsPath();

  QApplication app( argc, argv );
  SoVolumeRendering::init();

  //////////////// begin GUI

  QMainWindow* window = new QMainWindow;
  window->setWindowTitle( "Amplitude + Velocity" );
  window->resize( 949, 614 );
  window->show();

  QWidget* centralWidget = new QWidget( window );
  window->setCentralWidget( centralWidget );
  centralWidget->setLayout( new QVBoxLayout );
  centralWidget->layout()->setContentsMargins( 0, 0, 0, 0 );
  centralWidget->layout()->setSpacing( 0 );

  QWidget* topWidget = new QWidget( centralWidget );
  centralWidget->layout()->addWidget( topWidget );
  topWidget->setLayout( new QHBoxLayout );
  topWidget->layout()->setContentsMargins( 10, 10, 10, 10 );
  topWidget->setSizePolicy( QSizePolicy::Preferred, QSizePolicy::Maximum );

  //// begin velocity part

  QGroupBox* velocityGroup = new QGroupBox( topWidget );

  topWidget->layout()->addWidget( velocityGroup );
  velocityGroup->setTitle( "Velocity 150 x 150 x 150" );
  velocityGroup->setObjectName( "velocityGroup" );
  velocityGroup->setLayout( new QVBoxLayout );

  const QString velocityStyle = "QGroupBox#velocityGroup {\
          border-width:1px;\
          border-style:solid;\
          border-radius: 3px;\
          border-color: rgb(0,255,0);\
          margin-top: 6px;\
        }\
        QGroupBox#velocityGroup::title {\
        subcontrol-origin: margin;\
        left: 10px;\
        }\
  ";
  velocityGroup->setStyleSheet( velocityStyle );

  QWidget* velocitySliceWd = new QWidget;
  velocitySliceWd->setLayout( new QHBoxLayout );
  velocitySliceWd->layout()->setContentsMargins( 0, 0, 0, 0 );
  velocitySliceWd->layout()->setSpacing( 0 );

  QCheckBox* velocitySliceChb = new QCheckBox( velocitySliceWd );
  velocitySliceWd->layout()->addWidget( velocitySliceChb );
  velocitySliceChb->setText( "slice" );
  velocitySliceChb->setChecked( true );

  QSlider* velocitySliceSld = new QSlider( Qt::Orientation::Horizontal, velocitySliceWd );
  velocitySliceWd->layout()->addWidget( velocitySliceSld );
  velocitySliceSld->setRange( 0, 150 );
  velocitySliceSld->setValue( 32 );
  velocitySliceSld->setEnabled( true );

  // bind checkbox and slider
  QObject::connect( velocitySliceChb, &QCheckBox::stateChanged, velocitySliceSld, &QSlider::setEnabled );

  velocityGroup->layout()->addWidget( velocitySliceWd );

  QCheckBox* velocityVrChb = new QCheckBox( velocityGroup );
  velocityVrChb->setText( "volume rendering" );
  velocityVrChb->setChecked( false );

  velocityGroup->layout()->addWidget( velocityVrChb );

  //// end velocity part
  //// begin amplitude

  QGroupBox* amplitudeGroup = new QGroupBox( topWidget );

  topWidget->layout()->addWidget( amplitudeGroup );
  amplitudeGroup->setTitle( "Amplitude 205 x 361 x 58" );
  amplitudeGroup->setObjectName( "amplitudeGroup" );
  amplitudeGroup->setLayout( new QVBoxLayout );

  const QString amplitudeStyle = "QGroupBox#amplitudeGroup {\
          border-width:1px;\
          border-style:solid;\
          border-radius: 3px;\
          border-color: red;\
          margin-top: 6px;\
        }\
        QGroupBox#amplitudeGroup::title {\
        subcontrol-origin: margin;\
        left: 10px;\
        }\
  ";
  amplitudeGroup->setStyleSheet( amplitudeStyle );

  QWidget* amplitudeSliceWd = new QWidget;
  amplitudeSliceWd->setLayout( new QHBoxLayout );
  amplitudeSliceWd->layout()->setContentsMargins( 0, 0, 0, 0 );
  amplitudeSliceWd->layout()->setSpacing( 0 );

  QCheckBox* amplitudeSliceChb = new QCheckBox( amplitudeSliceWd );
  amplitudeSliceWd->layout()->addWidget( amplitudeSliceChb );
  amplitudeSliceChb->setText( "slice" );
  amplitudeSliceChb->setChecked( true );

  QSlider* amplitudeSliceSld = new QSlider( Qt::Orientation::Horizontal, amplitudeSliceWd );
  amplitudeSliceWd->layout()->addWidget( amplitudeSliceSld );
  amplitudeSliceSld->setRange( 0, 58 );
  amplitudeSliceSld->setValue( 32 );
  amplitudeSliceSld->setEnabled( true );

  // bind checkbox and slider
  QObject::connect( amplitudeSliceChb, &QCheckBox::stateChanged, amplitudeSliceSld, &QSlider::setEnabled );

  amplitudeGroup->layout()->addWidget( amplitudeSliceWd );

  QCheckBox* amplitudeVrChb = new QCheckBox( amplitudeGroup );
  amplitudeVrChb->setText( "volume rendering" );
  amplitudeVrChb->setChecked( false );

  amplitudeGroup->layout()->addWidget( amplitudeVrChb );


  //// end amplitude
  //// begin blend part

  QGroupBox* combineGroup = new QGroupBox( topWidget );
  topWidget->layout()->addWidget( combineGroup );
  combineGroup->setTitle( "Blend velocity / amplitude" );
  combineGroup->setLayout( new QVBoxLayout );

  QWidget* blendWd = new QWidget;
  blendWd->setLayout( new QHBoxLayout );
  blendWd->layout()->setContentsMargins( 0, 0, 0, 0 );
  blendWd->layout()->setSpacing( 0 );

  combineGroup->layout()->addWidget( blendWd );

  QLabel* level = new QLabel( "level %" );
  blendWd->layout()->addWidget( level );

  QSlider* combineSld = new QSlider( Qt::Orientation::Horizontal, blendWd );
  blendWd->layout()->addWidget( combineSld );
  combineSld->setRange( 0, 100 );
  combineSld->setValue( 50 );

  qobject_cast<QBoxLayout*>( topWidget->layout() )->addStretch();
  //// end blend part

  //////////////// end GUI
  //////////////// begin Scene Graph

  SoRef<SoSeparator> root = new SoSeparator;

  SoMultiDataSeparator* mds = new SoMultiDataSeparator;
  root->addChild( mds );

  SoTransformSeparator* tfs1 = new SoTransformSeparator;
  mds->addChild( tfs1 );

  SoVolumeData* volData1 = new SoVolumeData;
  tfs1->addChild( volData1 );
  volData1->fileName = "$OIVHOME/examples/data/VolumeViz/colt-float.ldm";
  volData1->dataSetId = AMPLITUDE_ID;
  volData1->extent.setValue( SbVec3f( -1.0f, -1.0f, -1.0f ), SbVec3f( 1.0f, 1.0f, 1.0f ) );

  SoTransferFunction* pTransFunc1 = new SoTransferFunction;
  tfs1->addChild( pTransFunc1 );
  pTransFunc1->predefColorMap = SoTransferFunction::INTENSITY;
  pTransFunc1->transferFunctionId.connectFrom( &volData1->dataSetId );

  SoTransformSeparator* tfs2 = new SoTransformSeparator;
  mds->addChild( tfs2 );

  SoTabBoxDragger* tbbDragger = new SoTabBoxDragger;
  SoBaseColor* baseColor = new SoBaseColor;
  baseColor->rgb = SbVec3f( 0.0f, 0.5f, 0.0f );
  baseColor->setOverride( true );
  SoDrawStyle* drawStyle = new SoDrawStyle;
  drawStyle->lineWidth = 2.0f;
  drawStyle->setOverride( true );
  // Dragger custom
  static_cast<SoSeparator*>( tbbDragger->getPart( "boxGeom", true ) )->insertChild( baseColor, 0 );
  static_cast<SoSeparator*>( tbbDragger->getPart( "boxGeom", true ) )->insertChild( drawStyle, 0 );

  tfs2->addChild( tbbDragger );

  SoTranslation* trans = new SoTranslation;
  tfs2->addChild( trans );

  SoScale* scale = new SoScale;
  tfs2->addChild( scale );

  SoVolumeData* volData2 = makeVelocityVolume( SbVec3i32( 150, 150, 150 ) );
  tfs2->addChild( volData2 );
  volData2->extent.setValue( SbVec3f( -1.0f, -1.0f, -1.0f ), SbVec3f( 1.0f, 1.0f, 1.0f ) );
  volData2->dataSetId = VELOCITY_ID;

  SoTransferFunction* pTransFunc2 = new SoTransferFunction;
  tfs2->addChild( pTransFunc2 );
  pTransFunc2->predefColorMap = SoTransferFunction::STANDARD;
  pTransFunc2->transferFunctionId.connectFrom( &volData2->dataSetId );

  SoFragmentShader* fragmentShader = new SoFragmentShader;
  fragmentShader->sourceProgram.setValue( "$OIVHOME/examples/source/VolumeViz/multiData/AmplitudeVelocity/AmplitudeVelocity_sliceFrag.glsl" );

  fragmentShader->addShaderParameter1i( "dataVelocity", VELOCITY_ID );
  fragmentShader->addShaderParameter1i( "dataAmplitude", AMPLITUDE_ID );
  SoShaderParameter1f* combinedPercentageShaderParam = fragmentShader->addShaderParameter1f( "combinedPercentage", 0.5f );

  SoVolumeRenderingQuality* volRenderQuality = new SoVolumeRenderingQuality;
  volRenderQuality->shaderObject.set1Value( SoVolumeShader::FRAGMENT_COMPUTE_COLOR, fragmentShader );
  mds->addChild( volRenderQuality );

  SoSeparator* contVol = new SoSeparator;
  mds->addChild( contVol );

  SoPickStyle* volumePickStyle = new SoPickStyle;
  volumePickStyle->style = SoPickStyle::UNPICKABLE;
  contVol->addChild( volumePickStyle );

  SoSwitch* volumeSwitch = new SoSwitch;
  contVol->addChild( volumeSwitch );
  volumeSwitch->whichChild = SO_SWITCH_NONE;
  SoVolumeRender* vRender = new SoVolumeRender;
  volumeSwitch->addChild( vRender );

  SoVolumeShader* volShader = new SoVolumeShader;
  volShader->shaderObject.set1Value( SoVolumeShader::FRAGMENT_COMPUTE_COLOR, fragmentShader );
  mds->addChild( volShader );

  SoSwitch* orthoAmplitudeSwitch = new SoSwitch;
  mds->addChild( orthoAmplitudeSwitch );
  orthoAmplitudeSwitch->whichChild = SO_SWITCH_ALL;
  SoOrthoSlice* orthoAmplitudeSlice = new SoOrthoSlice;
  orthoAmplitudeSwitch->addChild( orthoAmplitudeSlice );
  orthoAmplitudeSlice->sliceNumber = 32;
  orthoAmplitudeSlice->dataSetId = AMPLITUDE_ID;

  SoSwitch* orthoVelocitySwitch = new SoSwitch;
  mds->addChild( orthoVelocitySwitch );
  orthoVelocitySwitch->whichChild = SO_SWITCH_ALL;
  SoOrthoSlice* orthoVelocitySlice = new SoOrthoSlice;
  orthoVelocitySwitch->addChild( orthoVelocitySlice );
  orthoVelocitySlice->sliceNumber = 32;
  orthoVelocitySlice->dataSetId = VELOCITY_ID;


  root->addChild( createBBox( volData1->extent.getValue() ) );

  //////////////// end Scene Graph
  //////////////// bengin connections

  //// begin Dragger Connection

  trans->translation.connectFrom( &tbbDragger->translation );
  scale->scaleFactor.connectFrom( &tbbDragger->scaleFactor );
  tbbDragger->translation = SbVec3f( -0.5f, -0.5f, 0.0f );
  tbbDragger->scaleFactor = SbVec3f( 0.5f, 0.5f, 0.5f );

  //// end Dragger Connection
  //// begin GUI connection

  // velocity connect
  QObject::connect( velocitySliceChb, &QCheckBox::clicked, [orthoVelocitySwitch]( bool checked ) {
    checked ? orthoVelocitySwitch->whichChild = SO_SWITCH_ALL : orthoVelocitySwitch->whichChild = SO_SWITCH_NONE;
  } );

  QObject::connect( velocitySliceSld, &QSlider::valueChanged, [orthoVelocitySlice]( int value ) { orthoVelocitySlice->sliceNumber = value; } );

  QObject::connect( velocityVrChb, &QCheckBox::clicked, [vRender, volumeSwitch, amplitudeVrChb, velocityVrChb]() {
    manageVolumeRender( vRender, volumeSwitch, amplitudeVrChb, velocityVrChb );
  } );

  // amplitude connects
  QObject::connect( amplitudeSliceChb, &QCheckBox::clicked, [orthoAmplitudeSwitch]( bool checked ) {
    checked ? orthoAmplitudeSwitch->whichChild = SO_SWITCH_ALL : orthoAmplitudeSwitch->whichChild = SO_SWITCH_NONE;
  } );

  QObject::connect( amplitudeSliceSld, &QSlider::valueChanged, [orthoAmplitudeSlice]( int value ) { orthoAmplitudeSlice->sliceNumber = value; } );

  QObject::connect( amplitudeVrChb, &QCheckBox::clicked, [vRender, volumeSwitch, amplitudeVrChb, velocityVrChb]() {
    manageVolumeRender( vRender, volumeSwitch, amplitudeVrChb, velocityVrChb );
  } );

  // blend connects
  QObject::connect( combineSld, &QSlider::valueChanged, [combinedPercentageShaderParam]( int value ) {
    combinedPercentageShaderParam->value = ( float )value / 100;
  } );

  //// end GUI connection

  //////////////// end connections
  //////////////// begin Viewer

  RenderAreaOrbiter* renderArea = new RenderAreaOrbiter( window );
  const float grayVal = 0.99f;
  renderArea->setClearColor( SbColorRGBA( grayVal, grayVal, grayVal, 1.0f ) );
  centralWidget->layout()->addWidget( renderArea->getContainerWidget() );
  renderArea->setSceneGraph( root.ptr() );
  renderArea->viewAll( SbViewportRegion() );
  renderArea->setAntialiasingMode( SoSceneManager::AntialiasingMode::FSAA );
  renderArea->setTransparencyType( SoGLRenderAction::SORTED_PIXEL );
  renderArea->setAntialiasingQuality( 1 );

  updateTfOpacity( pTransFunc1, 0.5f );
  updateTfOpacity( pTransFunc2, 0.5f );

  //// custom viewing cube
  SceneOrbiter* orbiter = renderArea->getSceneInteractor();
  SoViewingCube* vCube = orbiter->getViewingCube();
  vCube->edgeStyle = SoViewingCube::CORNER;
  vCube->size = SbVec2f( 200.0f, 200.0f );
  vCube->opacityMin = 0.0f;
  vCube->opacityMax = 0.8f;
  SoFile* compass = new SoFile;
  compass->name = "$OIVHOME/examples/data/Inventor/ViewingCube/Compass/compass-northsouth-y.iv";
  vCube->compass = compass;
  vCube->facePosX = "$OIVHOME/examples/data/Inventor/ViewingCube/Faces/NSEW/xpos.png";
  vCube->faceNegX = "$OIVHOME/examples/data/Inventor/ViewingCube/Faces/NSEW/xneg.png";
  vCube->facePosY = "$OIVHOME/examples/data/Inventor/ViewingCube/Faces/NSEW/ypos.png";
  vCube->faceNegY = "$OIVHOME/examples/data/Inventor/ViewingCube/Faces/NSEW/yneg.png";
  vCube->facePosZ = "$OIVHOME/examples/data/Inventor/ViewingCube/Faces/NSEW/zpos.png";
  vCube->faceNegZ = "$OIVHOME/examples/data/Inventor/ViewingCube/Faces/NSEW/zneg.png";

  //////////////// end Viewer

  //// App Exec
  app.exec();

  //// Exit
  root = nullptr;
  delete window;
  SoVolumeRendering::finish();
  return 0;
}

//// Helper for Bbox creation
SoSeparator*
createBBox( const SbBox3f& bbox )
{
  SoSeparator* bboxSep = new SoSeparator;

  SoPickStyle* pickStyle = new SoPickStyle;
  bboxSep->addChild( pickStyle );
  pickStyle->style = SoPickStyle::Style::UNPICKABLE;

  SoDrawStyle* drawStyle = new SoDrawStyle;
  bboxSep->addChild( drawStyle );
  drawStyle->style = SoDrawStyle::Style::LINES;
  drawStyle->lineWidth = 1.5f;
  drawStyle->linePattern = 0x8888;

  SoBaseColor* bboxMat = new SoBaseColor;
  bboxMat->rgb = SbVec3f( 0.75f, 0.0f, 0.0f );
  bboxSep->addChild( bboxMat );

  SoLightModel* lightModel = new SoLightModel;
  bboxSep->addChild( lightModel );
  lightModel->model = SoLightModel::Model::BASE_COLOR;

  SoTranslation* translation = new SoTranslation;
  bboxSep->addChild( translation );
  translation->translation = bbox.getCenter();

  SoCube* cube = new SoCube;
  const SbVec3f bboxSize = bbox.getSize();
  cube->width = bboxSize[0];
  cube->height = bboxSize[1];
  cube->depth = bboxSize[2];
  bboxSep->addChild( cube );

  return bboxSep;
}

//// Make synthetic velocity volume (not realistic)
SoVolumeData*
makeVelocityVolume( const SbVec3i32& volDim )
{
  // Allocate memory for volume2 and assign values
  int testXdim = volDim[0];
  int testYdim = volDim[1];
  int testZdim = volDim[2];
  int numTestBytes = testXdim * testYdim * testZdim;
  unsigned char* testData = new unsigned char[numTestBytes];
  int counter = 0;
  int ymax = testYdim - 1;
  for ( int i = 0; i < testZdim; i++ )
  {
    for ( int j = 0; j < testYdim; j++ )
    {
      unsigned char value = ( float )j / ymax * 255;
      for ( int k = 0; k < testXdim; k++ )
      {
        float rad = ( float )k / ( testXdim - 1 );
        if ( rad > 0.75 )
          rad = 0.75;
        rad *= 2 * ( float )M_PI;
        float offset = ( float )-sin( rad );
        offset *= 30 * ( 1 - ( ( float )j / ymax ) );
        int val = value + offset - i / 2;
        if ( val < 1 )
          val = 1;
        else if ( val > 255 )
          val = 255;
        testData[counter++] = 255 - val;
      }
    }
  }

  // Create node and associate data with it.
  SoVolumeData* pVolData2 = new SoVolumeData;
  SoMemoryObject* memObj = new SoMemoryObject( testData, numTestBytes * sizeof( unsigned char ) );
  pVolData2->data.setValue( volDim, 8, memObj, SoSFArray::NO_COPY_AND_DELETE );
  return pVolData2;
}

void
updateTfOpacity( SoTransferFunction* tf, float opacity )
{
  opacity = 1.f - std::pow( 1 - opacity, 1.f / 16.f );
  SbVec4f* rgba = ( SbVec4f* )tf->actualColorMap.startEditing();
  for ( size_t i = 0; i < 256; i++ )
  {
    rgba[i][3] = opacity;
  }
  tf->actualColorMap.finishEditing();
}

void
manageVolumeRender( SoVolumeRender* vr, SoSwitch* sw, QCheckBox* chbAmplitude, QCheckBox* chbVelocity )
{
  vr->dataSetIds.deleteValues( 0, 2 );
  int k = 0;
  if ( chbAmplitude->isChecked() )
  {
    vr->dataSetIds.set1Value( k, 1 );
    k++;
  }
  if ( chbVelocity->isChecked() )
  {
    vr->dataSetIds.set1Value( k, 2 );
  }
  sw->whichChild = ( vr->dataSetIds.getNum() == 0 ) ? SO_SWITCH_NONE : SO_SWITCH_ALL;
}
