/*----------------------------------------------------------------------------------------
 * Example program.
 * Purpose : Demonstrate how to use CPU data combining feature
 * author : Jerome Hummel
 * date :   November 2005
 *
 * This program combines two volumes - data volume1 and "mask" volume2
 * The mask volume is used to selectively make voxels completely
 * transparent, effectively clipping volume1 at voxel precision.
 *
 * To achieve this effect we define the mask volume to contain only the
 * values zero and one.  We create an SoDataCompositor node (or custom
 * compositor) that combines the two volumes by multiplying the data
 * values.  In other words, for each voxel:
 *     result_voxel = volume1_voxel * volume2_voxel
 * Any voxel where volume2 contains zero will be forced to zero, while
 * all voxels where volume2 contains one will be unchanged.
 * Finally we make sure that value zero maps to completely transparent
 * in the transfer function.
 *
 * The program can be built two different ways:
 *
 * 1. If USE_CUSTOM_COMPOSITOR is defined, then a custom compositor
 *    class is derived from SoDataCompositor.  In this case the custom
 *    compositor implements one of the predefined algorithms so we can
 *    easily confirm that the result is correct.
 *
 * 2. Else a standard SoDataCompositor node is created and its
 *    preDefCompositor field is set to SoDataCompositor::MULTIPLY.
 *
 *----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoPhysicalMaterial.h>
#include <Inventor/SoPreferences.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <LDM/nodes/SoDataCompositor.h>
#include <LDM/nodes/SoMultiDataSeparator.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>

#ifdef WIN32
//#include <Inventor/SoWinApp.h>
#endif

const char* FILENAME = "$OIVHOME/examples/data/VolumeViz/3DHEAD.ldm";

// Comment out following line to use the predefined compositor node

#define USE_CUSTOM_COMPOSITOR 1

#if USE_CUSTOM_COMPOSITOR 
#include "CPUDataCompositor.h"
#endif

SoSeparator *makeVolBBox( SbBox3f volSize );


////////////////////////////////////////////////////////////////////////
// main function

int main(int argc, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  SoVolumeRendering::init();
#if USE_CUSTOM_COMPOSITOR 
  CPUDataCompositor::initClass();
  SbString envVar = SbString("OIV_USER_LIBS");
  SbString pathToCustomNode = "$OIVHOME/examples/bin/$OIVARCH/VolumeViz/CPUDataCompositor";
  SoPreferences::setString(envVar, pathToCustomNode);
#endif

  const char *filename = (argc >= 2 ? argv[1] : FILENAME);

  // Load volume1 then get size and dimension
  SoVolumeData* pVolData1 = new SoVolumeData;
  pVolData1->fileName     = filename;
  pVolData1->dataSetId    = 1;
  SbBox3f size            = pVolData1->extent.getValue();
  SbVec3i32 vol1dim       = pVolData1->data.getSize();

  // Compute values for volume2 (mask volume)
  int testXdim = vol1dim[0];
  int testYdim = vol1dim[1];
  int testZdim = vol1dim[2];
  int numTestBytes = testXdim * testYdim * testZdim;
  unsigned char *testData = new unsigned char[numTestBytes];
  int counter = 0;
  for (int k = 0; k < testZdim; k++){
    for (int j = 0; j < testYdim; j++){
      for (int i = 0; i < testXdim; i++){
        double phase = sin(4*3.1*i/testXdim) + 4*k*k/(testYdim*testYdim);//same for each given j
        float waveFactor = 23.;
        float t = char(127 + 127*sin(j/waveFactor + phase ));//now make it vary on j
        testData[counter++] = (t > -10 && t < 50) ? 0 : 1;
      }
    }
  }

  // Create volume2
  //
  // Note: setVolumeSize must be called after setVolumeData to be effective
  SoVolumeData* pVolData2 = new SoVolumeData();
  pVolData2->data.setValue(SbVec3i32(testXdim, testYdim, testZdim), SbDataType::UNSIGNED_BYTE,
                           0, testData, SoSFArray3D::NO_COPY);  
  pVolData2->dataSetId = 2;
  pVolData2->extent = size;
  pVolData2->ldmResourceParameters.getValue()->tileDimension.setValue( 64, 64, 64 );

  // Use a predefined colorMap with the SoTransferFunction
  SoTransferFunction* pTransFunc = new SoTransferFunction;
  pTransFunc->predefColorMap = SoTransferFunction::GLOW;
  pTransFunc->minValue = 40;
  pTransFunc->maxValue = 250;

  // Node in charge of drawing the volume
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
  pVolRender->numSlicesControl = SoVolumeRender::ALL;

  // CPU compositor:
  SoDataCompositor* dataCompositor = new SoDataCompositor;
  dataCompositor->dataType = SoDataCompositor::UNSIGNED_BYTE;
  dataCompositor->preDefCompositor = SoDataCompositor::MULTIPLY;

  // Property node which allows SoVolumeRender to draw lighted volumes
  SoVolumeRenderingQuality *pVRVolQuality = new SoVolumeRenderingQuality;
  pVRVolQuality->lighting = TRUE;
  pVRVolQuality->preIntegrated = TRUE;

  SoLightModel* lightModel = new SoLightModel;
  lightModel->model = SoLightModel::PHYSICALLY_BASED;

  SoPhysicalMaterial* material = new SoPhysicalMaterial;
  material->baseColor = SbColorRGBA(1.0f, 1.0f, 1.0f, 1.0f);
  material->specular = 1.0f;
  material->roughness = 0.3f;

  SoMultiDataSeparator *mds = new SoMultiDataSeparator;

  // Assemble the scene graph
  SoSeparator *root = new SoSeparator;
  root->ref();

  root->addChild( makeVolBBox( pVolData1->extent.getValue() ));

  root->addChild(new SoDirectionalLight); // additional light source
  root->addChild( lightModel );
  root->addChild( material );

  //Add the multidata separator
  root->addChild(mds);

  // Add the data compositor node
  // Either a predefined compositor or a custom compositor
#if USE_CUSTOM_COMPOSITOR
  CPUDataCompositor *composer = new CPUDataCompositor;
  // Enable this to experiment generation of RGBA dataset on the fly. 
  // composer->rgbaMode.setValue(TRUE);
  mds->addChild( composer );
#else
  mds->addChild( dataCompositor );
#endif

  mds->addChild( pVolData1 );
  mds->addChild( pVolData2 );
  mds->addChild( pTransFunc );
  mds->addChild( pVRVolQuality );
  mds->addChild( pVolRender );

  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setSceneGraph(root);
  myViewer->setTitle("Volume rendering");
  myViewer->setBackgroundColor(SbColor(.5,.5,.5));
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  root->unref();
  delete myViewer;
  delete [] testData;

#if USE_CUSTOM_COMPOSITOR 
  CPUDataCompositor::exitClass();
#endif

  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}

///////////////////////////////////////////////////////////////////////
//
// Create a wireframe box showing bounding box of volume

#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoIndexedLineSet.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoVertexProperty.h>

SoSeparator *makeVolBBox( SbBox3f volSize )
{
  // The box will be easier to see without lighting and with wide lines
  SoLightModel *pLModel = new SoLightModel;
  pLModel->model = SoLightModel::BASE_COLOR;

  SoDrawStyle *pStyle = new SoDrawStyle;
  pStyle->lineWidth = 2;

  // The box should be unpickable so manip can be used inside it
  SoPickStyle *pPickable = new SoPickStyle;
  pPickable->style = SoPickStyle::UNPICKABLE;

  // Create a cube with the geometric size of the volume
  float xmin, xmax, ymin, ymax, zmin, zmax;
  volSize.getBounds( xmin,ymin, zmin, xmax, ymax, zmax );
  SoVertexProperty *pProp = new SoVertexProperty;
  pProp->vertex.set1Value( 0, SbVec3f(xmin,ymin,zmin) );
  pProp->vertex.set1Value( 1, SbVec3f(xmax,ymin,zmin) );
  pProp->vertex.set1Value( 2, SbVec3f(xmax,ymax,zmin) );
  pProp->vertex.set1Value( 3, SbVec3f(xmin,ymax,zmin) );
  pProp->vertex.set1Value( 4, SbVec3f(xmin,ymin,zmax) );
  pProp->vertex.set1Value( 5, SbVec3f(xmax,ymin,zmax) );
  pProp->vertex.set1Value( 6, SbVec3f(xmax,ymax,zmax) );
  pProp->vertex.set1Value( 7, SbVec3f(xmin,ymax,zmax) );
  pProp->orderedRGBA.set1Value( 0, 0xFF0000FF );

  // Draw it with a line set
  int coordIndices[] = {0,1,2,3,0,-1,4,5,6,7,4,-1,
                        0,4,-1, 1,5,-1, 2,6,-1, 3,7};
  int numCoordIndices = sizeof(coordIndices)/sizeof(int);
  SoIndexedLineSet *pLines = new SoIndexedLineSet;
  pLines->vertexProperty = pProp;
  pLines->coordIndex.setValues( 0, numCoordIndices, coordIndices );

  // Assemble scene graph
  SoSeparator *pBoxSep = new SoSeparator;
  pBoxSep->addChild( pLModel );
  pBoxSep->addChild( pPickable );
  pBoxSep->addChild( pStyle );
  pBoxSep->addChild( pLines );
  return pBoxSep;
}


