/*----------------------------------------------------------------------------------------
 * Example program.
 * Purpose : Demonstrate how to use GPU combining feature
 * author : Jerome Hummel
 * date :22 November 2005
/*----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoPhysicalMaterial.h>
#include <Inventor/nodes/SoFragmentShader.h>

#include <Inventor/helpers/SbFileHelper.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <LDM/nodes/SoDataCompositor.h>
#include <LDM/nodes/SoMultiDataSeparator.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoVolumeShader.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>

// set to 1 tu use ldm file instead of generating in memory data (usefull
// when recording ART)
#define USE_LDM 0

#define FILENAME "$OIVHOME/examples/data/VolumeViz/3DHEAD.ldm"

// Comment out the following line in order to apply lighting and
// pre-integration to the combined data
//#define LIGHTING_AND_PREINTEGRATION

// Main function
int main(int argc, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  SoVolumeRendering::init();

  const char *filename = (argc >= 2 ? argv[1] : FILENAME);

  // Data 1
  SoVolumeData* pVolData1 = new SoVolumeData;
  pVolData1->fileName     = filename;
  SbBox3f size            = pVolData1->extent.getValue();
  SbVec3i32 vol1dim       = pVolData1->data.getSize();

  SoVolumeData* pVolData = new SoVolumeData();
#if USE_LDM
  pVolData->fileName = "$OIVHOME/examples/source/VolumeViz/multiData/GPUDataCompose/mask.ldm";
#else
  // Data 2: mask volume
  int testXdim = vol1dim[0];
  int testYdim = vol1dim[1];
  int testZdim = vol1dim[2];
  int numTestBytes = testXdim * testYdim * testZdim;
  unsigned char *testData = new unsigned char[numTestBytes];
  int counter = 0;
  for (int k = 0; k < testZdim; k++){
    for (int j = 0; j < testYdim; j++){
      for (int i = 0; i < testXdim; i++){
        double phase = sin(4*3.1*i/testXdim) + 4*k*k/(testYdim*testYdim);//same for each given j
        float waveFactor = 23.;
        float t = char(127 + 127*sin(j/waveFactor + phase ));//now make it vary on j
        testData[counter++] = (t > -10 && t < 50) ? 0 : 1;
      }
    }
  }
  SbVec3i32 tileDimension = pVolData1->ldmResourceParameters.getValue()->tileDimension.getValue();
  pVolData->ldmResourceParameters.getValue()->tileDimension.setValue(tileDimension);
  pVolData->data.setValue(SbVec3i32(testXdim, testYdim, testZdim), pVolData->getDataType(), 0, testData, SoSFArray::NO_COPY );
#endif
  pVolData->dataSetId = 2;
  pVolData->extent.setValue( size );

  // Use a predefined colorMap with the SoTransferFunction
  SoTransferFunction* pTransFunc = new SoTransferFunction;
  pTransFunc->predefColorMap = SoTransferFunction::GLOW;
  pTransFunc->minValue = 40;
  pTransFunc->maxValue = 250;
  pTransFunc->transferFunctionId = 0;

  SoTransferFunction* pTransFunc1 = new SoTransferFunction;
  pTransFunc1->predefColorMap = SoTransferFunction::SEISMIC;
  pTransFunc1->transferFunctionId = 1;

  // Initialize and set the volume shader program
  SoVolumeRenderingQuality* pVolShader = new SoVolumeRenderingQuality;

  // Specify how to combine data with each other
  SoFragmentShader* fragmentShader = new SoFragmentShader;

  // Data1 in texture unit 1
  fragmentShader->addShaderParameter1i("data1",1);
  // Data2 in texture unit 2
  fragmentShader->addShaderParameter1i("data2",2);

  // Specify to what part the shader prog is intended to. Here fragments only.
#ifdef LIGHTING_AND_PREINTEGRATION
  fragmentShader->sourceProgram.setValue("$OIVHOME/examples/source/VolumeViz/multiData/GPUDataCompose/combine_frag.glsl");
  pVolShader->shaderObject.set1Value(SoVolumeShader::DATA_COMBINE_FUNCTION, fragmentShader);
  pVolShader->preIntegrated=TRUE;
  pVolShader->lighting=TRUE;

#else
  // TF in texture unit 0
  fragmentShader->sourceProgram.setValue("$OIVHOME/examples/source/VolumeViz/multiData/GPUDataCompose/volume_frag.glsl");
  pVolShader->shaderObject.set1Value(SoVolumeShader::FRAGMENT_COMPUTE_COLOR, fragmentShader);
#endif

  // Node in charge of drawing the volume
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->numSlicesControl = SoVolumeRender::ALL;
  pVolRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;

  SoMultiDataSeparator *mds = new SoMultiDataSeparator;

  SoLightModel* lightModel = new SoLightModel;
  lightModel->model = SoLightModel::PHYSICALLY_BASED;

  SoPhysicalMaterial* material = new SoPhysicalMaterial;
  material->baseColor = SbColorRGBA(1.0f, 1.0f, 1.0f, 1.0f);
  material->specular = 1.0f;
  material->roughness = 0.3f;

  // Assemble the scene graph
  SoSeparator *root = new SoSeparator;
  root->ref();

  //Add the multidata separator
  root->addChild(mds);

  mds->addChild( lightModel );
  mds->addChild( material );
  mds->addChild( pVolData );
  mds->addChild( pVolData1 );
  mds->addChild( pTransFunc );
#ifndef LIGHTING_AND_PREINTEGRATION
  mds->addChild( pTransFunc1 );
#endif
  mds->addChild( pVolShader );
  mds->addChild( pVolRender );

  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setSceneGraph(root);
  myViewer->setTitle("Volume rendering");
  myViewer->setBackgroundColor(SbColor(.5,.5,.5));
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete myViewer;
  root->unref();

  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}


