/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Federico Gamba (Feb 2008)
**=======================================================================*/
/*----------------------------------------------------------------------------------------
 * Example program.
 * Purpose: This program shows how to combine multiple volumes using the GPU.
 *          Specifically we will R,G,B components from 3 different volumes, and
 *          then compose them to get the final image.
 *--------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <Inventor/nodes/SoFragmentShader.h>
#include <VolumeViz/nodes/SoVolumeShader.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <LDM/nodes/SoMultiDataSeparator.h>
#include <Inventor/devices/SoCpuBufferObject.h>
#include <Inventor/nodes/SoTexture2.h>

#define RGB_GF 1

// set to 1 to use ldm file instead of generating in memory data (usefull
// when recording ART)
#define USE_LDM 0

//main function
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  // Initialize of VolumeViz extension
  SoVolumeRendering::init();

  SoRef<SoTexture2> myTexture2 = new SoTexture2;
  myTexture2->setSynchronizable(false);
  myTexture2->filename = "$OIVHOME/examples/source/VolumeViz/multiData/LDM2D_RGB/img.jpg";
  myTexture2->loadTexture();

  SbVec2s ImgSize( 0, 0 );
  int ImgNC = 0;
  const unsigned char* textureBuffer = myTexture2->image.getValue( ImgSize, ImgNC );

  SoRef<SoCpuBufferObject> cpuBufferR = new SoCpuBufferObject;
  SoRef<SoCpuBufferObject> cpuBufferG = new SoCpuBufferObject;
  SoRef<SoCpuBufferObject> cpuBufferB = new SoCpuBufferObject;

  //Volume Size
  const int dimX = ImgSize[0];
  const int dimY = ImgSize[1];

  int NumBytes = dimX * dimY;

  cpuBufferR->setSize(NumBytes);
  cpuBufferG->setSize(NumBytes);
  cpuBufferB->setSize(NumBytes);

  unsigned char* bufferR  = (unsigned char*)cpuBufferR->map( SoBufferObject::SET );
  unsigned char* bufferG  = (unsigned char*)cpuBufferG->map( SoBufferObject::SET );
  unsigned char* bufferB  = (unsigned char*)cpuBufferB->map( SoBufferObject::SET );

  // Fill each buffer with R,G and B components.
  for ( int i = 0; i < NumBytes; i++ )
  {
    bufferR[i] = textureBuffer[i*ImgNC];
    bufferG[i] = textureBuffer[i*ImgNC + 1];
    bufferB[i] = textureBuffer[i*ImgNC + 2];
  }

  cpuBufferR->unmap();
  cpuBufferG->unmap();
  cpuBufferB->unmap();

  SbVec3i32 dims( dimX, dimY, 1 );
  SbBox3f box( 0.0f, 0.0f, 0.0f, (float)dimX, (float)dimY, 1.0f );

  // Node to hold the volume data
  // In the first call we specify the voxel dimensions and the actual data.
  // In the second call we specify the 3D extent of the volume in modeling coords.
  SoVolumeData* pVolDataR = new SoVolumeData();
  SoVolumeData* pVolDataG = new SoVolumeData();
  SoVolumeData* pVolDataB = new SoVolumeData();
#if USE_LDM
  pVolDataR->fileName = "$OIVHOME/examples/source/VolumeViz/multiData/LDM2D_RGB/img1.ldm";
  pVolDataG->fileName = "$OIVHOME/examples/source/VolumeViz/multiData/LDM2D_RGB/img2.ldm";
  pVolDataB->fileName = "$OIVHOME/examples/source/VolumeViz/multiData/LDM2D_RGB/img3.ldm";
#else
  pVolDataR->data.setValue( dims, SbDataType::UNSIGNED_BYTE, 8, bufferR, SoSFArray::NO_COPY );
  pVolDataR->extent.setValue( box );

  pVolDataG->data.setValue( dims, SbDataType::UNSIGNED_BYTE, 8, bufferG, SoSFArray::NO_COPY );
  pVolDataG->extent.setValue( box );

  pVolDataB->data.setValue( dims, SbDataType::UNSIGNED_BYTE, 8, bufferB, SoSFArray::NO_COPY );
  pVolDataB->extent.setValue( box );
#endif

  pVolDataR->dataSetId = 1;
  pVolDataG->dataSetId = 2;
  pVolDataB->dataSetId = 3;

  pVolDataR->ldmResourceParameters.getValue()->tileDimension.setValue( 256, 256, 1 );
  pVolDataG->ldmResourceParameters.getValue()->tileDimension.setValue( 256, 256, 1 );
  pVolDataB->ldmResourceParameters.getValue()->tileDimension.setValue( 256, 256, 1 );

  // Use a predefined colorMap with the SoTransferFunction
  SoTransferFunction* pTransFunc = new SoTransferFunction;
  pTransFunc->predefColorMap = SoTransferFunction::INTENSITY;

  // Node in charge of drawing the volume
  SoOrthoSlice* pOrthoSlice = new SoOrthoSlice;

  // Specify how to combine data with each other
  SoFragmentShader* fragmentShader = new SoFragmentShader;

#if RGB_GF
  fragmentShader->sourceProgram.setValue( "$OIVHOME/examples/source/VolumeViz/multiData/LDM2D_RGB/RGB.glsl" );
#else
  fragmentShader->sourceProgram.setValue( "$OIVHOME/examples/source/VolumeViz/multiData/LDM2D_RGB/RG.glsl" );
#endif

  // Create uniform parameters allowing shader to access textures
  // Add dataSet as shader parameter with there corresponding Id.
  fragmentShader->addShaderParameter<SoShaderParameter1i>( "data1", 1 );
  fragmentShader->addShaderParameter<SoShaderParameter1i>( "data2", 2 );
#if RGB_GF
  fragmentShader->addShaderParameter<SoShaderParameter1i>( "data3", 3 );
#endif

  // Initialize and set the volume shader program
  SoVolumeShader* pVolShader = new SoVolumeShader();
  // Specify to what part the shader prog is intended to. Here fragments only.
  pVolShader->shaderObject.set1Value(SoVolumeShader::FRAGMENT_COMPUTE_COLOR, fragmentShader);
  // Specify that shader is only applied to slice or skin
  pVolShader->forVolumeOnly = FALSE;


  // Assemble the scene graph
  // Note: SoVolumeRender must appear after the SoVolumeData node.
  SoSeparator *root = new SoSeparator;
  root->ref();

  SoMultiDataSeparator *mds = new SoMultiDataSeparator;

  mds->addChild( pVolShader );

  mds->addChild( pVolDataR );
  mds->addChild( pVolDataG );
#if RGB_GF
  mds->addChild( pVolDataB );
#endif
  mds->addChild( pTransFunc );
  mds->addChild( pOrthoSlice );

  root->addChild(mds);
  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setSceneGraph(root);
  myViewer->setTitle("Volume rendering");
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  root->unref();
  delete myViewer;
  myTexture2 = NULL;
  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}
