/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Julien Sallane (feb 2014)
**=======================================================================*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoFragmentShader.h>
#include <Inventor/elements/SoViewVolumeElement.h>
#include <Inventor/actions/SoHandleEventAction.h>
#include <Inventor/misc/SbExtrusionGenerator.h>
#include <Inventor/drawers/SoPolyLineScreenDrawer.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeShader.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/helpers/SbDataTypeMacros.h>

#include "utils.h"

#define DEMO_ROOT_DIR "$OIVHOME/examples/source/VolumeViz/multiTransferFunctions"

SoSeparator *g_sceneGraph = NULL;
int g_currentLabel = 0;

// This function will be called each time a line created by the line drawer will be finalized.
// It retrieve points of the line in camera space, extrude it and write it in dataSet.
void
lineDrawerFinalizeCallback( SoPolyLineScreenDrawer::EventArg& eventArg )
{
  SoPolyLineScreenDrawer* lineDrawer = eventArg.getSource();
  SoHandleEventAction* action = eventArg.getAction();

  // If less than 1 point, shape cannot be generated.
  if ( lineDrawer->point.getNum() < 1 )
  {
    lineDrawer->clear();
    return;
  }

  // retrieve points of line in cam space
  std::vector<SbVec2f> lineInCam( lineDrawer->point.getNum() );
  for ( unsigned int i = 0; i < lineInCam.size(); ++i )
    lineInCam[i].setValue( lineDrawer->point[i][0], lineDrawer->point[i][1] );

  SoVolumeData* vdLabel = searchName<SoVolumeData>(g_sceneGraph, "VD_LABEL");

  // create a new extruded shape :
  // retrieve path of dataset
  SoSearchAction searchAction;
  searchAction.setNode( vdLabel );
  searchAction.apply( g_sceneGraph );
  SoPath* pathToExtrudedShapeRoot = searchAction.getPath();

  // retrieve bbox of volumeData node.
  SbBox3f bbox = getBbox( vdLabel );

  // create an extruded shape from specified line. Line is extruded along view
  // direction between bounding box enclosing planes.
  SoShape* extrudedShape = SbExtrusionGenerator::createFrom2DPoints( lineInCam,
                                                                     pathToExtrudedShapeRoot,
                                                                     SoViewVolumeElement::get(action->getState()),
                                                                     bbox );
  if ( extrudedShape == NULL )
  {
    lineDrawer->clear();
    return;
  }

  // write shape to dataset
  extrudedShape->ref();
  int editionId = 0;
  vdLabel->startEditing( editionId );
  vdLabel->editSolidShape( extrudedShape, (double)g_currentLabel );
  vdLabel->finishEditing( editionId );

  extrudedShape->unref();

  lineDrawer->clear();
  action->setHandled();
}

// Setup the volume shader node
void setupShader()
{
  // Load fragment shader program
  SoFragmentShader* fragmentShader = new SoFragmentShader;
  fragmentShader->sourceProgram.setValue( SbString(DEMO_ROOT_DIR) + "/fragmentShader.glsl" );

  // Create uniform parameters allowing shader to access textures
  // Set Label texture unit
  SoVolumeData* vdLabel = searchName<SoVolumeData>(g_sceneGraph, "VD_LABEL");
  SoShaderParameter1i *paramTexVdLabel = new SoShaderParameter1i;
  paramTexVdLabel->name = "volumeLabel";
  paramTexVdLabel->value.connectFrom(&vdLabel->dataSetId);

  // Set Data texture unit
  SoVolumeData* vdData = searchName<SoVolumeData>(g_sceneGraph, "VD_DATA");
  SoShaderParameter1i *paramTexVdData = new SoShaderParameter1i;
  paramTexVdData->name = "volumeData";
  paramTexVdData->value.connectFrom(&vdData->dataSetId);

  fragmentShader->parameter.set1Value(0, paramTexVdLabel);
  fragmentShader->parameter.set1Value(1, paramTexVdData);

  // Initialize and set the volume shader program
  SoVolumeRenderingQuality* vrqShader = searchNode<SoVolumeRenderingQuality>(g_sceneGraph);
  // Specify to what part the shader prog is intended to. Here fragments color only.
  vrqShader->shaderObject.set1Value(SoVolumeShader::FRAGMENT_COMPUTE_COLOR, fragmentShader);
}

// Fill VolumeData vdLabel with the value val using vdData parameters.
template <typename T>
void fillLabelsVolumeData(SoVolumeData* vdLabel, SoVolumeData* vdData, T val)
{
  const SbVec3i32 dataDim = vdData->data.getSize();
  int dataSize = dataDim[0] * dataDim[1] * dataDim[2] * ((SbDataType)vdData->data.getDataType()).getSize();
  std::vector<T> data(dataSize);
  for(int i = 0; i < dataSize; ++i)
    data[i] = (T)val;

  vdLabel->data.setValue(dataDim, vdData->data.getDataType(), vdData->data.getNumSigBits(), &data[0], SoSFArray3D::COPY);
}

// Class used to interpret event from the GUI.
class auditorClass : public SoDialogAuditor
{
  void dialogComboBox(SoDialogComboBox* cpt)
  {
    if (cpt->auditorID.getValue() == "transferFunction")
      g_currentLabel = cpt->selectedItem.getValue();
  }

  void dialogIntegerSlider(SoDialogIntegerSlider* cpt)
  {
    std::vector<SoTransferFunction*> transferFunctions = searchNodes<SoTransferFunction>(g_sceneGraph);
    if (cpt->auditorID.getValue() == "minValue")
      transferFunctions[g_currentLabel]->minValue = cpt->value.getValue();
    else if (cpt->auditorID.getValue() == "maxValue")
      transferFunctions[g_currentLabel]->maxValue = cpt->value.getValue();
  }
};

////////////////////////////////////////////////////////////////////////
//main function
int main(int, char **argv)
{
  // Create the window
  Widget window = SoXt::init(argv[0]);
  if (!window) return 0;

  // DialogViz functions used to build the GUI.
  ////////////////////////////////////////////
  SoDialogViz::init();
  SoTopLevelDialog* topLevelDialog = (SoTopLevelDialog*) SoDialogViz::loadFromFile( SbString(DEMO_ROOT_DIR) + "/interface.iv");
  topLevelDialog->ref();
  topLevelDialog->buildDialog(window, TRUE);
  SoDialogCustom* customNode = (SoDialogCustom*)topLevelDialog->searchForAuditorId("viewer");
  Widget myWindow = customNode->getWidget();
  auditorClass* auditor = new auditorClass;
  topLevelDialog->label.setValue("Multi transfer functions");
  topLevelDialog->addAuditor(auditor);
  topLevelDialog->show();
  ////////////////////////////////////////////
  // End of DialogViz function

  // Initialize of VolumeViz extension
  SoVolumeRendering::init();

  g_sceneGraph = readFile( (SbString(DEMO_ROOT_DIR) + "/scene.iv").toLatin1());
  if( g_sceneGraph == NULL )
    return 0;

  g_sceneGraph->ref();

  // Assemble the scene graph
  // Note: SoVolumeRender must appear after the SoVolumeData node.
  SoPolyLineScreenDrawer* lineDrawer = searchNode<SoPolyLineScreenDrawer>( g_sceneGraph );
  SoVolumeData* vdLabel = searchName<SoVolumeData>(g_sceneGraph, "VD_LABEL");
  SoVolumeData* vdData = searchName<SoVolumeData>(g_sceneGraph, "VD_DATA");

  if ( vdLabel == NULL || vdData == NULL || lineDrawer == NULL )
    return 0;

  // Set fragment shader and corresponding uniform values.
  setupShader();

  // The callback that, after the line is drawn, retrieve points and extract label subVolume.
  lineDrawer->onFinish.add( lineDrawerFinalizeCallback );

  // Init label data
  SB_DATATYPE_CALL(fillLabelsVolumeData,( vdLabel, vdData, 0 ), (SbDataType::DataType)vdData->data.getDataType());
  vdLabel->ldmResourceParameters.getValue()->tileDimension.setValue(vdData->ldmResourceParameters.getValue()->tileDimension.getValue());
  vdLabel->extent = vdData->extent.getValue();

  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setSceneGraph(g_sceneGraph);
  myViewer->setTitle("Multi transfer functions");
  myViewer->viewAll();

  myViewer->show();

  SoXt::show(window);
  SoXt::mainLoop();
  delete auditor;
  topLevelDialog->hide();
  topLevelDialog->unref();
  g_sceneGraph->unref();
  delete myViewer;

  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();
  return 0;
}
