/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

/*
 This demo requires a high-end GPU in order to run properly. Otherwise, the execution of a shader
 may take too much time, and an OS timeout may occur (Timeout Detection and Recovery (TDR))
 If you find yourself in this situation, setting these environment variables can help :
 - IVVR_RAYCAST_TILING_SUBDIVISION 8
 - IVVR_RAYCAST_MAXSTEP_THRESHOLD 100
 */

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/SoOffscreenRenderArea.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoEnvironmentMap.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPhysicalMaterial.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoShadowGroup.h>
#include <Inventor/helpers/SbFileHelper.h>

#include <VolumeViz/nodes/SoVolumeRendering.h>

#include <DialogViz/auditors/SoDialogAuditor.h>
#include <DialogViz/dialog/SoDialogCheckBox.h>
#include <DialogViz/dialog/SoDialogComboBox.h>
#include <DialogViz/dialog/SoDialogCustom.h>
#include <DialogViz/dialog/SoDialogPushButton.h>
#include <DialogViz/dialog/SoDialogRealSlider.h>
#include <DialogViz/dialog/SoDialogEditText.h>
#include <DialogViz/dialog/SoTopLevelDialog.h>

SoPerspectiveCamera* g_cam = NULL;
SoXtExaminerViewer* g_myViewer = NULL;
SoSwitch* g_lightModelSwitch = NULL;
SoEnvironmentMap* g_envMap = NULL;
SoMaterial* g_material = NULL;
SoPhysicalMaterial* g_physicalMaterial = NULL;
SoShadowGroup* g_shadowGroup = NULL;

SoDialogRealSlider* g_roughness = NULL;
SoDialogRealSlider* g_shadowsOpacity = NULL;
SoDialogRealSlider* g_shadowsQuality = NULL;
SoDialogRealSlider* g_focalDistance = NULL;
SoDialogRealSlider* g_snapshotScale = NULL;
SoDialogRealSlider* g_envMapIntensity = NULL;

SoDialogEditText* g_snapshotPath = NULL;

SoDialogComboBox* g_envMapCombo = NULL;


// Class used to interpret event from the GUI.
class auditorClass : public SoDialogAuditor
{
  void
  dialogCheckBox( SoDialogCheckBox* cpt )
  {
    const SbBool value = cpt->state.getValue();
    if ( cpt->auditorID.getValue() == "enableShadow" )
    {
      g_shadowGroup->isActive = value;
      g_shadowsOpacity->enable = value;
      g_shadowsQuality->enable = value;
    }
    else if ( cpt->auditorID.getValue() == "tonemapping" )
    {
      g_cam->exposureMode = value ? SoCamera::HDR_NEUTRAL : SoCamera::LDR_LINEAR;
    }
  }

  void
  dialogRealSlider( SoDialogRealSlider* cpt )
  {
    if ( cpt->auditorID.getValue() == "roughness" )
    {
      if ( g_lightModelSwitch->whichChild.getValue() == 0 )
        g_material->shininess = 0.05f * (-1.0f + 1.0f / std::max(cpt->value.getValue(), 0.001f));
    }
  }

  void
  dialogPushButton( SoDialogPushButton* cpt )
  {
    if ( cpt->auditorID.getValue() == "snapshot" )
    {
      const float scale = g_snapshotScale->value.getValue();
      const SbString& filename = g_snapshotPath->editText.getValue();

      if ( scale == 1.0f )
      {
        g_myViewer->saveSnapshot( filename );
      }
      else
      {
        // Make image same size and shape as viewer image
        SbViewportRegion vpRegion = g_myViewer->getViewportRegion();
        vpRegion.setWindowSize( short( scale * vpRegion.getWindowSize_i32()[0] ), short( scale * vpRegion.getWindowSize_i32()[1] ) );
        SoRef<SoOffscreenRenderArea> offscreenRenderArea = new SoOffscreenRenderArea(g_myViewer->getNormalSoContext());
        offscreenRenderArea->setSize( vpRegion.getWindowSize_i32() );

        // Transfer viewer settings to renderer
        SbColor bkgRGB = g_myViewer->getBackgroundColor();
        offscreenRenderArea->setClearColor( SbColorRGBA(bkgRGB[0], bkgRGB[1], bkgRGB[2], 1.f) );
        offscreenRenderArea->setTransparencyType( g_myViewer->getTransparencyType() );
        offscreenRenderArea->setTile( SbVec2i32(1024, 1024), 2 );
        offscreenRenderArea->setSceneGraph( g_myViewer->getSceneManager()->getSceneGraph() );

        // Render scene (including camera and headlight)
        offscreenRenderArea->renderToFile( filename );
      }
    }
  }

  void
  dialogComboBox( SoDialogComboBox* cpt )
  {
    const int32_t value = cpt->selectedItem.getValue();
    if ( cpt->auditorID.getValue() == "shadingstyle" )
    {
      g_lightModelSwitch->whichChild = value;
      g_envMapCombo->enable = (value == 1);
      g_envMapIntensity->enable = (value == 1);
    }
    else if ( cpt->auditorID.getValue() == "envmap" )
    {
      static const SbString filenames[] = {
        "$OIVHOME/data/textures/cubemaps/sky/skyHDR.hdr",
        "$OIVHOME/data/textures/cubemaps/rnl/rnl_cross.hdr",
        "$OIVHOME/data/textures/cubemaps/stpeters/stpeters_cross.hdr",
        "$OIVHOME/data/textures/cubemaps/grace/grace_cross.hdr",
        "$OIVHOME/data/textures/cubemaps/uffizi/uffizi.hdr",
        "$OIVHOME/data/textures/cubemaps/tenerife/posx.png",
        "$OIVHOME/data/textures/cubemaps/tenerife/posy.png",
        "$OIVHOME/data/textures/cubemaps/tenerife/posz.png",
        "$OIVHOME/data/textures/cubemaps/tenerife/negx.png",
        "$OIVHOME/data/textures/cubemaps/tenerife/negy.png",
        "$OIVHOME/data/textures/cubemaps/tenerife/negz.png",
      };
      static const int numFilenames[] = { 1, 1, 1, 1, 1, 6 };
      g_envMap->filenames.setNum(0);
      if (value < 6)
        g_envMap->filenames.setValues(0, numFilenames[value], filenames + value);
    }
  }
};

// Updates the focalDistance in GUI when the camera moves
void focalSensorCB( void*, SoSensor* sensor )
{
  const float focalDistance = g_cam->focalDistance.getValue();
  if ( focalDistance != g_focalDistance->value.getValue() )
  {
    SoFieldSensor* thisSensor = static_cast<SoFieldSensor*>(sensor);
    thisSensor->detach(); // Avoid re-triggering the sensor
    g_focalDistance->max.setValue( 2.0f * focalDistance );
    g_focalDistance->value.setValue( focalDistance );
    thisSensor->attach( &g_cam->focalDistance );
  }
}

SoSeparator*
readFile( const char* filename )
{
  // Open the input file
  SoInput mySceneInput;
  if ( !mySceneInput.openFile( filename ) )
  {
    std::cerr << "Cannot open file " << filename << std::endl;
    return NULL;
  }

  // Read the whole file into the database
  SoSeparator* myGraph = SoDB::readAll( &mySceneInput );
  if ( myGraph == NULL )
  {
    std::cerr << "Problem reading file" << std::endl;
    return NULL;
  }

  mySceneInput.closeFile();
  return myGraph;
}

////////////////////////////////////////////////////////////////////////
// main function
int
main( int, char** argv )
{
  // Create the window
  Widget window = SoXt::init( argv[0] );
  if ( !window )
    return 0;

  // DialogViz functions used to build the GUI.
  ////////////////////////////////////////////
  SoDialogViz::init();
  SoTopLevelDialog* topLevelDialog =
    (SoTopLevelDialog*) SoDialogViz::loadFromFile( "$OIVHOME/examples/source/VolumeViz/physicalRendering/physicalRendering_interface.iv" );
  topLevelDialog->ref();
  topLevelDialog->buildDialog( window, TRUE );
  SoDialogCustom* customNode = (SoDialogCustom*) topLevelDialog->searchForAuditorId( "viewer" );
  Widget myWindow = customNode->getWidget();
  topLevelDialog->label.setValue( "Advanced Volume Effects" );
  topLevelDialog->show();
  ////////////////////////////////////////////
  // End of DialogViz function

  SoPreferences::setFloat("IVVR_ALPHA_THRESHOLD_INTERACTIVE", 0.99f);

  // Initialize of VolumeViz extension
  SoVolumeRendering::init();

  SoSeparator* root = readFile( "$OIVHOME/examples/source/VolumeViz/physicalRendering/physicalRendering_scene.iv" );
  if ( root == NULL )
    return 0;

  root->ref();

  g_cam = dynamic_cast<SoPerspectiveCamera*>(SoNode::getByName(SbName("CAMERA")));

  g_shadowGroup = dynamic_cast<SoShadowGroup*>(SoNode::getByName(SbName("SHADOW_GROUP")));

  // Initialize and set the volume shader program
  g_lightModelSwitch = dynamic_cast<SoSwitch*>( SoNode::getByName(SbName("LIGHT_MODEL_SWITCH")) );
  g_envMap = dynamic_cast<SoEnvironmentMap*>( SoNode::getByName(SbName("ENVIRONMENT_MAP")) );
  g_material = dynamic_cast<SoMaterial*>( SoNode::getByName(SbName("MATERIAL")) );
  g_physicalMaterial = dynamic_cast<SoPhysicalMaterial*>( SoNode::getByName(SbName("PHYSICAL_MATERIAL")) );

  g_roughness = dynamic_cast<SoDialogRealSlider*>( topLevelDialog->searchForAuditorId( "roughness" ) );
  g_shadowsOpacity = dynamic_cast<SoDialogRealSlider*>( topLevelDialog->searchForAuditorId( "shadowsOpacity" ) );
  g_shadowsQuality = dynamic_cast<SoDialogRealSlider*>( topLevelDialog->searchForAuditorId( "shadowsQuality" ) );
  g_focalDistance = dynamic_cast<SoDialogRealSlider*>( topLevelDialog->searchForAuditorId( "focalDistance" ) );
  g_envMapCombo = dynamic_cast<SoDialogComboBox*>( topLevelDialog->searchForAuditorId( "envmap" ) );
  g_envMapIntensity = dynamic_cast<SoDialogRealSlider*>( topLevelDialog->searchForAuditorId("envmapIntensity") );
  g_snapshotScale = dynamic_cast<SoDialogRealSlider*>( topLevelDialog->searchForAuditorId( "snapshotScale" ) );
  g_snapshotPath = dynamic_cast<SoDialogEditText*>( topLevelDialog->searchForAuditorId( "snapshotPath" ) );

  SoDialogRealSlider* blurFactor = dynamic_cast<SoDialogRealSlider*>(topLevelDialog->searchForAuditorId("blurFactor"));
  SoDialogRealSlider* roughnessSlider = dynamic_cast<SoDialogRealSlider*>(topLevelDialog->searchForAuditorId("roughness"));

  g_shadowGroup->intensity.connectFrom( &g_shadowsOpacity->value );
  g_shadowGroup->quality.connectFrom( &g_shadowsQuality->value );
  g_cam->focalDistance.connectFrom( &g_focalDistance->value );
  g_envMap->intensity.connectFrom( &g_envMapIntensity->value );
  g_cam->blur.connectFrom( &blurFactor->value );
  g_physicalMaterial->roughness.connectFrom( &roughnessSlider->value );

  SoFieldSensor* focalSensor = new SoFieldSensor;
  focalSensor->setFunction( focalSensorCB );
  focalSensor->attach( &g_cam->focalDistance );

  // Initialize focal distance value in GUI
  focalSensorCB( NULL, focalSensor );

  auditorClass* auditor = new auditorClass;
  topLevelDialog->addAuditor( new auditorClass );

  // Set up viewer:
  g_myViewer = new SoXtExaminerViewer( myWindow );
  g_myViewer->setHeadlight( false );
  g_myViewer->setTransparencyType( SoGLRenderAction::NO_SORT );
  g_myViewer->setSceneGraph( root );
  g_myViewer->setTitle( "Physical Rendering" );

  g_myViewer->show();

  SoXt::show( window );
  SoXt::mainLoop();

  delete auditor;
  delete g_myViewer;
  delete focalSensor;

  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();
  return 0;
}
