#include <QApplication>
#include <QBoxLayout>
#include <QMainWindow>
#include <QWidget>
#include <Inventor/ViewerComponents/Qt/QtHelper.h>
#include <Inventor/ViewerComponents/Qt/RenderAreaOrbiter.h>
#include <Inventor/ViewerComponents/nodes/SoViewingCube.h>
#include <Inventor/nodes/SoVertexProperty.h>
#include <Inventor/nodes/SoIndexedLineSet.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoBBox.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoText2.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/nodes/SoEventCallback.h>

SoVolumeRender* m_volumeRender = nullptr;
int m_volumeRenderMode = 0;
SoText2* m_helpText = nullptr;


///////////////////////////////////////////////////////////////////////////////
// Convenience function to draw a specified bounding box as a wireframe.
SoSeparator*
createBoundingBox( const SbBox3f& bbox, const SbColor& color )
{
  auto bboxSep = new SoSeparator;
  // Create a cube with the geometric size of the volume
  float xmin, xmax, ymin, ymax, zmin, zmax;
  bbox.getBounds( xmin, ymin, zmin, xmax, ymax, zmax );
  auto vProp = new SoVertexProperty;
  vProp->vertex.set1Value( 0, SbVec3f( xmin, ymin, zmin ) );
  vProp->vertex.set1Value( 1, SbVec3f( xmax, ymin, zmin ) );
  vProp->vertex.set1Value( 2, SbVec3f( xmax, ymax, zmin ) );
  vProp->vertex.set1Value( 3, SbVec3f( xmin, ymax, zmin ) );
  vProp->vertex.set1Value( 4, SbVec3f( xmin, ymin, zmax ) );
  vProp->vertex.set1Value( 5, SbVec3f( xmax, ymin, zmax ) );
  vProp->vertex.set1Value( 6, SbVec3f( xmax, ymax, zmax ) );
  vProp->vertex.set1Value( 7, SbVec3f( xmin, ymax, zmax ) );

  vProp->orderedRGBA.set1Value( 0, color.getPackedValue() );

  // Draw it with a line set
  int coordIndices[] = { 0, 1, 2, 3, 0, -1, 4, 5, 6, 7, 4, -1, 0, 4, -1, 1, 5, -1, 2, 6, -1, 3, 7 };
  int numCoordIndices = sizeof( coordIndices ) / sizeof( int );

  auto lineset = new SoIndexedLineSet;
  lineset->vertexProperty = vProp;
  lineset->coordIndex.setValues( 0, numCoordIndices, coordIndices );
  bboxSep->addChild( lineset );

  return bboxSep;
}

void
myKeyPressCB( void* /*usrData*/, SoEventCallback* sender )
{
  const SoKeyboardEvent* event = dynamic_cast<const SoKeyboardEvent*>( sender->getEvent() );


  if ( SO_KEY_PRESS_EVENT( event, SoKeyboardEvent::Key::UP_ARROW ) )
    m_volumeRenderMode++;
  else if ( SO_KEY_PRESS_EVENT( event, SoKeyboardEvent::Key::DOWN_ARROW ) )
    m_volumeRenderMode--;
  else
    return;

  m_volumeRenderMode = ( m_volumeRenderMode < 0 ) ? 4 : m_volumeRenderMode % 5;

  if ( m_volumeRenderMode == 0 )
  {
    m_volumeRender->renderMode = SoVolumeRender::RenderMode::MAX_INTENSITY_DIFFERENCE_ACCUMULATION;
    m_helpText->string.set1Value( 0, "MAX_INTENSITY_DIFFERENCE_ACCUMULATION" );
  }
  else if ( m_volumeRenderMode == 1 )
  {
    m_volumeRender->renderMode = SoVolumeRender::RenderMode::INTENSITY_DIFFERENCE_ACCUMULATION;
    m_helpText->string.set1Value( 0, "INTENSITY_DIFFERENCE_ACCUMULATION" );
  }
  else if ( m_volumeRenderMode == 2 )
  {
    m_volumeRender->renderMode = SoVolumeRender::RenderMode::MAX_GRADIENT_DIFFERENCE_ACCUMULATION;
    m_helpText->string.set1Value( 0, "MAX_GRADIENT_DIFFERENCE_ACCUMULATION" );
  }
  else if ( m_volumeRenderMode == 3 )
  {
    m_volumeRender->renderMode = SoVolumeRender::RenderMode::GRADIENT_DIFFERENCE_ACCUMULATION;
    m_helpText->string.set1Value( 0, "GRADIENT_DIFFERENCE_ACCUMULATION" );
  }
  else if ( m_volumeRenderMode == 4 )
  {
    m_volumeRender->renderMode = SoVolumeRender::RenderMode::VOLUME_RENDERING;
    m_helpText->string.set1Value( 0, "VOLUME_RENDERING" );
  }
}
SoSeparator*
buildSceneGraph()
{
  auto root = new SoSeparator;

  // Events to change renderMode.
  auto myEventCB = new SoEventCallback();
  myEventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), myKeyPressCB );
  root->addChild( myEventCB );

  auto pVolData = new SoVolumeData();
  pVolData->fileName = "$OIVHOME/examples/data/VolumeViz/ENGINE.VOL";

  auto pTransFunc = new SoTransferFunction;
  pTransFunc->predefColorMap = SoTransferFunction::PredefColorMap::NONE;
  auto ramp = new SbVec4f[256];
  for ( int i = 0; i < 256; ++i )
  {
    float val = ( float )i / ( float )255.0;
    ramp[i].setValue( val, val, val, val );
  }
  pTransFunc->colorMap.setValues( 0, 4 * 256, ( float* )ramp );

  auto volQual = new SoVolumeRenderingQuality();
  volQual->interpolateOnMove = TRUE;
  volQual->deferredLighting = FALSE;

  m_volumeRender = new SoVolumeRender;
  m_volumeRender->renderMode = SoVolumeRender::RenderMode::MAX_INTENSITY_DIFFERENCE_ACCUMULATION;
  m_volumeRender->samplingAlignment = SoVolumeRender::VIEW_ALIGNED;

  root->addChild( volQual );
  root->addChild( pVolData );
  root->addChild( pTransFunc );
  root->addChild( m_volumeRender );
  root->addChild( createBoundingBox( pVolData->extent.getValue(), SbColor( 1, 0, 0 ) ) );

  auto helpSep = new SoSeparator;
  root->addChild( helpSep );
  {
    auto bboxNode = new SoBBox();
    bboxNode->mode = SoBBox::NO_BOUNDING_BOX;
    helpSep->addChild( bboxNode );

    auto orthoCam = new SoOrthographicCamera;
    helpSep->addChild( orthoCam );
    orthoCam->viewportMapping = SoCamera::LEAVE_ALONE;

    auto helpColor = new SoBaseColor;
    helpSep->addChild( helpColor );
    helpColor->rgb = SbColor( 0, .9f, 0 );

    auto helpTrans = new SoTranslation;
    helpSep->addChild( helpTrans );
    helpTrans->translation = SbVec3f( 0, -.88f, 0 );

    auto helpFont = new SoFont;
    helpSep->addChild( helpFont );
    helpFont->name = "Arial:Bold";
    helpFont->renderStyle = SoFont::TEXTURE;
    helpFont->size = 15;

    m_helpText = new SoText2;
    m_helpText->justification = SoText2::Justification::CENTER;
    helpSep->addChild( m_helpText );
    m_helpText->string.setNum( 2 );
    m_helpText->string.set1Value( 0, "MAX_INTENSITY_DIFFERENCE_ACCUMULATION" );
    m_helpText->string.set1Value( 1, "Use Up or Down Arrow to switch the renderMode" );
  }

  return root;
}
int
main( int argc, char** argv )
{
  QtHelper::addPlatformPluginsPath();

  QApplication app( argc, argv );
  SoVolumeRendering::init();

  ////////////////////////////////////////////////////////
  /// Scene Graph
  SoRef<SoSeparator> root = buildSceneGraph();

  ////////////////////////////////////////////////////////
  /// GUI
  auto window = new QMainWindow;
  window->setWindowTitle( "RenderModes" );
  window->resize( 800, 600 );
  window->show();

  auto centralWidget = new QWidget( window );
  window->setCentralWidget( centralWidget );
  centralWidget->setLayout( new QVBoxLayout );
  centralWidget->layout()->setContentsMargins( 0, 0, 0, 0 );
  centralWidget->layout()->setSpacing( 0 );

  ////////////////////////////////////////////////////////
  /// Viewer
  auto renderArea = new RenderAreaOrbiter( window );
  renderArea->setClearColor( SbColorRGBA( 0, .1f, .1f, 1.0f ) );
  centralWidget->layout()->addWidget( renderArea->getContainerWidget() );
  auto viewingCube = renderArea->getSceneInteractor()->getViewingCube();
  viewingCube->edgeStyle = SoViewingCube::EdgeStyle::CORNER;
  renderArea->setSceneGraph( root.ptr() );
  renderArea->setTransparencyType( SoGLRenderAction::TransparencyType::OPAQUE_FIRST );

  renderArea->getSceneInteractor()->getCamera()->orientation = SbRotation( SbVec3f( -1.0f, 0.0f, 0.0f ), 0.5f ) * SbRotation( SbVec3f( 0.0f, 1.0f, 0.0f ), -2.4f );
  renderArea->viewAll( SbViewportRegion() );
  renderArea->setAntialiasingMode( SoSceneManager::AntialiasingMode::AUTO );
  renderArea->setAntialiasingQuality( 1 );



  ////////////////////////////////////////////////////////
  /// App Exec
  app.exec();

  ////////////////////////////////////////////////////////
  /// Exit
  root = nullptr;
  delete window;
  SoVolumeRendering::finish();
  return 0;
}

