/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Pascal Estrade (MMM 2002)
**=======================================================================*/

/*----------------------------------------------------------------------------------------
Purpose : Demonstrate how to create and use a SoROIManip node.

Note: The tab color changes depending on whether we are manipulating the
      ROI box or the subvolume box.  This color change is implemented in
      the example program, not in the SoROIManip node.
----------------------------------------------------------------------------------------*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/draggers/SoTabBoxDragger.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <LDM/nodes/SoDataRange.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeSkin.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <LDM/manips/SoROIManip.h>

#include <Inventor/helpers/SbFileHelper.h>

#include <DialogViz/SoDialogVizAll.h>

Widget buildInterface(Widget);

SoSeparator * m_root;
SoROI       * m_ROI;
SoROIManip  * m_ROIManip;
SoSwitch    * m_renderSwitch;

///////////////////////////////////////////////////////////////////////
//
// This callback is called at the end of a user interaction with the
// viewer (basically when the mouse button is released).

void
viewerFinishCB( void* /*userData*/, SoXtViewer* /*viewer*/ )
{
  // When using a TabBoxDragger, the scale tab size is computed based on
  // screen size for the current orientation.  If the orientation has
  // changed because of user interaction, then generally the scale tab
  // size will need to be recomputed.

  ((SoTabBoxDragger*)m_ROIManip->getDragger())->adjustScaleTabSize();
}

///////////////////////////////////////////////////////////////////////

int
main(int , char**)
{
  Widget myWindow = SoXt::init("");
  if (myWindow == NULL) exit(1);

  //initialisation of VolumeViz and DialogViz extensions
  SoVolumeRendering::init();
  SoDialogViz::init();

  // Volume Data
  SoVolumeData *volData = new SoVolumeData();
  volData->fileName.setValue(SbFileHelper::expandString("$OIVHOME/examples/data/VolumeViz/virus.am"));

  // Get the voxel dimensions of the volume
  SbVec3i32 dimensions = volData->data.getSize();
  //volData->getVolumeData( dimensions, data, dataType );

  // Build Scene Graph
  // Top separator
  m_root = new SoSeparator;
  m_root->ref();

  // Use orthographic camera by default.
  m_root->addChild( new SoOrthographicCamera );

  // Volume Data
  m_root->addChild( volData );

  // If necessary, specify the actual range of the data values.
  //    By default VolumeViz maps the entire range of the voxel data type
  //    (e.g. 0..65535 for unsigned short) into the colormap.  This works
  //    great for byte (8 bit) voxels, but not so well for 16 bit voxels
  //    and not at all for floating point voxels. So it's not actually
  //    necessary for this data set, but shown here for completeness.
  //    NOTE: Min/max values are stored in the header for LDM format
  //    files, but for other formats the getMinMax query can take a
  //    long time because VolumeViz has to examine every voxel.
  SoDataRange *pRange = new SoDataRange();
  int voxelSize = volData->getDataSize();
  if (voxelSize > 1) {
    double minval, maxval;
    volData->getMinMax( minval, maxval );
    pRange->min = minval;
    pRange->max = maxval;
  }
  m_root->addChild( pRange );

  // Transfer function
  SoTransferFunction* transferFunction = new SoTransferFunction();
  transferFunction->predefColorMap = SoTransferFunction::GREY;
  m_root->addChild( transferFunction );

  // ROI / ROIManip
  // Initialize both ROI box and subvolume to be the entire volume.
  // Constrain the ROIManip to stay inside the volume.
  m_ROI = new SoROI();
  m_ROIManip = new SoROIManip();

  SbVec3i32 volMin = SbVec3i32(0,0,0);
  SbVec3i32 volMax = dimensions-SbVec3i32(1,1,1);

  SbVec3i32 subvolMin = volMin;
  SbVec3i32 subvolMax = volMax;
  subvolMin = (volMax-volMin)/3;
  subvolMax = (volMax-volMin)*2/3;
  volMax[2] = (volMax[2]-volMin[2])/2;

  m_ROIManip->box.setValue( subvolMin, subvolMax );
  m_ROIManip->subVolume.setValue( volMin,volMax );
  m_ROIManip->constrained = TRUE;
  m_root->addChild( m_ROIManip );

  // PickStyle
  // So that VolumeRender is not pickable -- remember that the volume
  // is pickable even where the voxels are completely transparent.
  // If we don't do this the manip will not be useable in some cases
  // where the user should expect it to be useable.
  SoPickStyle *pickStyle = new SoPickStyle;
  pickStyle->style = SoPickStyle::UNPICKABLE;
  m_root->addChild( pickStyle );


  m_renderSwitch = new SoSwitch;
  m_renderSwitch->whichChild =0;
  m_root->addChild( m_renderSwitch );

  // VolumeRender
  SoVolumeRender *m_volRender = new SoVolumeRender();
  m_volRender->samplingAlignment = SoVolumeRender::DATA_ALIGNED;
  m_volRender->fixedNumSlicesInRoi.setValue(TRUE);
  m_renderSwitch->addChild( m_volRender );

  // VolumeSkin
  SoVolumeSkin *m_volSkin = new SoVolumeSkin();
  m_renderSwitch->addChild( m_volSkin );

  // Set up viewer and dialog
  Widget parent = buildInterface(myWindow);
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(parent);
  myViewer->addFinishCallback( viewerFinishCB, NULL );
  myViewer->setSceneGraph( m_root );
  myViewer->setViewing( FALSE );
  myViewer->viewAll();
  myViewer->show();

  SoXt::show( myWindow );
  SoXt::mainLoop();
  m_root->unref();
  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();

  return 0;
}


/****************************************************************************
 **
 ** Dialog section
 **
 ****************************************************************************/

class myAuditorClass : public SoDialogAuditor
{
 public:
  myAuditorClass();

 private:
  void dialogCheckBox(SoDialogCheckBox* cpt);
  void dialogComboBox(SoDialogComboBox* cpt);
  SbBool m_ROIManipNode;
};

myAuditorClass::myAuditorClass()
{
  m_ROIManipNode = TRUE;
}

void
myAuditorClass::dialogCheckBox(SoDialogCheckBox* cpt)
{
  SbBool state = cpt->state.getValue();

  SoROI *roi = m_ROIManipNode ? m_ROIManip : m_ROI;

  if (cpt->auditorID.getValue() == "roiNodes") {
    // Search for the current SoROI / SoROIManip and replace it
    roi->ref(); // In order it is not deleted when replaced
    SoSearchAction searchAction;
    searchAction.setNode( roi );
    searchAction.apply( m_root );
    SoPath *path = searchAction.getPath();
    if ( !m_ROIManipNode ) {
      m_ROIManip->replaceNode( path );
      ((SoTabBoxDragger*)m_ROIManip->getDragger())->adjustScaleTabSize();
    }
    else
      m_ROIManip->replaceManip( path, m_ROI );
    m_ROIManipNode = !m_ROIManipNode;
    roi->unrefNoDelete(); // Balance the ref() above
  }
  if (cpt->auditorID.getValue() == "renderNodes") {
    if ( state)
      m_renderSwitch->whichChild =0;
    else
      m_renderSwitch->whichChild =1;
  }
  else if (cpt->auditorID.getValue() == "constrained")
    m_ROIManip->constrained = state;

  else if (cpt->auditorID.getValue() == "controlling") {
    m_ROIManip->boxOn = state;
    // Different color according to which box is dragged.
    SoMaterial *material = (SoMaterial*)(m_ROIManip->getDragger()->getPart("tabPlane1.scaleTabMaterial",0));
    if ( state ) {
      material->diffuseColor.setValue(0,1,0);
      material->emissiveColor.setValue(0,1,0);
    } else {
      material->diffuseColor.setValue(1,0,0);
      material->emissiveColor.setValue(1,0,0);
    }
  }

  else if (cpt->auditorID.getValue() == "relative")
    roi->relative = state;

}

void myAuditorClass::dialogComboBox(SoDialogComboBox* cpt)
{
  int selectedItem = cpt->selectedItem.getValue();

  SoROI *roi = m_ROIManipNode ? m_ROIManip : m_ROI;

  int flags[] = { SoROI::SUB_VOLUME, SoROI::EXCLUSION_BOX, SoROI::CROSS, SoROI::CROSS_INVERT, SoROI::FENCE, SoROI::FENCE_INVERT };
  roi->flags = flags[selectedItem];
}

Widget
buildInterface(Widget window)
{
  SoInput myInput;
  if (! myInput.openFile( "$OIVHOME/examples/source/VolumeViz/roiManip/roiManipDialog.iv" ))
    return NULL;
  SoGroup *myGroup = SoDB::readAll( &myInput );
  if (! myGroup)
    return NULL;
  SoTopLevelDialog *myTopLevelDialog = (SoTopLevelDialog *)myGroup->getChild( 0 );

  myAuditorClass *myAuditor = new myAuditorClass;
  myTopLevelDialog->addAuditor(myAuditor);

  SoDialogCustom *customNode = (SoDialogCustom *)myTopLevelDialog->searchForAuditorId(SbString("Viewer"));

  myTopLevelDialog->buildDialog( window, customNode != NULL );
  myTopLevelDialog->show();

  return customNode ? customNode->getWidget() : window;
}


