/**
 * This demonstate how to use the segmentedInterpolation and colorInterpolation fields in order to avoid doing interpolation
 * on segmented data.
 */
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/SoPreferences.h>

#include <VolumeViz/nodes/SoVolumeDataDrawStyle.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoVolumeIsosurface.h>
#include <Inventor/actions/SoSearchAction.h>

#include <DialogViz/SoDialogVizAll.h>

#include "utils.h"

#include <Inventor/STL/algorithm>

#include <Inventor/helpers/SbFileHelper.h>

// set to 1 tu use ldm file instead of generating in memory data (usefull
// when recording ART)
#define USE_LDM 0

SbString DataFile = "$OIVHOME/examples/source/VolumeViz/segmentedInterpolation/scene.iv";

const size_t VOLUMESIZE = 256;
SoVolumeRenderingQuality* g_vrq;

/*********************************************************************************/
class InterfaceAuditor : public SoDialogAuditor
{
public:
  InterfaceAuditor(SoTopLevelDialog* topDialog){m_top = topDialog;}

private:
    SoTopLevelDialog* m_top;

  virtual void dialogRealSlider( SoDialogRealSlider *cpt )
  {
    if ( cpt->auditorID.getValue() == "segmentedThreshold" )
    {
      g_vrq->segmentedInterpolationThreshold = cpt->value.getValue();
    }
  }

  virtual void dialogComboBox(SoDialogComboBox* cpt)
  {
    int selectedItem = cpt->selectedItem.getValue();
    SbString item = cpt->items[selectedItem];

    if (cpt->auditorID.getValue() == "interpolation")
    {
      if ( item == "Segmented" )
      {
        g_vrq->segmentedInterpolation = TRUE;
        g_vrq->colorInterpolation = TRUE;
      }
      else if ( item == "Gradient" )
      {
        g_vrq->segmentedInterpolation = FALSE;
        g_vrq->colorInterpolation = FALSE;
      }
      else
      {
        g_vrq->segmentedInterpolation = FALSE;
        g_vrq->colorInterpolation = TRUE;
      }
    }
  }

};

/*********************************************************************************/
struct Body
{
  Body(const SbVec3i32& pos, unsigned int radius, float value)
    : pos(pos), radius(radius), value(value) {}
  SbVec3i32 pos;
  unsigned int radius;
  float value;
};

/*********************************************************************************/
void generateBodies(SoVolumeData* vd, size_t size, const std::vector<Body>& bodies)
{
  unsigned char* data = new unsigned char[size*size*size];
  std::fill(data, data+size*size*size, 0);

  size_t offs = 0;
  for ( unsigned int k= 0; k < size; k++ )
    for ( unsigned int j = 0; j < size; j++ )
      for ( unsigned int i = 0; i < size; i++ )
      {
        for ( size_t b = 0; b < bodies.size(); b++ )
        {
          const Body& body = bodies[b];
          SbVec3f radiusVec = SbVec3f(SbVec3i32(i, j, k)-body.pos);
          float dist = radiusVec.length();
          radiusVec /= dist;
          float value = 0;
          float radius = body.radius*sin(15*radiusVec[0]*radiusVec[1]*radiusVec[2]);
          if ( dist < radius )
            value = body.value;
          data[offs] += (unsigned char)value;
        }
        offs++;
      }

  vd->data.setValue(SbVec3i32((int)size, (int)size, (int)size), SbDataType::UNSIGNED_BYTE,
                    0, data, SoSFArray::NO_COPY_AND_DELETE);
}

/*********************************************************************************/
int main(int , char **argv)
{
  SoTopLevelDialog *myTopLevelDialog;
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow)
    return 0;

  SoVolumeRendering::init();
  SoDialogViz::init();
  SoLDMGlobalResourceParameters::setViewCulling(FALSE);
  SoLDMGlobalResourceParameters::setViewpointRefinement(FALSE);
  SoLDMGlobalResourceParameters::setScreenResolutionCulling(FALSE);

  SoSeparator* scene = readFile(DataFile.toLatin1());
  if(!scene)
  {
    std::cerr << DataFile.toStdString() << "not found" << std::endl;
    return 0;
  }

  SbString InterfaceName = "$OIVHOME/examples/source/VolumeViz/segmentedInterpolation/interface.iv";

  Widget parent = buildInterface(myWindow, InterfaceName.toLatin1(), "Viewer", &myTopLevelDialog);
  myTopLevelDialog->addAuditor(new InterfaceAuditor(myTopLevelDialog));

  SoSeparator *root = new SoSeparator;
  root->ref();
  root->addChild(scene);

  SoVolumeData* vd = searchNode<SoVolumeData>(scene);
  SoVolumeIsosurface* iso = searchNode<SoVolumeIsosurface>(scene);

  if ( iso->isovalues.getNum() < 3 )
  {
    std::cerr << "Not enough isovalues, 3 needed" << std::endl;
    return 0;
  }
  const float* values = iso->isovalues.getValues(0);

  std::vector<Body> bodies;
  bodies.push_back(Body(SbVec3i32(30, 7*VOLUMESIZE/8, 30), 30, values[0]));
  bodies.push_back(Body(SbVec3i32(VOLUMESIZE/3, VOLUMESIZE/3, VOLUMESIZE/3), 90, values[1]));
  bodies.push_back(Body(SbVec3i32(3*VOLUMESIZE/4, 3*VOLUMESIZE/4, 3*VOLUMESIZE/4), 40, values[2]));
#if USE_LDM
  vd->fileName = "$OIVHOME/examples/source/VolumeViz/segmentedInterpolation/trumpet.ldm";
#else
  generateBodies(vd, VOLUMESIZE, bodies);
#endif

  g_vrq = searchNode<SoVolumeRenderingQuality>(scene);

  // Enable light for isoSurface
  g_vrq->lighting.setValue( TRUE );
  g_vrq->deferredLighting.setValue( FALSE );

  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(parent);
  myViewer->setSceneGraph(root);
  myViewer->setTitle("segmented interpolation");
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete myViewer;
  root->unref();
  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();
  return 0;
}


