/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to create and use a simple volumeData and volumeRender node.
author : David Beilloin
August 2002
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoSphere.h>
#include <Inventor/nodes/SoMaterial.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeClippingGroup.h>
#include <LDM/nodes/SoDataRange.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>

//main function
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  // Initialize of VolumeViz extension
  SoVolumeRendering::init();

  // Node to hold the volume data  
  SoVolumeData* pVolData = new SoVolumeData();
  pVolData->fileName =  "$OIVHOME/examples/data/VolumeViz/3DHEAD.ldm";

  // If necessary, specify the actual range of the data values.
  //    By default VolumeViz maps the entire range of the voxel data type
  //    (e.g. 0..65535 for unsigned short) into the colormap.  This works
  //    great for byte (8 bit) voxels, but not so well for 16 bit voxels
  //    and not at all for floating point voxels. So it's not actually
  //    necessary for this data set, but shown here for completeness.
  //    NOTE: Min/max values are stored in the header for LDM format
  //    files, but for other formats the getMinMax query can take a
  //    long time because VolumeViz has to examine every voxel.
  SoDataRange *pRange = new SoDataRange();
  int voxelSize = pVolData->getDataSize();
  if (voxelSize > 1) {
    double minval, maxval;
    pVolData->getMinMax( minval, maxval );
    pRange->min = minval;
    pRange->max = maxval;
  }

  // Use a predefined colorMap with the SoTransferFunction
  SoTransferFunction* pTransFunc = new SoTransferFunction;
  pTransFunc->predefColorMap = SoTransferFunction::GLOW;
  pTransFunc->minValue = 42;
  pTransFunc->maxValue = 120;

  // Node in charge of drawing the volume
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->samplingAlignment = SoVolumeRender::VIEW_ALIGNED;

  // Node in charge of handling the Clipping geometry
  SoVolumeClippingGroup *pVolClip = new SoVolumeClippingGroup;
  
  //Clip with a sphere
  SoSphere* sphere = new SoSphere;
  sphere->radius = 0.6f;
  pVolClip->addChild(sphere);

  //Add a SoVolumeRenderingQuality to make it nicer
  SoVolumeRenderingQuality* vrq = new SoVolumeRenderingQuality;
  vrq->preIntegrated = TRUE;
  vrq->jittering     = TRUE;

  // Assemble the scene graph
  // Note: SoVolumeRender must appear after the SoVolumeData node.
  SoSeparator *root = new SoSeparator;
  root->ref();
  root->addChild( pVolClip );
  root->addChild( pVolData );
  root->addChild( pRange   );
  root->addChild( pTransFunc );
  root->addChild( vrq );  
  root->addChild( pVolRender );

  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setSceneGraph(root);
  myViewer->setTitle("Volume rendering");
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete myViewer;
  root->unref();
  SoVolumeRendering::finish();
  SoXt::finish();

  return 0;
}


