/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to render an heightfield with a SoHeightFieldRender.
----------------------------------------------------------------------------------------*/

// header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoGradientBackground.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoPhysicalMaterial.h>
#include <Inventor/nodes/SoScale.h>
#include <Inventor/nodes/SoSeparator.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoHeightFieldGeometry.h>
#include <VolumeViz/nodes/SoHeightFieldRender.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>

#include "utils.h"

static const SbString HorizonFile( "$OIVHOME/examples/data/VolumeViz/horizons/horizon.ldm" );

/*******************************************************************************/
int
main( int, char** argv )
{
  // Create the window
  SoTopLevelDialog* myTopLevelDialog;
  Widget myWindow = SoXt::init( argv[0] );
  if ( !myWindow )
    return 0;

  // Initialize of VolumeViz extension
  SoVolumeRendering::init();
  SoDialogViz::init();

  // SoComplexity node is used to adjust tessellation level of HeightFieldRender and simplify rendering.
  SoComplexity* complexity = new SoComplexity;

  // Assemble the scene graph
  SoRef<SoSeparator> root = new SoSeparator;
  {
    root->addChild( new SoGradientBackground );
    root->addChild( new SoDirectionalLight ); // additional light source
    root->addChild( complexity );

    SoLightModel* lightModel = new SoLightModel;
    lightModel->model = SoLightModel::PHYSICALLY_BASED;
    root->addChild( lightModel );

    // Material
    SoPhysicalMaterial* material = new SoPhysicalMaterial;
    material->baseColor = SbColorRGBA(1.0f, 1.0f, 1.0f, 1.0f);
    material->specular = 1.0f;
    material->roughness = 0.4f;
    root->addChild( material );

    SoTransferFunction* tf = new SoTransferFunction;
    tf->predefColorMap = SoTransferFunction::STANDARD;
    root->addChild( tf );

    // Add a scale on z
    SoScale* scale = new SoScale;
    scale->scaleFactor = SbVec3f( 1.f, 1.f, 0.1f );
    root->addChild( scale );

    // Node to hold the property
    SoHeightFieldGeometry* hg = new SoHeightFieldGeometry;
    hg->fileName = HorizonFile;
    root->addChild( hg );

    SoHeightFieldRender* hf = new SoHeightFieldRender;
    root->addChild( hf );
  }

  SbString InterfaceFile = "$OIVHOME/examples/source/VolumeViz/simpleHorizon/interface.iv";
  Widget parent = buildInterface( myWindow, InterfaceFile.toLatin1(), "Viewer", &myTopLevelDialog );

  // Connect dialog slider to complexity node value
  SoDialogIntegerSlider* slider = static_cast<SoDialogIntegerSlider*>( myTopLevelDialog->searchForAuditorId( "NumTris" ) );
  complexity->value.connectFrom( &slider->value );

  // Set up viewer:
  SoXtExaminerViewer* myViewer = new SoXtExaminerViewer( parent );
  myViewer->setSceneGraph( root.ptr() );
  myViewer->setTitle( "Simple Horizon" );
  myViewer->viewAll();
  myViewer->saveHomePosition();
  myViewer->show();
  SoXt::show( myWindow );
  SoXt::mainLoop();
  delete myViewer;
  root = nullptr;

  SoDialogViz::finish();
  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}
