/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to render an heightfield with a SoHeightFieldRender.
----------------------------------------------------------------------------------------*/
//#define USE_BBOX 1

//header files
#include <ScaleViz/SoScaleViz.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoGradientBackground.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoScale.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeIsosurface.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoHeightFieldGeometry.h>
#include <VolumeViz/nodes/SoHeightFieldProperty.h>
#include <VolumeViz/nodes/SoHeightFieldRender.h>
#include <LDM/nodes/SoMultiDataSeparator.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <Inventor/sensors/SoTimerSensor.h>
#include <Inventor/stats/SoPerfCounterManager.h>
#include <Inventor/stats/SoPerfCounter.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoFontStyle.h>
#include <Inventor/nodes/SoCallback.h>
#include <Inventor/nodes/SoBBox.h>
#include <Inventor/STL/iostream>
#include <Inventor/STL/sstream>

#include <Inventor/helpers/SbFileHelper.h>

#include "utils.h"

SoText2* g_infoText = NULL;


void GenerateHeight_Property(SbVec3i32& Dims, float*& height, float*& data, SbBox3f& HorizonBox, float /*UndefValue*/) 
{
 const float Pi2 = 2.0f*acos(-1.0f);

 int dimX = Dims[0];
 int dimY = Dims[1];
 int dimZ = Dims[2];

  // Create data :
  // Note: 0 < data[i] < number of entries in the colorMap.
  // If each data value has the same value A the volume will be unicolor
  // with the color described by the Ath entries of the colorMap.
  // This volume data will use every color in the colorMap
  height = new float[dimX * dimY* dimZ];
  data = new float[dimX * dimY* dimZ];

  double irel,jrel;
  for( int i=0; i<dimX; i++ )
  {
    irel = ( double( i ) / double( dimX ) );
    for ( int j = 0; j < dimY; j++ )
    {
      jrel = ( double( j ) / double( dimY ) );
      height[j*dimX + i] = float( sin( Pi2 * irel ) + cos( Pi2 * jrel ) );
      data[j*dimX + i] = float( jrel );
    }
  }

  //find zmin and zmax for HorizonBox
  float zmin,zmax,zz;
  zmin = 1e+24f; zmax = -1e+24f;
  for (int ii=0;ii<dimX * dimY * dimZ;ii++)
  {
    zz = height[ii];
    if ( zz < zmin ) zmin = zz;
    if ( zz > zmax ) zmax = zz;
  }

  HorizonBox.setBounds(-2.f*dimX,-2.f*dimY,zmin,2.f*dimX,2.f*dimY,zmax);
 
}

void buildHorizonFromElevationMap(
  SoHeightFieldGeometry* hg,
  SoHeightFieldProperty* hp,
  const SbVec3i32 Dims,
  const SbBox3f& box,
  float* elevationMap,
  float* dataMap,
  float /*undefValue*/,
  int /*tileSize*/
  )
{

  unsigned int dimXYZ = Dims[0]*Dims[1]*Dims[2];

  SoMemoryObject* memObjHeight = new SoMemoryObject(elevationMap, dimXYZ *sizeof(float), SbDataType::FLOAT,SoMemoryObject::NO_COPY);
  SoMemoryObject* memObjData = new SoMemoryObject(dataMap, dimXYZ *sizeof(float), SbDataType::FLOAT,SoMemoryObject::NO_COPY);

  // Node to hold the property
  hg->data.setValue(Dims ,32, memObjHeight, SoSFArray::NO_COPY );
  hg->extent.setValue( box );
  hg->dataSetId = 2;

  // Node to hold the heights
  hp->data.setValue( Dims,32, memObjData, SoSFArray::NO_COPY );
  hp->extent.setValue( box );
  hp->dataSetId = 1;
}

/*******************************************************************************/
int main(int, char **argv)
{
  // Create the window
  SoTopLevelDialog *myTopLevelDialog;
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  if ( !SoScaleViz::isRunningART() )
  {
    SoPreferences::setBool(SbString("OIV_PERFCOUNTER_ENABLE"), TRUE);
    SoPreferences::setString(SbString("OIV_PERFCOUNTER_OUTPUT"), "perfcounter.log");
  }

  // Initialize of VolumeViz extension
  SoVolumeRendering::init();
  SoDialogViz::init();

  SbBox3f HorizonBox;
  float *height = NULL;
  float *data = NULL;
  float UndefValue = -999.9f;

  SoHeightFieldGeometry* hg = new SoHeightFieldGeometry();
  SoHeightFieldProperty* hp = new SoHeightFieldProperty();
  SoHeightFieldRender* hf = new SoHeightFieldRender;

  SbVec3i32 Dims(1024,1024,1);
  GenerateHeight_Property(Dims,height,data,HorizonBox,UndefValue);
  buildHorizonFromElevationMap( hg,hp,Dims, HorizonBox,height,data,UndefValue, 64 );

  float dx,dy,dz;
  HorizonBox.getSize(dx,dy,dz);

  // SoComplexity node is used to adjust tessellation level of HeightFieldRender and simplify rendering.
  SoComplexity* complexity = new SoComplexity;

  // Add a scale on z
  SoScale* scale = new SoScale;
  scale->scaleFactor = SbVec3f(1.f, 1.f, (dx*0.3f)/dz);

#ifdef USE_BBOX
  SoBBox* bbox = new SoBBox();
  bbox->boundingBox.setValue(HorizonBox);
  bbox->mode.setValue("USER_DEFINED");
#endif

  SoMultiDataSeparator* mds = new SoMultiDataSeparator;
  mds->addChild(scale);  

#ifdef USE_BBOX
  mds->addChild(bbox);
#endif

  mds->addChild(hp);
  mds->addChild(hg);
  mds->addChild(hf);

  SoTransferFunction* tf = new SoTransferFunction;
  tf->predefColorMap = SoTransferFunction::STANDARD;

  //Material
  SoMaterial *matVolRend = new SoMaterial;
  matVolRend->diffuseColor.setValue(1,1,1);
  matVolRend->transparency.setValue( 0.0f );
  matVolRend->specularColor.setValue(0.1f,0.1f,0.1f);
  matVolRend->shininess.setValue(0.3f);

  // Assemble the scene graph
  SoSeparator *root = new SoSeparator;
  root->ref();
  root->addChild(new SoPerspectiveCamera);

  root->addChild( new SoGradientBackground );
  root->addChild( complexity );
  root->addChild( matVolRend );
  root->addChild( tf );
  root->addChild( mds );

  SbString InterfaceFile = SbFileHelper::expandString("$OIVHOME/examples/source/VolumeViz/simpleHorizonInMemory/interface.iv");
  Widget parent = buildInterface(myWindow, InterfaceFile.toLatin1(), "Viewer", &myTopLevelDialog);
  SoDialogIntegerSlider *slider;
  slider = (SoDialogIntegerSlider *)myTopLevelDialog->searchForAuditorId("NumTris");

  complexity->value.connectFrom( &slider->value );

  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(parent);
  myViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
  myViewer->setSceneGraph(root);
  myViewer->setTitle("Simple Horizon");
  myViewer->viewAll();
  myViewer->saveHomePosition();
  myViewer->show();

  myViewer->bindNormalContext();
  if (!SoHeightFieldRender::isSupported())
  {
    SoError::post("Some extensions are not supported on the current platform to launch this demo\n");
    return -1;
  }
  myViewer->unbindNormalContext();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete myViewer;
  root->unref();

  SoDialogViz::finish();
  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}
