//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoShadowGroup.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>

#include <Inventor/helpers/SbFileHelper.h>

#include "utils.h"

#define FILENAME "$OIVHOME/examples/data/VolumeViz/bonsai.am"
#define COLORMAPFILENAME "$OIVHOME/examples/data/VolumeViz/bonsai.txt"

#ifndef MAX_PATH
  #define MAX_PATH 4096
#endif

// Forward declarations
void loadColormap(const char *filename, SoTransferFunction *tf);
unsigned char* loadData(const SbString &filename, SoVolumeData *volData);
void updateOpaqueRegion( SoTransferFunction *tf, int minCm, int maxCm, int minOp, int maxOp, SbBool invertFlag );

// Global variables for user interface
SoInteractiveComplexity* g_cplx  = NULL;
SoTransferFunction* g_pTransFunc = NULL;
int g_minOpaqueMap;
int g_maxOpaqueMap;

class InterfaceAuditor : public SoDialogAuditor
{
public:
  InterfaceAuditor(SoTopLevelDialog* topDialog){m_top = topDialog;}

private:
  SoTopLevelDialog* m_top;

  void dialogComboBox(SoDialogComboBox* cpt)
  {
    if (cpt->auditorID.getValue() == "interactiveMode")
    {
      int selectedItem = cpt->selectedItem.getValue();
      int interactiveMode[] = { SoInteractiveComplexity::AUTO,
                                SoInteractiveComplexity::FORCE_INTERACTION,
                                SoInteractiveComplexity::FORCE_STILL,
                                SoInteractiveComplexity::DISABLED};
      g_cplx->interactiveMode = interactiveMode[selectedItem];
    }
  }

  void dialogIntegerSlider(SoDialogIntegerSlider* cpt)
  {
    int value = cpt->value.getValue();

    if (cpt->auditorID.getValue() == "OpaqueMin")
    {
      g_minOpaqueMap = value;
      updateOpaqueRegion(g_pTransFunc, 0, 255, g_minOpaqueMap, g_maxOpaqueMap, FALSE);
    }
    else if (cpt->auditorID.getValue() == "OpaqueMax")
    {
      g_maxOpaqueMap = value;
      updateOpaqueRegion(g_pTransFunc, 0, 255, g_minOpaqueMap, g_maxOpaqueMap, FALSE);
    }
  }

};

//main function
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  // Initialize of VolumeViz extension
  SoVolumeRendering::init();
  SoDialogViz::init();

  //Setup an interactive complexity node
  //VolumeRender will be rendered with 200 slices when moving and 4000 when still
  //Shadows will be disabled when moving
  g_cplx = new SoInteractiveComplexity;
  g_cplx->fieldSettings.set1Value(0, "SoVolumeRender numSlices 200 4000 2000");
  g_cplx->fieldSettings.set1Value(1, "SoShadowGroup isActive FALSE");

  // Node to hold the volume data
  SoVolumeData* pVolData = new SoVolumeData();
  pVolData->fileName = FILENAME;

  // Load the colorMap from a file
  g_pTransFunc = new SoTransferFunction;
  loadColormap(COLORMAPFILENAME, g_pTransFunc);

  // Property node which allows SoVolumeRender to enable preintegration and jittering
  SoVolumeRenderingQuality *pVRVolQuality = new SoVolumeRenderingQuality;
  pVRVolQuality->preIntegrated = TRUE;
  pVRVolQuality->jittering = TRUE;

  // Node in charge of drawing the volume
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->numSlicesControl = SoVolumeRender::ALL;

  //Add shadows to the scene
  SoShadowGroup* sg = new SoShadowGroup;
  sg->method = SoShadowGroup::VARIANCE_SHADOW_MAP;

  // Assemble the scene graph
  // Note: SoVolumeRender must appear after the SoVolumeData node.
  SoSeparator *root = new SoSeparator;
  root->ref();
  root->addChild(g_cplx);
  root->addChild(sg);

  sg->addChild( pVolData );
  sg->addChild( g_pTransFunc );
  sg->addChild( pVRVolQuality );
  sg->addChild( pVolRender );

  SoTopLevelDialog *myTopLevelDialog;
  SbString InterfaceFile = "$OIVHOME/examples/source/VolumeViz/simpleInteractiveParameters/interface.iv";
  Widget parent = buildInterface(myWindow, InterfaceFile.toLatin1(), "Viewer", &myTopLevelDialog);
  InterfaceAuditor* myInterfaceAuditor = new InterfaceAuditor(myTopLevelDialog);
  myTopLevelDialog->addAuditor(myInterfaceAuditor);

  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(parent);
  myViewer->setSceneGraph(root);
  myViewer->setTitle("Simple Interactive Parameters");
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete myViewer;
  root->unref();

  delete myInterfaceAuditor;
  releaseInterface();

  SoDialogViz::finish();
  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}

void loadColormap( const char *colorFile, SoTransferFunction *tf )
{
  FILE *file;
  const int BUFSIZE = 256;
  char buf[BUFSIZE];

  file = SbFileHelper::open(colorFile, "r");

  int num_colors = 0;

  // Get first line of file and try to identify it.
  fscanf(file, "%d", &num_colors);
  fgets( buf, BUFSIZE, file );

  float rgba[256][4];

  int numComponents = 0;
  for (int i = 0; i < num_colors; ) {
    char line[256];
    fgets( line, 256, file );
    float color[4];
    numComponents = sscanf( line, "%g %g %g %g", &color[0], &color[1], &color[2], &color[3]);
    for (int k = 0; k < numComponents; k++)
      color[k] /= 65535.0f;
    rgba[i][0] = color[0];
    rgba[i][1] = color[1];
    rgba[i][2] = color[2];
    rgba[i][3] = color[3];
    i++;
  }
  fclose(file);

  int mode = 256 / num_colors;

  static float rgba256[256][4];
  int index = 0;
  for (int k = 0; k < num_colors; k++) {
    int k1 = k + 1;
    if (k1 >= num_colors) k1 = num_colors-1;
    for (int i = 0; i < mode; i++) {
      rgba256[index][0] = rgba[k][0] + (float)i/mode*(rgba[k1][0]-rgba[k][0]);
      rgba256[index][1] = rgba[k][1] + (float)i/mode*(rgba[k1][1]-rgba[k][1]);
      rgba256[index][2] = rgba[k][2] + (float)i/mode*(rgba[k1][2]-rgba[k][2]);
      rgba256[index][3] = rgba[k][3] + (float)i/mode*(rgba[k1][3]-rgba[k][3]);
      index++;
    }
  }

  tf->predefColorMap = SoTransferFunction::NONE;
  tf->colorMapType = SoTransferFunction::RGBA;
  tf->colorMap.setNum( 256 * 4 );
  tf->colorMap.setValues( 0, 256 * 4, &rgba256[0][0] );
  g_minOpaqueMap = 0;
  g_maxOpaqueMap = 255;
}

///////////////////////////////////////////////////////////////////////
//
// Make the color map completely transparent *except*
// in the specified range (where alpha is left alone)
//
// Modified from Paul's original code.  Original took
// a list of float values, this one takes an array of
// unsigned ints, each of which is a packed RGBA value.
//
void updateOpaqueRegion( SoTransferFunction *tf, int minCm, int maxCm, int minOp, int maxOp, SbBool invertFlag )
{
  {
    // If min is zero, no change.
    // If min is less than zero, also no change (bogus case).
    tf->minValue = minCm;
    tf->maxValue = maxCm;

    int length = tf->actualColorMap.getNum() / 4;
    float *actualColorMapF = tf->actualColorMap.startEditing();
    if (actualColorMapF != NULL)
    {
      if(!invertFlag)//remap inside min map
      {
        // If min is zero, no change.
        // If min is less than zero, also no change (bogus case).
        for (int i = 0; i < minOp; i++)
          actualColorMapF[4*i+3] = 0.;
        // If max = length-1 (last index in color map), no change.
        // If max > length-1, also no change (bogus case).
        for (int k = maxOp+1; k < length; k++)
          actualColorMapF[4*k+3] = 0.;
      }
      else
      {
        for (int i = minOp; i <= maxOp; i++)
          actualColorMapF[4*i+3] = 0.;
      }
    }
    tf->actualColorMap.finishEditing();
  }
}


