/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to use a SoVolumeRenderingQuality property node
          for lighting a volume.
          It is also showing how to display progress for a fixedResolution loading/rendering.
author : Benjamin Grange
September 2005
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoPhysicalMaterial.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <LDM/nodes/SoDataRange.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>

#include <iomanip>
#include <Inventor/SoProgressIndicator.h>

#include <Inventor/helpers/SbFileHelper.h>

#define FILENAME "$OIVHOME/examples/data/VolumeViz/bonsai.am"
#define COLORMAPFILENAME "$OIVHOME/examples/data/VolumeViz/bonsai.txt"

#ifndef MAX_PATH
  #define MAX_PATH 4096
#endif

void loadColormap(const char *filename, SoTransferFunction *tf);
unsigned char* loadData(const SbString &filename, SoVolumeData *volData);


SbElapsedTime localTime;

// Callback for enableFixedResolutionMode
void fixedResCB( SoVolumeData::SoLDMResourceParameters::FixedResolutionReport& report, void* /*vData*/ )
{
  static bool s_isPrintDone = false;

  switch ( report.what )
  {
    //this will be called in fixed resolution mode for each tile
  case SoVolumeData::SoLDMResourceParameters::FixedResolutionReport::PROGRESS:
  {
    if ( report.numTilesToLoad && report.numTilesLoaded < report.numTilesToLoad )
    {
      std::cout << std::fixed << std::setprecision( 2 ) << double( report.numTilesLoaded )*100. / double( report.numTilesToLoad ) << "%\r" << std::flush;
    }
    if ( report.numTilesToLoad == report.numTilesLoaded && !s_isPrintDone )
    {
      std::cout << "\nTime to read: " << localTime.getElapsed() << "s" << std::endl;
      s_isPrintDone = true;
    }
    break;
  }
  default:
    return;
  }
}

class EventRaisedProgressBar
{
public:

  EventRaisedProgressBar() {};

  void onBeginTaskFunc( SoProgressIndicator::TaskEventArg& eventArg )
  {
    std::cout << "BEGIN_TASK" << ", name: " << eventArg.getEventName() << ", with " << eventArg.getNumSubTasks() << " phases." << std::endl;
  }
  void onEndTaskFunc( SoProgressIndicator::TaskEventArg& eventArg )
  {
    std::cout << "END_TASK" << ", name: " << eventArg.getEventName() << ", with " << eventArg.getNumSubTasks() << " phases." << std::endl;
    SoVolumeRender* vr = dynamic_cast<SoVolumeRender*>( eventArg.getSource() );
    if ( vr )
    {
      vr->setRenderProgress( NULL );
    }
  }

  void onBeginSubtaskFunc( SoProgressIndicator::SubTaskEventArg& eventArg )
  {
    std::cout << "\tBEGIN_SUBTASK " << eventArg.getNumSubTasksDone() << "/" << eventArg.getNumSubTasks() << ", name: " << eventArg.getEventName() << ", with " << eventArg.getNumSteps() << " steps." << std::endl;
  }

  void onEndSubtaskFunc( SoProgressIndicator::SubTaskEventArg& eventArg )
  {
    std::cout << "\tEND_SUBTASK " << eventArg.getNumSubTasksDone() << "/" << eventArg.getNumSubTasks() << ", name: " << eventArg.getEventName() << ", with " << eventArg.getNumSteps() << " steps." << std::endl;
  }


  void onProgressFunc( SoProgressIndicator::StepEventArg& eventArg )
  {
    std::cout << std::fixed << std::setprecision( 2 ) << "\t\t" << eventArg.getNumStepsDone()*100.f / (double)eventArg.getNumSteps() << "%\r" << std::flush;
  }
};

//main function
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  // Initialize of VolumeViz extension
  SoVolumeRendering::init();

  // Node to hold the volume data
  SoVolumeData* pVolData = new SoVolumeData();

  //Load the model in the SoVolumeData
  pVolData->fileName = FILENAME;
  pVolData->ldmResourceParameters.getValue()->enableFixedResolutionMode( 0, fixedResCB );
  localTime.reset();

  // If necessary, specify the actual range of the data values.
  //    By default VolumeViz maps the entire range of the voxel data type
  //    (e.g. 0..65535 for unsigned short) into the colormap.  This works
  //    great for byte (8 bit) voxels, but not so well for 16 bit voxels
  //    and not at all for floating point voxels. So it's not actually
  //    necessary for this data set, but shown here for completeness.
  //    NOTE: Min/max values are stored in the header for LDM format
  //    files, but for other formats the getMinMax query can take a
  //    long time because VolumeViz has to examine every voxel.
  SoDataRange* pRange = new SoDataRange();
  int voxelSize = pVolData->getDataSize();
  if (voxelSize > 1) {
    double minval, maxval;
    pVolData->getMinMax( minval, maxval );
    pRange->min = minval;
    pRange->max = maxval;
  }

  // Load the colorMap from a file
  SoTransferFunction* pTransFunc = new SoTransferFunction;
  loadColormap(COLORMAPFILENAME, pTransFunc);

  // Property node which allows SoVolumeRender to draw lighted volumes
  SoVolumeRenderingQuality* pVRVolQuality = new SoVolumeRenderingQuality;
  pVRVolQuality->lighting = TRUE;
  pVRVolQuality->preIntegrated = TRUE;

  // Node in charge of drawing the volume
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->numSlicesControl = SoVolumeRender::ALL;
  pVolRender->subdivideTile = TRUE;

  // To track volumeRender progression we need to attach an SoProgressIndicator
  SoProgressIndicator ps;

  EventRaisedProgressBar e;
  ps.onBeginTask.add( e, &EventRaisedProgressBar::onBeginTaskFunc );
  ps.onEndTask.add( e, &EventRaisedProgressBar::onEndTaskFunc );
  ps.onBeginSubTask.add( e, &EventRaisedProgressBar::onBeginSubtaskFunc );
  ps.onEndSubTask.add( e, &EventRaisedProgressBar::onEndSubtaskFunc );
  ps.onEndStep.add( e, &EventRaisedProgressBar::onProgressFunc );

  pVolRender->setRenderProgress( &ps );

  SoLightModel* lightModel = new SoLightModel;
  lightModel->model = SoLightModel::PHYSICALLY_BASED;

  //Set the parameters of the material
  SoPhysicalMaterial* material = new SoPhysicalMaterial;
  material->baseColor = SbColorRGBA(1.0f, 1.0f, 1.0f, 1.0f);
  material->specular = 1.0f;
  material->roughness = 0.4f;


  // Assemble the scene graph
  // Note: SoVolumeRender must appear after the SoVolumeData node.
  SoRef<SoSeparator> root = new SoSeparator;
  SoDirectionalLight* light = new SoDirectionalLight;
  light->direction = SbVec3f(-0.4f, -0.4f, -0.8f);
  root->addChild( light );
  root->addChild( lightModel );
  root->addChild( material );
  root->addChild( pVolData );
  root->addChild( pRange   );
  root->addChild( pTransFunc );
  root->addChild( pVRVolQuality );
  root->addChild( pVolRender );

  // Set up viewer:
  SoXtExaminerViewer* myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setSceneGraph(root.ptr());
  myViewer->setTitle("Pre-integrated lighted Volume rendering");
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete myViewer;

  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}

void loadColormap( const char *colorFile, SoTransferFunction *tf )
{
  FILE *file;
  const int BUFSIZE = 256;
  char buf[BUFSIZE];

  file = SbFileHelper::open(colorFile, "r");

  int num_colors = 0;

  // Get first line of file and try to identify it.
  fscanf(file, "%d", &num_colors);
  fgets( buf, BUFSIZE, file );

  float rgba[256][4];

  int numComponents = 0;
  for (int i = 0; i < num_colors; ) {
    char line[256];
    fgets( line, 256, file );
    float color[4];
    numComponents = sscanf( line, "%g %g %g %g", &color[0], &color[1], &color[2], &color[3]);
    for (int k = 0; k < numComponents; k++)
      color[k] /= 65535.0f;
    rgba[i][0] = color[0];
    rgba[i][1] = color[1];
    rgba[i][2] = color[2];
    rgba[i][3] = color[3];
    i++;
  }
  fclose(file);

  int mode = 256 / num_colors;

  static float rgba256[256][4];
  int index = 0;
  for (int k = 0; k < num_colors; k++) {
    int k1 = k + 1;
    if (k1 >= num_colors) k1 = num_colors-1;
    for (int i = 0; i < mode; i++) {
      rgba256[index][0] = rgba[k][0] + (float)i/mode*(rgba[k1][0]-rgba[k][0]);
      rgba256[index][1] = rgba[k][1] + (float)i/mode*(rgba[k1][1]-rgba[k][1]);
      rgba256[index][2] = rgba[k][2] + (float)i/mode*(rgba[k1][2]-rgba[k][2]);
      rgba256[index][3] = rgba[k][3] + (float)i/mode*(rgba[k1][3]-rgba[k][3]);
      index++;
    }
  }

  tf->predefColorMap = SoTransferFunction::NONE;
  tf->colorMapType = SoTransferFunction::RGBA;
  tf->colorMap.setNum( 256 * 4 );
  tf->colorMap.setValues( 0, 256 * 4, &rgba256[0][0] );
}

