/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to create and use a simple volume shader
          (extends the simpleVolume example program)

          The fragment shader program supplied with this example is not
          exactly the same as the default VolumeViz shader, but it is
          funtionally equivalent.

          Note that, by default, the shader source lines are supplied as
          string data compiled into the application program.
          By defining the macro USE_SHADER_FILE you can force the program
          to load the shader from a source file on disk.

author : mmh
June 2006
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoTransferFunction.h>

#include <Inventor/nodes/SoFragmentShader.h>
#include <VolumeViz/nodes/SoVolumeShader.h>

#include <Inventor/helpers/SbFileHelper.h>

////////////////////////////////////////////////////////////////////////

// Uncomment this line if you want to load the shader from a file
#define USE_SHADER_FILE 1

// Define the name of the shader source file
SbString ShaderFileName= "simpleShader.glsl";
SbString ShaderPath = "$OIVHOME/examples/source/VolumeViz/simpleShader/";

// Setup the volume shader node
SoVolumeShader* setupShader()
{
#ifdef USE_SHADER_FILE
  // Just a sanity check that the shader file exists...
  SbString fullPath = ShaderPath.toLatin1();
  fullPath += ShaderFileName.toLatin1();

  if ( SbFileHelper::isAccessible( fullPath.toLatin1(), SbFileHelper::READ ) )
  {
    fprintf( stderr, "Successfully opened shader source file.\n" );
  }
  else
  {
    fprintf( stderr, "Shader source file not found!\n" );
  }
  fprintf( stderr, "Using ShaderPath = %s\n"
                   "      ShaderFileName = %s\n",
    ShaderPath.toLatin1(), ShaderFileName.toLatin1() );
#endif

  // Load fragment shader program
  SoFragmentShader* fragmentShader = new SoFragmentShader;
#ifdef USE_SHADER_FILE
  // NOTE! If the shader program source is in a file and the file is not
  //       in the application's current working directory, then you MUST
  //       either use the full explicit path OR add the shader file's
  //       directory path to SoInput's directory search list.
  SoInput::addDirectoryFirst( ShaderPath.toLatin1() );
  fragmentShader->sourceType = SoShaderObject::FILENAME;
  fragmentShader->sourceProgram.setValue( ShaderFileName );
#else
  // NOTE! If the shader program source is in a string (instead of a file)
  //       you MUST set sourceType field before sourceProgram field.
  fragmentShader->sourceType = SoShaderObject::GLSL_PROGRAM;
  fragmentShader->sourceProgram.setValue( shaderProgram );
#endif

  // Create uniform parameters allowing shader to access textures
  //
  // Transfer function is in texture unit 0 (default)
  SoShaderParameter1i *paramTex0 = new SoShaderParameter1i;
  paramTex0->name = "transfer";
  paramTex0->value.setValue(0);

  // Volume data is in texture unit 1 (default)
  SoShaderParameter1i *paramTex1 = new SoShaderParameter1i;
  paramTex1->name = "voldata";
  paramTex1->value.setValue(1);

  fragmentShader->parameter.set1Value(0, paramTex0);
  fragmentShader->parameter.set1Value(1, paramTex1);

  // Initialize and set the volume shader program
  SoVolumeShader* pVolShader = new SoVolumeShader;
  // Specify to what part the shader prog is intended to. Here fragments color only.
  pVolShader->shaderObject.set1Value(SoVolumeShader::FRAGMENT_COMPUTE_COLOR, fragmentShader);
  // Specify that shader is only applied to volumes (not to slice or skin)
  pVolShader->forVolumeOnly = TRUE;

  return pVolShader;
}

////////////////////////////////////////////////////////////////////////
//main function
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  // Initialize of VolumeViz extension
  SoVolumeRendering::init();

  //Volume Size
  const int dimX = 10;
  const int dimY = 10;
  const int dimZ = 10;

  // Create data :
  // Note: 0 < data[i] < number of entries in the colorMap.
  // If each data value has the same value A the volume will be unicolor
  // with the color described by the Ath entries of the colorMap.
  // This volume data will use every color in the colorMap
  unsigned char *data = new unsigned char[dimX * dimY * dimZ];
  for( int i=0; i<dimX * dimY * dimZ; i++ ) {
    data[i] = i%256;
  }

  // Node to hold the volume data
  // In the first call we specify the voxel dimensions and the actual data.
  // In the second call we specify the 3D extent of the volume in modeling coords.
  SoVolumeData* pVolData = new SoVolumeData();
  pVolData->data.setValue(SbVec3i32(dimX, dimY, dimZ), SbDataType::UNSIGNED_BYTE,
                          0, data, SoSFArray3D::NO_COPY);
  pVolData->extent = SbBox3f(float(-dimX), float(-dimY), float(-dimZ), float(dimX), float(dimY), float(dimZ));

  // Use a predefined colorMap with the SoTransferFunction
  SoTransferFunction* pTransFunc = new SoTransferFunction;
  pTransFunc->predefColorMap = SoTransferFunction::STANDARD;

  // Node in charge of drawing the volume
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->samplingAlignment = SoVolumeRender::DATA_ALIGNED;

  // Volume shader
  SoVolumeShader *pVolShader = setupShader();

  // Assemble the scene graph
  // Note: SoVolumeRender must appear after the SoVolumeData node.
  SoSeparator *root = new SoSeparator;
  root->ref();
  root->addChild( pVolShader );
  root->addChild( pVolData );
  root->addChild( pTransFunc );
  root->addChild( pVolRender );

  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setSceneGraph(root);
  myViewer->setTitle("Volume rendering");
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete myViewer;
  root->unref();
  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}


