/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (Oct 2010)
**=======================================================================*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoGradientBackground.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>


#define IMAGE_FILENAME "$OIVHOME/examples/data/VolumeViz/horizons/HorizonRGBA.jpg"
#define STACK_FILENAME "$OIVHOME/examples/data/VolumeViz/horizons/HorizonRGBA-property.lst"
#define LDM_FILENAME "$OIVHOME/examples/data/VolumeViz/horizons/HorizonRGBA-property.ldm"

#include <Inventor/image/SoRasterImageFile.h>
#include <Inventor/image/SoRasterReaderSet.h>

SoVolumeData*
inMemoryRGBAData(const char *filename)
{
  // Load an RGB or RGBA image in memory using SbRasterImage classes
  SbRasterImage rasterImage;
  SoRasterImageFile imageFile;
  imageFile.setFileName( filename );
  SoRasterImageRW* imageReader = SoRasterReaderSet::getReader(&imageFile);
  if ( !imageReader->open(&imageFile, SoRasterImageRW::OPEN_READ) )
  {
    SoError::post("Enable to open %s",filename);
    return NULL;
  }

  if ( !imageReader->read(&rasterImage, false) )
  {
    SoError::post("Enable to read %s",filename);
    imageReader->close();
    return NULL;
  }
  imageReader->close();

  // Get Volume information
  const int dimX = rasterImage.getSize()[0];
  const int dimY = rasterImage.getSize()[1];
  const int dimZ = 1;

  // Do some check on image input

  // - should be byte component else it should be converted 
  // (not implemented as it is provided by using an SoVRRasterStackReader)
  if ( SbDataType(rasterImage.getDataType()).getSize() != 1 )
  {
    SoError::post("this demo do not support image input with component different to 8bits");
    return NULL;
  }

  // - should be RGB or RGBA input
  int ImgNC = rasterImage.getComponentsCount();
  SoBufferObject *dataObject = NULL;
  if ( ImgNC == 4 )
    dataObject = rasterImage.getBufferObject();
  else
  {
    // if not RGBA then we convert it 
    SoBufferObject* imgDataObject = rasterImage.getBufferObject();
    SoRef<SoCpuBufferObject> cpuImgDataObject=new SoCpuBufferObject();
    imgDataObject->map(cpuImgDataObject.ptr(),SoBufferObject::READ_ONLY);
    unsigned char* imgData = (unsigned char*)cpuImgDataObject->map(SoBufferObject::READ_ONLY);

    SoCpuBufferObject* cpuDataObject = new SoCpuBufferObject();
    cpuDataObject->setSize(dimX*dimY*dimZ*4);
    unsigned char* data = (unsigned char*)cpuDataObject->map(SoCpuBufferObject::SET);

    for ( int zz=0;zz<dimZ;zz++)
    {
      for (int yy=0;yy<dimY;yy++)
      {
        for (int xx=0;xx<dimX;xx++)
        {
          int i=(xx+yy*dimX+zz*dimX*dimY);
          int nc;
          for (nc=0;nc<ImgNC;nc++)
            data[i*4+nc] = imgData[i*ImgNC+nc];
          for (;nc<4;nc++)
            data[i*4+nc] = 255;
        }
      }
    }
    cpuDataObject->unmap();
    imgDataObject->unmap(cpuImgDataObject.ptr());
    dataObject = cpuDataObject;
  }
  // put the data in a memory object
  SoMemoryObject* memObj = new SoMemoryObject( dataObject, SbDataType(SbDataType::UNSIGNED_INT32), true );

  // Node to hold the volume data
  // In the first call we specify the voxel dimensions and the actual data.
  // In the second call we specify the 3D extent of the volume in modeling coords.
  SoVolumeData* pVolData = new SoVolumeData();
  pVolData->data.setValue( SbVec3i32(dimX, dimY, dimZ), 32, memObj, SoSFArray::NO_COPY );
  pVolData->extent.setValue( SbBox3f((float)-dimX, (float)-dimY, (float)-dimZ, (float)dimX, (float)dimY, (float)dimZ) );

  // We need to explicitly indicate that data are RGBA
  pVolData->dataRGBA = TRUE;

  return pVolData;
}

//main function
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;
    
  // Initialize of VolumeViz extension
  SoVolumeRendering::init();

  // create a switch to show 3 ways to give RGBA input to VolumeViz
  SoSwitch* dataSwitch = new SoSwitch;
  dataSwitch->whichChild.setValue(2);

  // 1- in memory data (data generated inside the application
  {
    SoVolumeData* pVolData = inMemoryRGBAData(IMAGE_FILENAME);
    pVolData->dataSetId = 5;
    if ( pVolData )
      dataSwitch->addChild(pVolData);
  }
  
  // 2- Stack image file
  {
    SoVolumeData* pVolData = new SoVolumeData;
    pVolData->fileName.setValue(STACK_FILENAME);
    dataSwitch->addChild(pVolData);
  }

  // 3- LDM file
  {
    SoVolumeData* pVolData = new SoVolumeData;
    pVolData->fileName.setValue(LDM_FILENAME);
    dataSwitch->addChild(pVolData);
  }

  SoVolumeRenderingQuality* pVolQuality = new SoVolumeRenderingQuality();

  pVolQuality->forVolumeOnly = FALSE;
  SoOrthoSlice* pVolRender = new SoOrthoSlice;

  // Use a predefined colorMap with the SoTransferFunction
  SoTransferFunction* pTransFunc = new SoTransferFunction;
  pTransFunc->predefColorMap = SoTransferFunction::STANDARD;

  // Assemble the scene graph
  // Note: SoVolumeRender must appear after the SoVolumeData node.
  SoSeparator *root = new SoSeparator;
  root->ref();
  root->addChild( new SoGradientBackground);
  root->addChild( dataSwitch );
  root->addChild( pVolQuality );
  root->addChild( pTransFunc );
  root->addChild( pVolRender );

  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setSceneGraph(root);
  myViewer->setTitle("RGBA Slice rendering");
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete myViewer;
  root->unref();
  SoVolumeRendering::finish();
  SoXt::finish();

  return 0;
}


