/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to create and use a simple volumeData and volumeRender node.
author : Jerome Hummel
August 2002
----------------------------------------------------------------------------------------*/


//header files

#include <Inventor/nodes/SoCone.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoVolumeSkin.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoAnnoText3Property.h>

#include <DialogViz/SoDialogVizAll.h>
#include <DialogViz/dialog/SoTopLevelDialog.h>

#include <MeshViz/graph/PoGroup6Axis3.h>
#include <MeshViz/graph/PoGroup3Axis3.h>
#include <MeshViz/graph/PoLinearAxis.h>

#include <MeshViz/nodes/PoDomain.h>
#include <MeshViz/nodes/PoEllipticProfile.h>
#include <MeshViz/nodes/PoIrregularMesh1D.h>
#include <MeshViz/nodes/PoMesh1DHints.h>
#include <MeshViz/nodes/PoMiscTextAttr.h>

#include <VolumeViz/readers/SoVRSegyFileReader.h> 

// Defining interactive interface file definition
#define INTERFACE_FILE "$OIVHOME/examples/source/VolumeViz/simpleVolumeAxis/simpleVolumeAxisInterface.iv"
#define SCENEGRAPH_FILE "$OIVHOME/examples/source/VolumeViz/simpleVolumeAxis/simpleVolumeAxis.iv"

SoGroup *g_myGroup = NULL;

// Define interaction interface listener
class AxisTypeAuditor : public SoDialogAuditor
{
public:
  AxisTypeAuditor(SoTopLevelDialog* topDialog, SoSwitch *axis_switch)
  { 
    m_top = topDialog;
    m_axis_switch = axis_switch;

    SoDialogViz* elt = topDialog->searchForAuditorId("axisType");

    SoDialogComboBox* cpt = (SoDialogComboBox*)(elt);
    dialogComboBox(cpt);
  }

  ~AxisTypeAuditor()
  {
    m_top = NULL;
    m_axis_switch = NULL;
  }

private:
  SoTopLevelDialog  *m_top;
  SoSwitch          *m_axis_switch;

  void dialogComboBox(SoDialogComboBox* cpt)
  {
    if (cpt->auditorID.getValue() == "axisType")
    {
      int selectedItem = cpt->selectedItem.getValue();

      if(m_axis_switch != NULL)
      {
        m_axis_switch->whichChild = selectedItem-1;
      }
    }
  }
};

// Define interaction interface listener
class AnnoTextTypeAuditor : public SoDialogAuditor
{
public:
  AnnoTextTypeAuditor(SoTopLevelDialog* topDialog, SoAnnoText3Property* axisProperty)
  {
    m_top = topDialog;
    m_axisProperty = axisProperty;

    SoDialogViz* elt = topDialog->searchForAuditorId("annotationType");

    SoDialogComboBox* cpt = (SoDialogComboBox*)(elt);
    dialogComboBox(cpt);
  }

  ~AnnoTextTypeAuditor()
  {
    m_top = NULL;
    m_axisProperty = NULL;
  }

private:
  SoTopLevelDialog  *m_top;
  SoAnnoText3Property *m_axisProperty;

  void dialogComboBox(SoDialogComboBox* cpt)
  {
    if (cpt->auditorID.getValue() == "annotationType")
    {
      int selectedItem = cpt->selectedItem.getValue();

      if(m_axisProperty != NULL)
      {
        m_axisProperty->renderPrintType = selectedItem;
      }
    }
  }
};

// Load a scene graph from an OIV file
SoSeparator *
loadSceneGraph(const char *filename)
{
  // Open the input file
  SoInput mySceneInput;
  if (!mySceneInput.openFile(filename))
  {
    std::cerr << "Cannot open file " << filename << std::endl;
    return NULL;
  }

  // Read the whole file into the database
  SoSeparator *myGraph = SoDB::readAll(&mySceneInput);
  if (myGraph == NULL)
  {
    std::cerr << "Problem reading file" << std::endl;
    return NULL;
  }

  mySceneInput.closeFile();
  return myGraph;
}

// Build the interface from the file
Widget
buildInterface(Widget window, const char* filename, const char* viewer,
               SoTopLevelDialog** myTopLevelDialog)
{
  SoInput myInput;
  if (! myInput.openFile( filename ))
    return NULL;

  g_myGroup = SoDB::readAll( &myInput );
  if (! g_myGroup)
    return NULL;
  g_myGroup->ref();

  *myTopLevelDialog = (SoTopLevelDialog *)g_myGroup->getChild( 0 );

  SoDialogCustom *customNode = (SoDialogCustom *)(*myTopLevelDialog)->searchForAuditorId(SbString(viewer));

  (*myTopLevelDialog)->buildDialog( window, customNode != NULL );
  (*myTopLevelDialog)->show();

  return customNode ? customNode->getWidget() : window;
}


//main function
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  // Initialize (don't need any window system classes)
  SoDB::init();
  PoBase::init();
  SoVolumeRendering::init();
  SoDialogViz::init();

  // Assemble the scene graph
  SoSeparator *root = loadSceneGraph(SCENEGRAPH_FILE);
  root->ref();

  // Adding a box to select axis type
  SoTopLevelDialog *myTopLevelDialog;
  Widget myParent = buildInterface(myWindow, INTERFACE_FILE, "Viewer", &myTopLevelDialog);
  
  // Connect the switch to the combobox
  SoSwitch *axisSwitch = (SoSwitch*)root->getByName("AxisSwitch");
  AxisTypeAuditor* myAxisTypeAuditor = new AxisTypeAuditor(myTopLevelDialog, axisSwitch);
  myTopLevelDialog->addAuditor(myAxisTypeAuditor);

  // Connect the field to the combobox
  SoAnnoText3Property *axisProperty = (SoAnnoText3Property*)root->getByName("AxisProperty");
  AnnoTextTypeAuditor* myAxisTextTypeAuditor = new AnnoTextTypeAuditor(myTopLevelDialog, axisProperty);
  myTopLevelDialog->addAuditor(myAxisTextTypeAuditor);

  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myParent);
  myViewer->setSceneGraph(root);
  myViewer->setTitle("Volume rendering");
  myViewer->viewAll();
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();

  root->unref();
  g_myGroup->unref();

  delete myViewer;
  delete myAxisTextTypeAuditor;
  delete myAxisTypeAuditor;

  SoDialogViz::finish();
  SoVolumeRendering::finish();
  PoBase::finish();
  SoDB::finish();

  SoXt::finish();

  return 0;
}
