//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoGradientBackground.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoPhysicalMaterial.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeShader.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeIsosurface.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <LDM/nodes/SoMultiDataSeparator.h>

#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoVolumeMaskGroup.h>
#include <VolumeViz/nodes/SoVolumeMask.h>
#include <VolumeViz/nodes/SoVolumeDataDrawStyle.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <Inventor/STL/vector>

#include "DrawGeometry.h"

// set to 1 tu use ldm file instead of generating in memory data (usefull
// when recording ART)
#define USE_LDM 0

#define FILENAME "$OIVHOME/examples/data/VolumeViz/3DHEAD.ldm"

/*******************************************************************************/
const size_t NUM_MASKS = 6;
void generateMasks(SoVolumeData* vd, const std::vector<SoVolumeMask*>& vmList, const std::vector<SoVolumeDataDrawStyle*>& vmStyleList);
SoTransferFunction* generateTf(SbVec3f color);

/*******************************************************************************/
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  // Initialize of VolumeViz extension
  SoVolumeRendering::init();

  // Node to hold the volume data
  SoVolumeData* vd = new SoVolumeData();
  vd->fileName = FILENAME;

  // Node in charge of drawing the volume
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->subdivideTile = TRUE;
  pVolRender->numSlicesControl = SoVolumeRender::MANUAL;
  pVolRender->numSlices = 256;
  pVolRender->interpolation = SoVolumeShape::CUBIC;
  pVolRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;

  SoLightModel* lightModel = new SoLightModel;
  lightModel->model = SoLightModel::PHYSICALLY_BASED;

  SoPhysicalMaterial* material = new SoPhysicalMaterial;
  material->baseColor = SbColorRGBA(1.0f, 1.0f, 1.0f, 1.0f);
  material->specular = 1.0f;
  material->roughness = 0.2f;

  //Draw style and transfer function for intersecting masks
  SoVolumeDataDrawStyle* vdsIntersection = new SoVolumeDataDrawStyle;
  vdsIntersection->style = SoVolumeDataDrawStyle::VOLUME_RENDER;

  SoTransferFunction* tfIntersection = new SoTransferFunction;
  tfIntersection->predefColorMap = SoTransferFunction::BLUE_RED;
  tfIntersection->minValue = 42;
  tfIntersection->transferFunctionId = SoVolumeMaskGroup::TRANSFERFUNCTION_INTERSECTION_ID;

  //Add lighting and preintegration
  SoVolumeRenderingQuality* vrq = new SoVolumeRenderingQuality;
  //vrq->lighting = true;
  vrq->preIntegrated = true;
  vrq->interpolateOnMove = TRUE;
  vrq->deferredLighting = TRUE;

  //Add masks/styles/transfer funtion in the SoVolumeMaskGroup
  SoVolumeMaskGroup* vmg = new SoVolumeMaskGroup;
  std::vector<SoVolumeMask*> vmList;
  std::vector<SoVolumeDataDrawStyle*> vmStyleList;
  for ( size_t i = 0; i < NUM_MASKS; i++ )
  {
    SoVolumeMask* vm = new SoVolumeMask;
    vm->extent = vd->extent;
    vm->dataSetId = vd->dataSetId.getValue()+int(i)+1;
#if !USE_LDM
    // When using in memory data we have to explicitly call setProperties in order to manually initialize the volume mask
    vm->setProperties(vd->getDimension(), vd->ldmResourceParameters.getValue()->tileDimension.getValue(), 0);
#endif
    vm->ldmResourceParameters.getValue()->tileDimension = vd->ldmResourceParameters.getValue()->tileDimension;
    vmList.push_back(vm);

    vmStyleList.push_back(new SoVolumeDataDrawStyle);

    //Add some random color the the transfer function
    SbVec3f cmColor = SbVec3f((float)rand(), (float)rand(), (float)rand())/float(RAND_MAX);
    SoTransferFunction* tf = generateTf(cmColor);
    tf->transferFunctionId.connectFrom(&vm->dataSetId);
    vmg->addChild(tf);
    vmg->addChild(vmStyleList.back());
    vmg->addChild(vm);
  }

  //This draw style will be applied on the whole volume
  SoVolumeDataDrawStyle* vdsGlobal = new SoVolumeDataDrawStyle;
  vdsGlobal->style = SoVolumeDataDrawStyle::NONE;
  vdsGlobal->isovalues.set1Value(0, 42);
  vdsGlobal->isosurfacesMaterial = new SoMaterial;
  vdsGlobal->isosurfacesMaterial.getValue()->diffuseColor.set1Value(0, SbColor(0., 0., 1.));
  vdsGlobal->isosurfacesMaterial.getValue()->transparency.set1Value(0, 0.5);

  generateMasks(vd, vmList, vmStyleList);

  SoMultiDataSeparator* mds = new SoMultiDataSeparator;
  mds->addChild( vrq );
  mds->addChild( vdsGlobal );
  mds->addChild( vd );
  mds->addChild(tfIntersection);
  mds->addChild(vdsIntersection);
  mds->addChild(vmg);
  mds->addChild( pVolRender );

  // Assemble the scene graph
  // Note: SoVolumeRender must appear after the SoVolumeData node.
  SoSeparator *root = new SoSeparator;
  root->ref();

  root->addChild(new SoGradientBackground);
  SoDirectionalLight* light = new SoDirectionalLight;
  light->direction = SbVec3f(0.0f, 0.0f, 1.0f);
  root->addChild(light);
  root->addChild(lightModel);
  root->addChild(material);
  root->addChild(mds);

  pVolRender->lowResMode = SoVolumeRender::DECREASE_SCREEN_RESOLUTION;
  pVolRender->lowScreenResolutionScale = 2;

  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
  myViewer->setSceneGraph(root);
  myViewer->setTitle("Volume Mask");
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete myViewer;
  root->unref();
  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}

void
generateMasks(SoVolumeData* vd, const std::vector<SoVolumeMask*>& vmList,
              const std::vector<SoVolumeDataDrawStyle*>& vmStyleList)
{
  SbVec3i32 size = vd->data.getSize();

  vmStyleList[0]->style = SoVolumeDataDrawStyle::MASK_BOUNDARY|SoVolumeDataDrawStyle::ISOSURFACE|SoVolumeDataDrawStyle::VOLUME_RENDER;
  vmStyleList[0]->isovalues.setNum(1);
  vmStyleList[0]->isovalues.set1Value(0, 42);

  SoMaterial* mat;
  mat = vmStyleList[0]->isosurfacesMaterial = new SoMaterial;
  mat->diffuseColor.set1Value(0, SbColor(1., 1., 1.));
  mat->transparency.set1Value(0, 0.5f);

  mat = vmStyleList[0]->boundaryMaterial = new SoMaterial;
  mat->diffuseColor.set1Value(0, SbColor(1., 1., 1.));
  mat->transparency.set1Value(0, 0.5f);

  vmStyleList[4]->style = SoVolumeDataDrawStyle::MASK_BOUNDARY;
  mat = vmStyleList[4]->boundaryMaterial = new SoMaterial;
  mat->diffuseColor.set1Value(0, SbColor(0., 1., 0.));
  mat->transparency.set1Value(0, 0.5f);

#if USE_LDM
  vmList[0]->fileName = "$OIVHOME/examples/source/VolumeViz/simpleVolumeMask/box1.ldm";
  vmList[1]->fileName = "$OIVHOME/examples/source/VolumeViz/simpleVolumeMask/sphere1.ldm";
  vmList[2]->fileName = "$OIVHOME/examples/source/VolumeViz/simpleVolumeMask/sphere2.ldm";
  vmList[3]->fileName = "$OIVHOME/examples/source/VolumeViz/simpleVolumeMask/box2.ldm";
  vmList[4]->fileName = "$OIVHOME/examples/source/VolumeViz/simpleVolumeMask/sphere3.ldm";
  vmList[5]->fileName = "$OIVHOME/examples/source/VolumeViz/simpleVolumeMask/sphere4.ldm";
#else
  std::vector<unsigned char> maskData(size[0]*size[1]*size[2]);
  drawBox(&maskData[0], size, SbBox3i32(SbVec3i32(100, 0, size[2]/2), size), (unsigned char)1);
  vmList[0]->data.setValue(size, SbDataType::UNSIGNED_BYTE,
    0, &maskData[0], SoSFArray::COPY);

  std::fill(maskData.begin(), maskData.end(), 0);
  drawSphere(&maskData[0], size, SbSphere(SbVec3f(77, 127, 35), 10), (unsigned char)1);
  vmList[1]->data.setValue(size, SbDataType::UNSIGNED_BYTE,
    0, &maskData[0], SoSFArray::COPY);

  std::fill(maskData.begin(), maskData.end(), 0);
  drawSphere(&maskData[0], size, SbSphere(SbVec3f(79, 129, 76), 10), (unsigned char)1);
  vmList[2]->data.setValue(size, SbDataType::UNSIGNED_BYTE,
    0, &maskData[0], SoSFArray::COPY);

  std::fill(maskData.begin(), maskData.end(), 0);
  drawBox(&maskData[0], size, SbBox3i32(SbVec3i32(0, 0, 0), SbVec3i32(size[0],size[1], size[2]/2)), (unsigned char)1);
  vmList[3]->data.setValue(size, SbDataType::UNSIGNED_BYTE,
    0, &maskData[0], SoSFArray::COPY);

  std::fill(maskData.begin(), maskData.end(), 0);
  drawSphere(&maskData[0], size, SbSphere(SbVec3f(56, 147, 62), 15), (unsigned char)1);
  vmList[4]->data.setValue(size, SbDataType::UNSIGNED_BYTE,
    0, &maskData[0], SoSFArray::COPY);

  std::fill(maskData.begin(), maskData.end(), 0);
  drawSphere(&maskData[0], size, SbSphere(SbVec3f(90, 48, 64), 60), (unsigned char)1);
  vmList[5]->data.setValue(size, SbDataType::UNSIGNED_BYTE,
    0, &maskData[0], SoSFArray::COPY);
#endif

}

/*******************************************************************************/
//Add color to all tf entries
SoTransferFunction*
generateTf(SbVec3f color)
{
  SoTransferFunction* tf = new SoTransferFunction;
  tf->predefColorMap = SoTransferFunction::NONE;
  tf->colorMapType = SoTransferFunction::RGBA;
  tf->predefColorMap = SoTransferFunction::STANDARD;
  tf->minValue = 42;

  SbVec4f* rgba = (SbVec4f*)tf->actualColorMap.startEditing();
  for ( size_t i = 0; i < 256; i++)
  {
    float r = (color[0]+rgba[i][0])*0.5f;
    float g = (color[1]+rgba[i][1])*0.5f;
    float b = (color[2]+rgba[i][2])*0.5f;
    rgba[i] = SbVec4f(r, g, b, rgba[i][3]);
  }
  tf->actualColorMap.finishEditing();
  return tf;
}


