/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to use the data edition feature of
          SoVolumeData node. 
January 2010
----------------------------------------------------------------------------------------*/

//Inventor includes
#include <Inventor/Xt/SoXt.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoPhysicalMaterial.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/STL/algorithm>

// VolumeViz/LDM includes
#include <LDM/manips/SoROIManip.h>
#include <LDM/SoLDMTopoOctree.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

// DialogViz includes
#include <DialogViz/SoDialogVizAll.h>
#include <DialogViz/dialog/SoTopLevelDialog.h>

#if !defined _WIN32
#define _strdup strdup
#endif

// VolumeData filename
#define FILENAME "$OIVHOME/examples/data/VolumeViz/3DHEAD.ldm"

// Gui filename
#define GUI_FILE "$OIVHOME/examples/source/VolumeViz/volumeDataEditing/gui.iv"

/**
* Internal auditor for GUI components
*/
class guiAuditorClass : public SoDialogAuditor
{
  virtual void dialogPushButton(SoDialogPushButton* component);
  virtual void dialogComboBox(SoDialogComboBox*   component);
};
/***************************************************************/

SoTopLevelDialog * g_topLevelDialog = NULL;
SoROIManip*               pROIManip = NULL;
SoTransferFunction*      pTransFunc = NULL;
SoVolumeData*                    vd = NULL;
std::vector<int>         editionList;
std::vector<SoLDMTileID> tileIdList;

/**
* Some internal stuff to manage GUI. See at the end of tile for details
*/
void buildTileList(SoVolumeData* pVolData);
Widget buildInterface(Widget window, SoVolumeData *pVolData);
void resetRoi();


/**
* This function replaces all non transaprent value of a buffer with the given value
*/
void
editNonTransparentValue( SoCpuBufferObject* dataBufferObj, int selectedValue )
{
  // Map the buffer into CPU so we can edit them with given value
  unsigned char* dataBuffer = (unsigned char*)dataBufferObj->map(SoBufferObject::READ_WRITE);

  // Retrieve current color map values
  const float* colorMapValues = pTransFunc->actualColorMap.getValues(0);

  // Update buffer data for each non transparent values
  int index;
  for (size_t i = 0; i < dataBufferObj->getSize() ; i++ )
  {
    index = dataBuffer[i]*4 + 3;
    if ( colorMapValues[index] != 0 )
      dataBuffer[i] = selectedValue;
  }
  
  // The buffer can be unmapped, modification is finished.
  dataBufferObj->unmap();
}


/**
* Update the selected part of the volume with the selected value 
* In this example:
*     1- Application retrieves, in a user buffer, data from SoVolumeData (using DataAccess API)
*     2- Change non transparent value in this buffer with the selected one in the GUI
*     3- Edit the selected region with the modified buffer
*/
void 
editSubVolume()
{
  // retrieve the value given by the slider
  SoDialogIntegerSlider* valueSlider = (SoDialogIntegerSlider *)g_topLevelDialog->searchForAuditorId(SbString("dataValue"));
  int selectedValue = valueSlider->value.getValue();

  // Get properties of Region of Interest
  SbBox3i32 bBox = SbBox3i32 (pROIManip->subVolume.getValue().getMin(), pROIManip->subVolume.getValue().getMax());

  // Retrieve the needed size to store data using SoDataAccess API with a null pointer
  SoLDMDataAccess::DataInfoBox dataInfoBox = vd->getLdmDataAccess().getData(0, bBox, (SoBufferObject*)NULL);
  
  // The buffer can now be allocated with the correct size
  SoRef<SoCpuBufferObject> dataBufferObj = new SoCpuBufferObject;
  dataBufferObj->setSize((size_t)dataInfoBox.bufferSize);

  // 1- get the data from SoVolumeData using DataAccess API
  vd->getLdmDataAccess().getData(0, bBox, dataBufferObj.ptr());

  // 2- Edit all non transparent value of the buffer
  editNonTransparentValue( dataBufferObj.ptr(), selectedValue );

  // 3- Edit the SoVolumeData using the writeSubVolume API
  int transactionId;
  vd->startEditing(transactionId);          // Initiate edition
  vd->editSubVolume(bBox, dataBufferObj.ptr());  // write the whole edited subvolume
  vd->finishEditing(transactionId);         // finish edition

  // Store the new transation Id in a list, so we will be able to perform some undo operation
  SoDialogComboBox* transactionList = (SoDialogComboBox*)g_topLevelDialog->searchForAuditorId(SbString("transacList"));
  transactionList->addItem(SbString(transactionId));
  editionList.push_back(transactionId);
}

/**
* Update the selected tile with the selected value 
* In this example:
*     1- Application retrieves, in a user buffer, data from SoVolumeData (using DataAccess API)
*     2- Change non transparent value in this buffer with the selected one in the GUI
*     3- Edit the selected tile with the modified buffer
*/
void 
editTile ()
{
  // Retrieve selected tileId
  SoDialogComboBox* tileList = (SoDialogComboBox*)g_topLevelDialog->searchForAuditorId(SbString("tileList"));
  int selectedId = tileList->items[tileList->selectedItem.getValue()].toInt();

  // Get the bounding box of the selected tile
  SbBox3i32 tileBbox = vd->getLDMTopoOctree()->getTilePos( SoLDMTileID( selectedId ) );

  // retrieve the value given by the slider
  SoDialogIntegerSlider* valueSlider = (SoDialogIntegerSlider *)g_topLevelDialog->searchForAuditorId(SbString("tileValue"));
  int selectedValue = valueSlider->value.getValue();

  // For a tile, the buffer dimension is well known
  SoRef<SoCpuBufferObject> dataBufferObj = new SoCpuBufferObject;
  SbVec3i32 tileSize = vd->getTileDimension();

  int defaultValue = 0;
  dataBufferObj->setSize( tileSize[0]*tileSize[0]*tileSize[0]*vd->getDataSize() );
  dataBufferObj->memset( &defaultValue );
  
  // 1- get the data from SoVolumeData using DataAccess API
  vd->getLdmDataAccess().getData(0, tileBbox, dataBufferObj.ptr());

  // 2- Edit all non transparent value of the buffer
  editNonTransparentValue( dataBufferObj.ptr(), selectedValue );

  // 3- Edit the SoVolumeData using the writeTile API 
  int transactionId;
  vd->startEditing( transactionId );
  vd->editTile( SoLDMTileID(selectedId), dataBufferObj.ptr() );
  vd->finishEditing( transactionId );

  // Store the new transation Id in a list, so we will be able to perform some undo operation
  SoDialogComboBox* transactionList = (SoDialogComboBox*)g_topLevelDialog->searchForAuditorId(SbString("transacList"));
  transactionList->addItem(SbString(transactionId));
  editionList.push_back(transactionId);
}

/**
* Undo a given transaction Id (selected from combo box)
*/
void undoTransaction(int value)
{
  // Be sure there is something to undo
  if ( editionList.size() == 0 )
  {
    SoDebugError::post ("undoTransaction", "No transaction available. Nothing to undo\n");
    return;
  }

  // ask volume data to undo given transaction 
  vd->undoEditing(value);

  // Some list management to keep the gui consistent
  std::vector<int>::iterator transactionPos;
  transactionPos = std::find(editionList.begin(), editionList.end(), value);
  editionList.erase(transactionPos);

  // remove the transaction Id from the combo box
  SoDialogComboBox* transactionList = (SoDialogComboBox*)g_topLevelDialog->searchForAuditorId(SbString("transacList"));
  transactionList->removeItem(transactionList->selectedItem.getValue());
}

/*******************************************************************************/
int main(int, char **argv)
{
  // Create the window
  Widget mainWindow = SoXt::init(argv[0]);

  if ( !mainWindow ) 
    return 0;

  // Initialize of VolumeViz extension
  SoVolumeRendering::init();
  SoDialogViz::init();

  // Node to hold the volume data
  vd = new SoVolumeData();
  vd->fileName = FILENAME;

  // Node in charge of drawing the volume
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->numSlicesControl = SoVolumeRender::MANUAL;
  pVolRender->numSlices = 512;
  pVolRender->interpolation = SoVolumeShape::CUBIC;
  pVolRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;

  SoLightModel* lightModel = new SoLightModel;
  lightModel->model = SoLightModel::PHYSICALLY_BASED;

  SoPhysicalMaterial* material = new SoPhysicalMaterial;
  material->baseColor = SbColorRGBA(1.0f, 1.0f, 1.0f, 1.0f);
  material->specular = 1.0f;
  material->roughness = 0.4f;

  // Use a predefined colorMap with the SoTransferFunction
  pTransFunc = new SoTransferFunction;
  pTransFunc->predefColorMap = SoTransferFunction::BLUE_RED;
  pTransFunc->minValue = 42;
  pTransFunc->transferFunctionId = 3;

  SoVolumeRenderingQuality* vrq = new SoVolumeRenderingQuality;
  vrq->lighting = true;

  // Add a ROI
  pROIManip = new SoROIManip();
  resetRoi();

  // Assemble the scene graph
  SoSeparator* root = new SoSeparator;
  root->ref();
  root->addChild(vrq);
  root->addChild(new SoDirectionalLight);
  root->addChild( lightModel );
  root->addChild( material );
  root->addChild( vd );
  root->addChild( pROIManip );
  root->addChild(pTransFunc);
  root->addChild( pVolRender );

  // Build GUI
  Widget gui = buildInterface( mainWindow, vd );

  // Set up viewer
  SoXtExaminerViewer *viewer = new SoXtExaminerViewer( gui );

  viewer->setTransparencyType(SoGLRenderAction::NO_SORT);
  viewer->setSceneGraph(root);
  viewer->setTitle("Volume Mask Editing");
  viewer->show();

  SoXt::show(mainWindow);
  SoXt::mainLoop();

  // Be sure to undo all current modification to unlock all edited tiles
  delete viewer;
  root->unref();
  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();
  return 0;
}

/**
*   This is some internal stuff to manage GUI.
*/

/**
* combo box management
*/
void 
guiAuditorClass::dialogComboBox(SoDialogComboBox*   component)
{
  if (component->auditorID.getValue() == "tileList")
  {
    // retrieve selected id
    SoDialogComboBox* tileList = (SoDialogComboBox*)g_topLevelDialog->searchForAuditorId(SbString("tileList"));
    int selectedId = tileList->items[tileList->selectedItem.getValue()].toInt();

    // A tileId has been selected, adjust ROI to focus only on this tile
    SbBox3i32 tileBBox = vd->getLDMTopoOctree()->getTilePos( SoLDMTileID( selectedId ) );
    pROIManip->box.setValue( tileBBox );
    pROIManip->subVolume.setValue( tileBBox );
    pROIManip->constrained = FALSE;
    pROIManip->boxOn = FALSE;
  }
}

/**
* Button management
*/
void 
guiAuditorClass::dialogPushButton (SoDialogPushButton* component)
{
  if (component->auditorID.getValue() == "editButton")
  {
    editSubVolume();
  }
  else if (component->auditorID.getValue() == "undoButton")
  {
    SoDialogComboBox* transactionList = (SoDialogComboBox*)g_topLevelDialog->searchForAuditorId(SbString("transacList"));
    int value = transactionList->items[transactionList->selectedItem.getValue()].toInt();
    undoTransaction(value);
  }
 else if (component->auditorID.getValue() == "resetRoiButton")
  {
    resetRoi();
  }
 else if (component->auditorID.getValue() == "editTile")
 {
   // Update selected tile
   editTile();
 }
 else if (component->auditorID.getValue() == "saveButton")
 {
   // We need to recompute histogram to store it in the file
   std::vector<char*> parameters;
   parameters.push_back( _strdup("-histogram") );

   vd->saveEditing( true, parameters );
 }
}

/**
* Simple function to get all tiles id of resolution 0 
* to populate the list in the GUI
*/
void 
buildTileList( SoVolumeData* pVolData )
{
  const SoLDMTopoOctree* topoOctree = pVolData->getLDMTopoOctree();
  // Now GUI is built, retrieve the list of tile Ids of resolution 0 to populate the combo list
  tileIdList.clear();
  for ( SoLDMTileID id = 0; id < topoOctree->getNumTileIDs(); id++ )
    if ( ( topoOctree->level( id ) == topoOctree->getLevelMax() ) && ( topoOctree->getFileID( id ) >= 0 ) )
      tileIdList.push_back( id );

  SoDialogComboBox* dialogList = (SoDialogComboBox *)g_topLevelDialog->searchForAuditorId(SbString("tileList"));
  for ( int i = 0; i < (int) tileIdList.size(); i++ )
    dialogList->insertItem(i, SbString((int)tileIdList[i].getID()) );
}

Widget 
buildInterface(Widget window, SoVolumeData *pVolData)
{
  SoInput myInput;
  if (! myInput.openFile( GUI_FILE ))
    return NULL;

  SoGroup *guiGroup = SoDB::readAll( &myInput );

  if (! guiGroup )
    return NULL;

  g_topLevelDialog = (SoTopLevelDialog *)guiGroup ->getChild( 0 );

  guiAuditorClass *guiAuditor = new guiAuditorClass ;
  g_topLevelDialog->addAuditor( guiAuditor );

  SoDialogCustom *customNode = (SoDialogCustom *)g_topLevelDialog->searchForAuditorId(SbString("Viewer"));

  g_topLevelDialog->buildDialog( window, TRUE );
  g_topLevelDialog->show();

  buildTileList( pVolData );
  return customNode->getWidget();

}

/**
* Reset the ROI to match the whole volume
*/
void 
resetRoi()
{
  pROIManip->box.setValue( SbVec3i32(0,0,0), vd->data.getSize() - SbVec3i32(1,1,1) );
  pROIManip->subVolume.setValue( SbVec3i32(0,0,0), vd->data.getSize()- SbVec3i32(1,1,1) );
  pROIManip->constrained = TRUE;
  pROIManip->boxOn = FALSE;
}


