#include "SoOffscreenVolumeRenderCustom.h"

#include <LDM/tiles/SoCpuBufferUniform.h>

SO_NODE_SOURCE(SoOffscreenVolumeRenderCustom)

/******************************************************************************/
SoOffscreenVolumeRenderCustom::SoOffscreenVolumeRenderCustom()
{
  SO_NODE_CONSTRUCTOR(SoOffscreenVolumeRenderCustom);
  SO_NODE_ADD_FIELD(boxSubdivision,(SbVec3i32(0,0,0)));
  m_remainingBox = 0;
}

/******************************************************************************/
SoOffscreenVolumeRenderCustom::~SoOffscreenVolumeRenderCustom()
{
}

/******************************************************************************/
template<typename T>
void
SoOffscreenVolumeRenderCustom::copyBox(SoBufferObject* data, 
                                       const SbBox3i32& box)
{
  // map the destination buffer for update
  SbVec3i32 minB, maxB;
  box.getBounds(minB, maxB);
  SbVec3i32 size;          
  SbDataType dataType;
  T* dest = (T*)m_resultVolData->data.startEditing(size, dataType);  


  //Check if the tile is uniform
  SoCpuBufferUniform* uniformBuffer = dynamic_cast<SoCpuBufferUniform*>(data);
  if ( uniformBuffer )
  {
    for ( int k = minB[2]; k < maxB[2] && k < size[2]; k++ )
      for ( int j = minB[1]; j < maxB[1] && j < size[1]; j++ )
        for ( int i = minB[0]; i < maxB[0] && i < size[0]; i++ )
        {
          size_t pos = i+j*size[0]+k*size[0]*size[1];
          dest[pos] = (T)uniformBuffer->getValue();
        }
  }
  else
  {
    SoRef<SoCpuBufferObject> cpuTileBuffer = new SoCpuBufferObject;
    data->map(cpuTileBuffer.ptr(), SoBufferObject::READ_ONLY);
    T* src = NULL;
    src = (T*)cpuTileBuffer->map(SoCpuBufferObject::READ_ONLY);
    int offs = 0;
    for ( int k = minB[2]; k < maxB[2] && k < size[2]; k++ )
      for ( int j = minB[1]; j < maxB[1] && j < size[1]; j++ )
        for ( int i = minB[0]; i < maxB[0] && i < size[0]; i++ )
        {
          size_t pos = i+j*size[0]+k*size[0]*size[1];
          dest[pos] = src[offs++];
        }
    cpuTileBuffer->unmap();
    data->unmap(cpuTileBuffer.ptr());
  }
  m_resultVolData->data.finishEditing();

}

/******************************************************************************/
void
SoOffscreenVolumeRenderCustom::computeDefaultIjkSubBoxes()
{
  SbVec3i32 subdivs = boxSubdivision.getValue();

  SbVec3i32 boxSizeIjk = boxSize.getValue();
  for ( int i = 0; i < 3; i++ )
    subdivs[i] = 1 << subdivs[i];

  for ( int refineX = 0; refineX < subdivs[0]; ++refineX )
    for ( int refineY = 0; refineY < subdivs[1]; ++refineY )
      for ( int refineZ = 0; refineZ < subdivs[2]; ++refineZ )
      {
        SbVec3f begin(float(refineX)/subdivs[0], float(refineY)/subdivs[1], float(refineZ)/subdivs[2]);
        SbVec3f end(float(refineX+1)/subdivs[0], float(refineY+1)/subdivs[1], float(refineZ+1)/subdivs[2]);
        m_subBoxesIjk.push_back(SbBox3i32(
          SbVec3i32((int32_t)(begin[0]*boxSizeIjk[0]),
          (int32_t)(begin[1]*boxSizeIjk[1]),
          (int32_t)(begin[2]*boxSizeIjk[2])),
          SbVec3i32((int32_t)(end[0]*boxSizeIjk[0]),
          (int32_t)(end[1]*boxSizeIjk[1]),
          (int32_t)(end[2]*boxSizeIjk[2]))
          ));
      }

      // init the number of subboxes to extract
      m_remainingBox = m_subBoxesIjk.size();
}

/******************************************************************************/
SbBool
SoOffscreenVolumeRenderCustom::getNextSubBox(SbBox3i32& nextBox)
{
  // update/compute m_subBoxesIjk list if not already done.
  if (m_subBoxesIjk.size()<=0)
    computeDefaultIjkSubBoxes();

  if ( m_remainingBox == 0 )
  {
    // update/cleanup m_subBoxesIjk list
    m_subBoxesIjk.clear();
    return FALSE;
  }

  m_remainingBox--;
  nextBox = m_subBoxesIjk[m_remainingBox];

  return TRUE;
}

/******************************************************************************/
void
SoOffscreenVolumeRenderCustom::boxComputed(SoGLRenderAction* /*action*/, 
                                           SoBufferObject* data, 
                                           const SbBox3i32& box
                                           )
{
  if ( m_resultVolData )
  {    
    // put the data in a memory object (for now we copy the results)
    switch (components.getValue())
    {
    case ALPHA:
      {
        copyBox<unsigned char>(data, box);
        m_resultVolData->dataRGBA = FALSE;
      }
      break;
    case RGBA:
      {
        copyBox<unsigned int>(data, box);
        m_resultVolData->dataRGBA = TRUE;
      }
      break;
    default:
      SoDebugError::post(OIV_FUNCTION,"Invalid components enum");
    } /* Switch */

    m_resultVolData->updateRegions(&box, 1);
  }

  return;
}

/******************************************************************************/
void
SoOffscreenVolumeRenderCustom::initClass()
{
  SO__NODE_INIT_CLASS(SoOffscreenVolumeRenderCustom, "OffscreenVolumeRenderCustom", SoOffscreenVolumeRender);
}

/******************************************************************************/
void
SoOffscreenVolumeRenderCustom::exitClass()
{
  SO__NODE_EXIT_CLASS(SoOffscreenVolumeRenderCustom);
}


