//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/STL/algorithm>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeShader.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeIsosurface.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoVolumeDataDrawStyle.h>

#include <LDM/nodes/SoMultiDataSeparator.h>
#include <LDM/manips/SoROIManip.h>

#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoVolumeMaskGroup.h>
#include <VolumeViz/nodes/SoVolumeMask.h>
#include <VolumeViz/nodes/SoTransferFunction.h>

#include <DialogViz/SoDialogVizAll.h>
#include <DialogViz/dialog/SoTopLevelDialog.h>


#include "DrawGeometry.h"

// VolumeData filename
SbString FILENAME =("$OIVHOME/examples/data/VolumeViz/3DHEAD.ldm");

// FileName where to save VolumeMask modification
SbString FILENAME_VM =("$OIVHOME/examples/data/VolumeViz/3DHEAD_vm_");

// Gui filename
SbString GUI_FILE =("$OIVHOME/examples/source/VolumeViz/volumeMaskEditing/gui.iv");
const size_t NUM_MASKS = 6;

SoTopLevelDialog * g_topLevelDialog = NULL;
SoROIManip*               pROIManip = NULL;
SoVolumeData*                    vd = NULL;
SoVolumeMask*            volumeMask = NULL;
bool                      firstSave = true;
std::vector<int>         editionList;
bool m_useBitSet = false;


/**
* Simply reset the ROI to match the whole volume
*/
void resetRoi()
{
  pROIManip->box.setValue( SbVec3i32(0,0,0), vd->data.getSize() - SbVec3i32(1,1,1) );
  pROIManip->subVolume.setValue( SbVec3i32(0,0,0), vd->data.getSize()- SbVec3i32(1,1,1) );
  pROIManip->constrained = TRUE;
  pROIManip->boxOn = FALSE;
}

/**
* Update the selected part of the volume mask with the selected value ( 0 | 1 )
*/
void updateVolumeMask(unsigned char value)
{
  SbBox3i32 bBox = SbBox3i32 (pROIManip->subVolume.getValue().getMin(), pROIManip->subVolume.getValue().getMax());

  int transactionId;
  volumeMask->startEditing(transactionId);
  volumeMask->editSubVolume(bBox, value);
  volumeMask->finishEditing(transactionId);

  SoDialogComboBox* transactionList = (SoDialogComboBox*)g_topLevelDialog->searchForAuditorId(SbString("transacList"));

  // Once the transaction is over, just add transaction Id to combo box to be able to undo it later
  transactionList->addItem(SbString(transactionId));
  editionList.push_back(transactionId);
}

/**
* Undo a given transaction Id (selected from combo box)
*/
void undoTransaction(int value)
{
  std::vector<int>::iterator transactionPos;
  transactionPos = std::find(editionList.begin(), editionList.end(), value);

  if ( transactionPos == editionList.end() )
    return;

  volumeMask->undoEditing(value);
  editionList.erase(transactionPos);

  // remove the transaction Id from the combo box
  SoDialogComboBox* transactionList = (SoDialogComboBox*)g_topLevelDialog->searchForAuditorId(SbString("transacList"));
  transactionList->removeItem(transactionList->selectedItem.getValue());
}

/**
* Internal auditor for GUI components
*/
class guiAuditorClass : public SoDialogAuditor
{
  virtual void dialogPushButton(SoDialogPushButton* component);
};

/**
* Button management
*/
void guiAuditorClass::dialogPushButton (SoDialogPushButton* component)
{
  if (component->auditorID.getValue() == "editButton")
  {
    // Update the mask with selected value.
    SoDialogRadioButtons* radioButton= (SoDialogRadioButtons*)g_topLevelDialog->searchForAuditorId(SbString("radioButton"));
    updateVolumeMask((unsigned char)(radioButton->selectedItem.getValue()));
  }
  else if (component->auditorID.getValue() == "undoButton")
  {
    if ( editionList.size() == 0 )
    {
      SoDebugError::post ("undoTransaction", "No transaction available. Nothing to undo\n");
      return;
    }

    SoDialogComboBox* transactionList = (SoDialogComboBox*)g_topLevelDialog->searchForAuditorId(SbString("transacList"));
    int value = transactionList->items[transactionList->selectedItem.getValue()].toInt();
    undoTransaction(value);
  }
 else if (component->auditorID.getValue() == "resetRoiButton")
  {
    resetRoi();
  }
 else if (component->auditorID.getValue() == "saveButton")
 {
   std::vector<char*> converterArg;
   if ( m_useBitSet )
   {
     converterArg.push_back((char*)"-B");
     converterArg.push_back((char*)"-c");
     converterArg.push_back((char*)"gzip");
   }

   SbString volumeMaskFile = SbString(FILENAME_VM);
   volumeMaskFile += volumeMask->getId();
   volumeMaskFile += ".ldm";
   volumeMask->saveEditing(volumeMaskFile, true, converterArg);
 }
}

Widget buildInterface(Widget window, SoVolumeData* /*pVolData*/)
{
  SoInput myInput;
  if (! myInput.openFile( GUI_FILE ))
    return NULL;

  SoGroup *guiGroup = SoDB::readAll( &myInput );

  if (! guiGroup )
    return NULL;

  g_topLevelDialog = (SoTopLevelDialog *)guiGroup ->getChild( 0 );

  guiAuditorClass *guiAuditor = new guiAuditorClass ;
  g_topLevelDialog->addAuditor( guiAuditor );

  SoDialogCustom *customNode = (SoDialogCustom *)g_topLevelDialog->searchForAuditorId(SbString("Viewer"));

  g_topLevelDialog->buildDialog( window, TRUE );
  g_topLevelDialog->show();

  return customNode->getWidget();

}

int 
parseCmdLine(int argc, char** argv)
{
  int status = 0;

  for ( int iarg = 1; iarg < argc ; iarg++) 
  {
    if (argv[iarg][0] == '-') 
    {
      char option = argv[iarg][1];

      switch (option) 
      {
      case 'b': 
        {
          iarg++;
          m_useBitSet = true;
          break;
        }
      default:
        fprintf(stderr,"\n unknwon option -%c\n", option);
        status = -1;
      }
    }
    else
      FILENAME = SbString(argv[iarg]);
  }

  return status;
}

/*******************************************************************************/
int main(int argc, char **argv)
{
  // Create the window
  Widget mainWindow = SoXt::init(argv[0]);

  if ( !mainWindow )
    return 0;

  parseCmdLine(argc, argv);

  // Initialize of VolumeViz extension
  SoVolumeRendering::init();
  SoDialogViz::init();

  // Node to hold the volume data
  vd = new SoVolumeData();
  vd->fileName = FILENAME;

  SoVolumeDataDrawStyle* globalVds = new SoVolumeDataDrawStyle;
  globalVds->style = SoVolumeDataDrawStyle::NONE;

  SoVolumeDataDrawStyle* maskVds = new SoVolumeDataDrawStyle;
  maskVds->style = SoVolumeDataDrawStyle::VOLUME_RENDER;

  // Node in charge of drawing the volume
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->numSlicesControl = SoVolumeRender::MANUAL;
  pVolRender->numSlices = 512;
  pVolRender->samplingAlignment = SoVolumeRender::VIEW_ALIGNED;

  //Material which defines the isosurface diffuse color, transparency,
  //specular color and shininess
  SoMaterial *matVolRend = new SoMaterial;
  matVolRend->diffuseColor.setValue(1,1,1);
  matVolRend->transparency.setValue( 0.0f );
  matVolRend->specularColor.setValue(1,1,1);
  matVolRend->shininess.setValue(0.5);

  // Use a predefined colorMap with the SoTransferFunction
  SoTransferFunction* pTransFunc = new SoTransferFunction;
  pTransFunc->predefColorMap = SoTransferFunction::BLUE_RED;
  pTransFunc->minValue = 42;
  pTransFunc->transferFunctionId = vd->dataSetId.getValue();

  // Build the volumeMask and volumeMaskGroup
  SoTransferFunction* pTransFuncVm = new SoTransferFunction;
  pTransFuncVm->predefColorMap = SoTransferFunction::BLUE_RED;
  pTransFuncVm->minValue = 42;

  // Define a volume mask with no data.
  volumeMask = new SoVolumeMask;
  volumeMask->setProperties( vd->data.getSize(), vd->getTileDimension(), 0);
  volumeMask->extent.connectFrom(&vd->extent);
  volumeMask->dataSetId = vd->dataSetId.getValue()+2;
  pTransFuncVm->transferFunctionId = volumeMask->dataSetId.getValue();
  //volumeMask->fileName = FILENAME_VM+ "3.ldm";

  // Use a SoVolumeMaskGroup
  SoVolumeMaskGroup* volumeMaskGroup = new SoVolumeMaskGroup;
  volumeMaskGroup->addChild(maskVds);
  volumeMaskGroup->addChild(pTransFuncVm);
  volumeMaskGroup->addChild(volumeMask);

  SoVolumeRenderingQuality* vrq = new SoVolumeRenderingQuality;
  vrq->lighting = true;

  // Add a ROI
  pROIManip = new SoROIManip();
  resetRoi();

  // Assemble the scene graph
  SoMultiDataSeparator* root = new SoMultiDataSeparator;
  root->ref();
  root->addChild(vrq);
  root->addChild(new SoDirectionalLight);
  root->addChild( matVolRend );
  root->addChild( globalVds );
  root->addChild( vd );
  root->addChild( pROIManip );
  root->addChild(pTransFunc);
  root->addChild(volumeMaskGroup);
  root->addChild( pVolRender );

  // Build GUI
  Widget gui = buildInterface( mainWindow, vd );

  // Set up viewer
  SoXtExaminerViewer *viewer = new SoXtExaminerViewer( gui );

  viewer->setTransparencyType(SoGLRenderAction::NO_SORT);
  viewer->setSceneGraph(root);
  viewer->setTitle("Volume Mask Editing");
  viewer->show();

  SoXt::show(mainWindow);
  SoXt::mainLoop();

  // Be sure to undo all current modification to unlock all edited tiles
  delete viewer;
  root->unref();
  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();
  return 0;
}


