#include "utils.h"

#include <Inventor/STL/algorithm>
#include <Inventor/SbRotation.h>
#include <Inventor/SbVec.h>
#include <Inventor/SbViewportRegion.h>

#include <Inventor/actions/SoGetBoundingBoxAction.h>

#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoGroup.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoCircularExtrusion.h>
#include <Inventor/nodes/SoSelection.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoExtrusion.h>
#include <Inventor/nodes/SoPickStyle.h>

#include <Inventor/STL/vector>
#include <Inventor/STL/iterator>


float 
floatRand( float range_min /*= FLT_EPSILON */, float range_max /*= FLT_MAX */)
{
  // Generate random numbers in the half-closed interval
  // [range_min, range_max). In other words,
  // range_min <= random number < range_max
  return (float)rand() / ((size_t)RAND_MAX + 1) * (range_max - range_min) + range_min;
}

SoSeparator *
drawExtentBbox( const SbBox3f & volBox  )
{
  SoSeparator *pSep = new SoSeparator;
  pSep->ref();
  SoPickStyle *pPSty = new SoPickStyle;
  pPSty->style = SoPickStyle::UNPICKABLE;
  pSep->addChild( pPSty );
  SoDrawStyle *pSty = new SoDrawStyle;
  pSty->style = SoDrawStyle::LINES;
  pSep->addChild( pSty );
  SoMaterial *pMat  = new SoMaterial;
  pMat->diffuseColor.setValue( 1.f, 1.f, 1.f );
  pSep->addChild( pMat );
  SbVec3f min = volBox.getMin();
  SbVec3f max = volBox.getMax();
  SoCube *pCub = new SoCube;
  pCub->width  =  max[0] - min[0];
  pCub->height =  max[1] - min[1];
  pCub->depth  =  max[2] - min[2];
  SoTranslation *pTran = new SoTranslation;
  pTran->translation = volBox.getCenter();
  pSep->addChild( pTran );
  pSep->addChild( pCub );
  pSep->unrefNoDelete();
  return pSep;
}

void
createPipeSpine( int num, int depthIdx, const SbBox3f &volBox,
                 const SbVec3f &planeNorm, const SbVec3f &planeOrig,
                 const float yminFactor,
                 std::vector<SbVec3f> &spine )
{
  if ( num > 1)
  {
    // creates a Spine on plane thru planeOrig and with normal planeNorm
    // depthIdx is the index of the axis which we considered the depth (default it is 0)
    SbVec3f min( volBox.getMin() );
    SbVec3f max( volBox.getMax() );
    // 
    SbVec3i32 indexPer[3] = { SbVec3i32( 2, 1, 0 ), SbVec3i32( 0, 1, 2 ), SbVec3i32( 1, 2, 0 ) };
    double xrang = ( max[indexPer[depthIdx][0]] - min[indexPer[depthIdx][0]] ) * .5 * .8;    // half extent 20% margin
    double ymin = ( max[indexPer[depthIdx][1]] - min[indexPer[depthIdx][1]] ) * yminFactor;   
    double vel = 1.;
    double k = ( xrang *  vel ) / (log ( 1 + max[indexPer[depthIdx][1]] - ymin ) );
    double xstep = xrang / (num - 1);
    spine.reserve( num );
    SbVec3f vers[3] = { SbVec3f( 0.f, 1.f, 0.f), SbVec3f( 0.f, 0.f, 1.f ), SbVec3f( 1.f, 0.f, 0.f) };
    SbRotation rot( vers[depthIdx], planeNorm );
    SbVec3f trlOrig(planeOrig);
    trlOrig[depthIdx] = .0f;
    for ( int i = 0; i < num; i++ )
    {
      double x = xstep * i;
      double y = ymin + exp( ( ( xrang - x ) * vel ) / k ) - 1;
      SbVec3f point( (float)x, (float)y, .0f);
      SbVec3f tpoint(point);
      rot.multVec( point, tpoint );
      tpoint = tpoint + trlOrig;
      SbBox3f test;
      test.setBounds( tpoint, tpoint );
      if ( volBox.contains( test ) )
        spine.push_back( tpoint );
      else
        break;
    }
  }  
}

void 
createSynthCircularExtrusions(  int depthIdx, SoGroup *appendTo, int num, const SbBox3f &bbox )
{
  // choose random origin around extent bbox
  // choose random plane normal in 
  SbVec3f bboxsize( bbox.getSize() );
  SbVec3f center( bbox.getCenter() );
  SbVec3i32 indexPer[3] = { SbVec3i32( 2, 1, 0 ), SbVec3i32( 0, 2, 1 ), SbVec3i32( 1, 0, 2 ) };
  float anglim = atan2( bboxsize[indexPer[depthIdx][1]], bboxsize[indexPer[depthIdx][0]] );
  const float PI = 3.1415926f;
  const float PI_2 = PI / 2;
  const float PI_3_2 = 3 * PI_2;
  for (int i = 0; i < num; i++)
  {
    float a;
    if (i % 2)
      a = floatRand( PI/2 - anglim, PI/2 + anglim );
    else
      a = floatRand(  PI_3_2 - anglim, PI_3_2 + anglim );
    SbVec3f norm( (float)cos( a ), .0f, (float)sin( a ) );
    SbVec3f orig;
    for ( int j = 0; j < 3; ++j) {
      if ( j == depthIdx )
        orig[j] = bbox.getMax()[j];
      else
        orig[j] = center[j] + floatRand( .0f, bboxsize[j] * .2f );
    }
    SbString ithPipeName = "PIPENODEN_" + SbString( i );
    std::vector<SbVec3f> spine;
    const int numPts = 3000;
    const int depthAxis = 1; // yAxis
    float yminFactor = floatRand( .2f, .4f ); // pipe min among .2 and .6 of the depth
    createPipeSpine( numPts, depthAxis, bbox, norm, orig, yminFactor, spine );
    int sz = (int)spine.size();
    SbVec3f *pvec = new SbVec3f[sz];
    std::vector<SbVec3f>::iterator it = spine.begin();
    for ( int k = 0; it != spine.end(); ++it, ++k )
      pvec[k] = *it;
    // std::copy( spine.begin(), spine.end(), stdext::checked_array_iterator<SbVec3f *>( pvec, sz ) );
    const float bboxRadiusRatio = .005f;
    SoCircularExtrusion *pSPIth = new SoCircularExtrusion;
    pSPIth->spine.setValues( 0, sz, pvec );
    pSPIth->radius = std::min(  std::min( bboxsize[0], bboxsize[1] ), bboxsize[2] ) * bboxRadiusRatio;
    delete [] pvec;
    pSPIth->setName( ithPipeName );
    appendTo->addChild( pSPIth );
  }
}

