/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Roberto Calabrese (Mar 2011)
**=======================================================================*/

/*******************************************************************************
* This example shows how to use a VolumeClippingGroup to clip a VolumeData with a CircularExtrusion
* than can be selected by the user from a set.
* Some SoCircularExtrusions are created on the fly inside the volume of the chosen VolumeData .ldm file,
* the VolumeData is sketched in the scene by mean of its bounding box.
* The fields ( radius and activesection start and end points ) of the seelcted SoCircularExtrusion
* can be modified interactively by the user.
* The SoPipeShas owns one SoComplexity node as to see how its value impacts on the Pipe representation.
* A check-box enables the user to see the VolumeData LDM topology.
* The scene graph contains a VolumeClippingGroup node that is used to clip the VolumeData.
* When a CircularExtrusion is selected the selection callback assigns the selected CircularExtrusion as 
* child of this VolumeClippingGroup node.
*******************************************************************************/
#include "utils.h"

#include <DialogViz/SoDialogVizAll.h>

#include <Inventor/SbBox.h>
#include <Inventor/actions/SoBoxHighlightRenderAction.h>

#include <Inventor/helpers/SbFileHelper.h>

#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoCircularExtrusion.h>
#include <Inventor/nodes/SoSelection.h>
#include <Inventor/nodes/SoSwitch.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoDataRange.h>

#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#define DEMOROOT "$OIVHOME/examples/source/VolumeViz/volumePipeClipping/"

/** selection and de-selection call back */
void selectionCB( void* udata, SoPath *pPath );
// void deselectionCB( void* udata, SoPath *pPath );

/** Global root scene graph loaded from scenegraph.iv */
SoSeparator *g_rootSceneGraph = NULL;

/** The current CircularExtrusion node */
SoCircularExtrusion *g_pCurrCircularExtrusion = NULL;

/** The current CircularExtrusion complexity */
SoComplexity *g_pCurrPipeCompl = NULL;

/** pipe sliders auditor  */
class MyAuditor : public SoDialogRealSliderAuditor
{
public:
  void dialogRealSlider( SoDialogRealSlider *cpt );
};

/** topology check-box auditor auditor  */
class MyAuditorCbox : public SoDialogCheckBoxAuditor
{
public:
  void dialogCheckBox ( SoDialogCheckBox* cpt );
};


/** setup the DialogViz interface */
class UIManager 
{
public:
  UIManager() : m_radiusPtr( 0 ), m_slStPtr( 0 ), m_slEnPtr( 0 ), m_pComp( 0 ) { }
  Widget setUp( Widget mainWindowWidget );
  void setControlsActive( SbBool val );
  void setControlsValue( SoCircularExtrusion *spPtr, SoComplexity *pComp );

private:
  MyAuditor m_auditor;
  MyAuditorCbox m_checkboxaud;
  SoDialogRealSlider *m_radiusPtr;
  SoDialogRealSlider *m_slStPtr;
  SoDialogRealSlider *m_slEnPtr;
  SoDialogRealSlider *m_pComp;
};

int
main( int /*argc*/, char **argv )
{
  // Lower epsilon value to avoid "twisting" artifacts
  SoPreferences::setFloat("OIV_EXTRUSION_EPSILON", 0.999995f);

  Widget mainWindow = SoXt::init( argv[0] );
  SoVolumeRendering::init(); // this must be done before loading VolumeViz nodes in the sg
  UIManager uiManager;

  // Build the scene graph
  {
    SoInput myInput;
    myInput.openFile( SbString(DEMOROOT) + "scenegraph.iv" );
    g_rootSceneGraph = SoDB::readAll( &myInput );
    g_rootSceneGraph->ref();
    myInput.closeFile();
    if ( g_rootSceneGraph )
    {
      //create pipes in the VolumeData extent
      SoVolumeData* pVolData = searchNode<SoVolumeData>( g_rootSceneGraph );
      if ( pVolData )
      {
        SbBox3f volBox = pVolData->extent.getValue();
        // draw the volume bbox
        SoSeparator *pExtBox = drawExtentBbox( volBox );
        g_rootSceneGraph->addChild( pExtBox );
        SoSwitch *pPipesSw = searchName<SoSwitch>( g_rootSceneGraph, "PIPES_SW" );
        if ( pPipesSw )
          createSynthCircularExtrusions(1, pPipesSw, 6, volBox );
        // add data range to manage the generic .ldm case
        SoDataRange *pRange = searchName<SoDataRange>( g_rootSceneGraph, "DATA_RANGE" );
        if (pRange) {
          double minval, maxval;
          pVolData->getMinMax( minval, maxval );
          pRange->min = minval;
          pRange->max = maxval;
        }
      }
    }
  }

  Widget parent = uiManager.setUp( mainWindow );

  // Viewer setup
  SoXtExaminerViewer *pViewer = new SoXtExaminerViewer( parent );
  pViewer->setTitle( "Volume clipping with CircularExtrusions" );
  pViewer->setSceneGraph( g_rootSceneGraph );
  pViewer->setBackgroundColor( SbColor(0.f, 0.f, 0.f ));
  pViewer->setTransparencyType( SoGLRenderAction::OPAQUE_FIRST );
  pViewer->setFloatingColorBuffer(TRUE, SoXtRenderArea::FLOAT_32_COLOR_BUFFER);

  pViewer->show();        
  pViewer->viewAll();

  SoSelection *pSel = searchName<SoSelection>( g_rootSceneGraph, "PIPES_SEL" );
  if ( pSel ) 
  {
    SoCircularExtrusion *pPS = searchName<SoCircularExtrusion>( g_rootSceneGraph, "PIPENODEN_0" );
    pSel->select( pPS );
  }

  // Main Inventor event loop
  SoXt::show( mainWindow );
  SoXt::mainLoop();

  // release global object
  g_rootSceneGraph->unref();
  delete pViewer;

  // release modules
  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();

  return 0;
}

// Sliders 
void MyAuditor::dialogRealSlider( SoDialogRealSlider *cpt )
{
  if ( g_pCurrPipeCompl ) 
  {
    if ( cpt->getName() == "SlidComplx" )
    {
      g_pCurrPipeCompl->value = cpt->value.getValue();
    }
  }
  if ( g_pCurrCircularExtrusion ) 
  {
    if ( cpt->getName() == "PipeRadius" )
    {
      g_pCurrCircularExtrusion->radius = cpt->value.getValue();
    } else 
    {
      SbVec2f acts( g_pCurrCircularExtrusion->activeSection.getValue() );
      if ( cpt->getName() == "SlidStart" )
      {
        acts[0] = cpt->value.getValue();
      } else if ( cpt->getName() == "SlidEnd" ) 
      {
        acts[1] = cpt->value.getValue();
      }
      g_pCurrCircularExtrusion->activeSection = acts;
    }
  }
}

void 
MyAuditorCbox::dialogCheckBox( SoDialogCheckBox* cpt )
{
  if (cpt->auditorID.getValue() == "SHOW_TOPOLOGY_AUD")
  {
    SbBool value( cpt->state.getValue() );
    SoLDMGlobalResourceParameters::setVisualFeedbackParam( SoLDMGlobalResourceParameters::DRAW_TOPOLOGY, value );
  }
}

Widget
UIManager::setUp( Widget mainWindowWidget  )
{
  SoDialogCustom *myCusDlg = 0;
  SoDialogViz::init();

  SbString guiDialogFName = SbString(DEMOROOT) + "interface.iv";
  SoTopLevelDialog *myTopDlg = (SoTopLevelDialog *)SoDialogViz::loadFromFile( guiDialogFName );
  if ( myTopDlg )
  {
    myCusDlg = (SoDialogCustom *)(myTopDlg)->searchForAuditorId( "DIALOG_CUSTOM" );
    m_radiusPtr = (SoDialogRealSlider*)myTopDlg->searchForAuditorId( "CHANGE_RADIUS_AUD" );
    m_radiusPtr->addAuditor( &m_auditor );
    // set min and max for radius
    SoCircularExtrusion *pPipe0 = searchName<SoCircularExtrusion>( g_rootSceneGraph, "PIPENODEN_0" );
    if ( pPipe0 )
    {
      m_radiusPtr->min = pPipe0->radius.getValue();
      m_radiusPtr->max = 10 * pPipe0->radius.getValue();
    }
    m_slStPtr = (SoDialogRealSlider*)myTopDlg->searchForAuditorId( "CHANGE_SLID_STA_AUD" );
    m_slStPtr->addAuditor( &m_auditor );
    m_slEnPtr = (SoDialogRealSlider*)myTopDlg->searchForAuditorId( "CHANGE_SLID_END_AUD" );
    m_slEnPtr->addAuditor( &m_auditor );
    m_pComp = (SoDialogRealSlider*)myTopDlg->searchForAuditorId( "CHANGE_SLID_COMP_AUD" );
    m_pComp->addAuditor( &m_auditor );
    SoDialogCheckBox *pChBox = (SoDialogCheckBox*)myTopDlg->searchForAuditorId( "SHOW_TOPOLOGY_AUD" );
    pChBox->addAuditor( &m_checkboxaud );

    // associate SoSelection with auditor
    SoSelection *pSel = searchName<SoSelection>( g_rootSceneGraph, "PIPES_SEL" );
    if ( pSel )
    {
      pSel->addSelectionCallback( selectionCB, (void *)this );
      // select first pipe
      pSel->select( pPipe0 );
    }
    

    myTopDlg->buildDialog(mainWindowWidget, TRUE);
    myTopDlg->show();
  }

  return myCusDlg ? myCusDlg->getWidget() : mainWindowWidget;
}

void
UIManager::setControlsActive( SbBool val )
{
  m_radiusPtr->enable = val;
  m_slStPtr->enable = val;
  m_slEnPtr->enable = val;
  m_pComp->enable = val;
}

void
UIManager::setControlsValue( SoCircularExtrusion *spPtr, SoComplexity *pComp )
{
  if ( g_pCurrCircularExtrusion  )
  {
    if ( spPtr ) 
    {
      m_radiusPtr->value.setValue( g_pCurrCircularExtrusion->radius.getValue() );
      m_slStPtr->value.setValue( g_pCurrCircularExtrusion->activeSection.getValue()[0] );
      m_slEnPtr->value.setValue( g_pCurrCircularExtrusion->activeSection.getValue()[1] );
    }
    if ( pComp )
      m_pComp->value.setValue( g_pCurrPipeCompl->value.getValue() );

    setControlsActive( TRUE );

  } else
    setControlsActive( FALSE );
}

SoSeparator *
createVolumeClippingGroupOperand( SoCircularExtrusion *spPtr, SoCircularExtrusion* &pCurrCircularExtrusion, SoComplexity* &pCurrPipeCompl )
{
  SoSeparator *pSep = new SoSeparator;
  pSep->setName("PIPE_CURR_SEP");
  pSep->ref();
  SoComplexity* pComp = new SoComplexity;
  pCurrPipeCompl = pComp;
  pSep->addChild( pComp );
  SoCircularExtrusion *clonedPipe = new SoCircularExtrusion;
  clonedPipe->setName( "PIPE_CURR" );
  clonedPipe->radius = spPtr->radius.getValue() * 8; // enhancement factor for radius of selected Pipe
  clonedPipe->spine = spPtr->spine;
  clonedPipe->activeSection.setValue( .3f, 1.0f );
  pCurrCircularExtrusion = clonedPipe;
  pSep->addChild( clonedPipe );
  pSep->unrefNoDelete();
  return pSep;
}

void 
switchOffCurrPipe()
{
  if ( g_pCurrCircularExtrusion )
  {
    // turn off VolumeData
    SoSwitch *pVolDataSw = searchName<SoSwitch>( g_rootSceneGraph, "VOLUME_DATA_ROOT" );
    if ( pVolDataSw )
      pVolDataSw->whichChild = -1;

    // turn off VolumeClipping
    SoSwitch *pVolClipper = searchName<SoSwitch>( g_rootSceneGraph, "VOL_CLIPPER" );
    if ( pVolClipper )
      pVolClipper->whichChild = -1;

    // get selected operand
    SoSeparator *volOp = searchName<SoSeparator>( g_rootSceneGraph, "PIPE_OPERAND" );
    if ( volOp )
    {
      SoSeparator *pPipeSep = searchName<SoSeparator>( volOp, "PIPE_CURR_SEP" );
      if ( pPipeSep )
        volOp->removeChild( pPipeSep );

      g_pCurrCircularExtrusion = NULL;
      g_pCurrPipeCompl = NULL;
    }
  }
}

void 
setCurrPipe( SoCircularExtrusion *spPtr )
{
  SoSeparator *pSep = createVolumeClippingGroupOperand( spPtr, g_pCurrCircularExtrusion, g_pCurrPipeCompl );
  if ( pSep )
  {
    // get selected operand
    SoSeparator *volOp = searchName<SoSeparator>( g_rootSceneGraph, "PIPE_OPERAND" );
    if ( volOp )
      volOp->addChild( pSep ); 

    // turn on VolumeData
    SoSwitch *pVolDataSw = searchName<SoSwitch>( g_rootSceneGraph, "VOLUME_DATA_ROOT" );
    if ( pVolDataSw )
      pVolDataSw->whichChild = 0;

    // turn on VolumeClipping
    SoSwitch *pVolClipper = searchName<SoSwitch>( g_rootSceneGraph, "VOL_CLIPPER" );
    if ( pVolClipper )
      pVolClipper->whichChild = 0;
  }
}

void 
selectionCB( void* udata, SoPath *pPath )
{
  if ( pPath )
  {
    if ( pPath->getTail()->isOfType( SoCircularExtrusion::getClassTypeId() ) &&
         pPath->getTail() != g_pCurrCircularExtrusion )
    {
      switchOffCurrPipe();
      setCurrPipe( (SoCircularExtrusion *) pPath->getTail() );
      // align GUI
      UIManager *uim = (UIManager *)udata;
      if ( uim )
        uim->setControlsValue( g_pCurrCircularExtrusion, g_pCurrPipeCompl );
    }
  }
}

